/*!
 *  \file veboas.h
 *
 *  \copyright Copyright (c) 2016 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */

#ifndef veboas_h
#define veboas_h

#include <iostream>
#include <vector>


namespace seq
{
    namespace details
    {        
        /*!
         * \brief This is a helper class used to store Van Emde Boas indexes
         */
        class vebindices
        {
        public:
            //! \brief Pre-computed indices for all trees
            static const std::size_t indices_[8192];
            
            //! \brief Pre-computed offsets for indices for accessing indexes given a height
            static const std::size_t offsets_[12];
        };

        /*!
         * \brief This class implements a complete Van Emde Boas tree layout
         */
        template <class T>
        class vebtree
        {
        public:
            
            //! \brief Creates a full complete binary tree with Van Emde Boas layout, with a given height
            vebtree(std::size_t height) : h_(height), maxindex_((1 << h_) - 1)
            {
                // Allocate sufficient elements for a complete binary tree
                data_ = new T[maxindex_];
            }
            
            //! \brief Deallocates all memory
            ~vebtree()
            {
                delete[] data_;
            }

            std::size_t minindex() const
            {
                return 1;
            }
            
            std::size_t maxindex() const
            {
                return maxindex_;
            }
            
            //! \brief Access elements by 1-based BFS indexes (i.e., only root has index 1)
            T& operator[](std::size_t idx)
            {
                if (idx > maxindex_ || idx == 0) throw std::domain_error("Index of VEB tree out of range.");
                
                return data_[bfs2veb(idx)];
            }

            //! \brief Access const elements by 1-based BFS indexes (i.e., only root has index 1)
            const T& operator[](std::size_t idx) const
            {
                if (idx > maxindex_ || idx == 0) throw std::domain_error("Index of VEB tree out of range.");
                
                return data_[bfs2veb(idx)];
            }

            //! \brief Computes the 0-based offset from a BFS 1-based indexing (i.e., only root has index 1)
            std::size_t bfs2veb(std::size_t idx)
            {
                if (idx > maxindex_ || idx == 0) throw std::domain_error("Index of VEB tree out of range.");
                
                // User precomputed indices if possible
                if (h_ <= 12)
                {
                    return vebindices::indices_[vebindices::offsets_[h_ - 1] + (idx - 1)];
                }
                
                const std::size_t maxheight = 2;
                
                // VEB cocincides with BFS with 3 nodes
//                if (false && (h_ <= maxheight) || (idx < 4))
//                {
//                    return idx - 1;
//                }
                
                // Initial height
                std::size_t temp = h_;
                
                // Offset from the beginning of the data_ array
                std::size_t offset = 0;
                
                while (temp > maxheight)
                {
                    std::size_t ntop, midl, botl, nbot, hbot, nidx;
                    
                    // Middle level starting from 0
                    midl = (temp >> 1);
                    
                    // Number of elements on top half
                    ntop = (2 << midl) - 1;
                    
                    // Bottom half starting level (0 is root)
                    botl = midl + 1;
                    
                    // Bottom chunks height
                    hbot = temp - botl;
                    
                    // Number of sub-trees in the bottom half
                    nbot = 2 << hbot;
                    
                    // Index is in the top half
                    if (idx <= ntop)
                    {
                        //offset += ntop;
                        temp    = midl + 1;
                    }
                    // Index is in the bottom half
                    else
                    {
                        // Is the node on the right child of its parent
                        bool rightchild = idx & 1;
                        
                        // Parent index
                        std::size_t parent = (idx & (~static_cast<std::size_t>(0) << 1)) >> 1;
                        
                        // New index, starting from the (putative) parent on top half, is 1
                        nidx = 1;
                        
                        // Find the parent in the top half
                        while (parent > ntop)
                        {
                            nidx       = (nidx << 1) + rightchild;
                            rightchild = parent & 1;
                            parent     = (parent & (~static_cast<std::size_t>(0) << 1)) >> 1;
                        }
                        
                        // Number of bottom blocks to skip
                        std::size_t nblocks = (parent - (1 << midl)) * 2 + rightchild;
                        
                        // Adjust index, offset, new index, and height
                        offset += ntop + nblocks * ((1 << hbot) - 1);
                        temp    = hbot;
                        idx     = nidx;
                    } // End computing offset
                }
                
                // Remaining height is 2, return 0-based index plus offset
                return offset + (idx - 1);
            }
            
        private:
            
            //! \brief Height of the tree
            const std::size_t h_;
            
            //! \brief Data in the tree
            T* data_;
            
            //! \brief Total number of nodes (index-1)
            const std::size_t maxindex_;
        };
    }
}

#endif
