/*!
 *  \file timer.h
 *
 *  \copyright Copyright (c) 2014 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */

#ifndef sequence_timer_h
#define sequence_timer_h

#include <iostream>
#include <chrono>
#include <string>
#include <ctime>
#include <cxxabi.h>

class X {
public:
    template<typename T> void operator[](T const&){
        size_t len;
        int s;
        char* p=abi::__cxa_demangle(typeid(T).name(), 0, &len, &s);
        std::cout << typeid(T).name() << ": " << p << std::endl;
    }
};

#include "common.h"

namespace seq
{
    
    /*!
     * A timer class used to measure wall-clock time
     */
    class timer
    {
        
        //! \brief Time points
        std::chrono::steady_clock::time_point s_, e_;
        
        //! \brief Timer description
        std::string log_;
        
        //! \brief Internal function that returns a human-readable time string
        template <class T>
        static std::string timerstring(T time)
        {
            std::string out;
            
            auto const msecs = time % 1000;
            time /= 1000;
            auto const secs = time % 60;
            time /= 60;
            auto const mins = time % 60;
            time /= 60;
            auto const hours = time % 24;
            time /= 24;
            auto const days = time;
            
            bool printed_earlier = false;
            
            if (days >= 1)
            {
                printed_earlier = true;
                out += std::to_string(days) + " d ";
            }
            
            if (printed_earlier || hours >= 1)
            {
                printed_earlier = true;
                out += std::to_string(hours) + " h ";
            }
            
            if (printed_earlier || mins >= 1)
            {
                printed_earlier = true;
                out += std::to_string(mins) + " m ";
            }
            
            if (printed_earlier || secs >= 1)
            {
                printed_earlier = true;
                out += std::to_string(secs) + " s ";
            }
            
            if (printed_earlier || msecs >= 1)
            {
                printed_earlier = true;
                out += std::to_string(msecs) + " ms";
            }
            
            return out;
        }
        
    public:
        
        //! \brief Start the timer
        inline void start(std::string log)
        {
            // Buffer
            char times[100];
            
            // Get current time and date
            auto timet = std::time(NULL);
            
            log_ = log;
            
            std::strftime(times, sizeof(times), "%Y/%m/%d %H:%M:%S", std::localtime(&timet));
            
            std::cout << std::endl << "[start] " << times << " --- " << log << std::endl;
            
            // START MEASURING
            s_ = std::chrono::steady_clock::now();
        }
        
        //! \brief Stop the timer
        inline void stop()
        {
            // STOP MEASURING
            e_ = std::chrono::steady_clock::now();
         
            // Buffer
            char times[100];
            
            // Compute difference for humans
            auto diff  = std::chrono::duration_cast<std::chrono::milliseconds>(e_ - s_).count();
            
            // Get current time and date
            auto timet = std::time(NULL);
            
            auto out = timerstring(diff);
            
            std::strftime(times, sizeof(times), "%Y/%m/%d %H:%M:%S", std::localtime(&timet));
            
            std::cout << "[ end ] " << times << " --- " << log_ << " : " << out << " (" << diff << " ms)" << std::endl;
        }
        
        //! \brief Measure a function runtime
        template <class F, class... Args>
        static void oneshot(std::string log, F function, Args... args)
        {
            // Buffer
            char times[100];
            
            // Output string
            std::string out;
            
            // Get current time and date
            auto timet = std::time(NULL);
            
            std::strftime(times, sizeof(times), "%Y/%m/%d %H:%M:%S", std::localtime(&timet));
            
            std::cout << std::endl << "[start] " << times << " --- " << log << std::endl;
            
            // START MEASURING
            auto s = std::chrono::steady_clock::now();
            
            function(args...);
            
            // STOP MEASURING
            auto e = std::chrono::steady_clock::now();
            
            // Compute difference for humans
            auto diff  = std::chrono::duration_cast<std::chrono::milliseconds>(e - s).count();
            
            // Get current time and date
            timet = std::time(NULL);
            
            out = timerstring(diff);
            
            std::strftime(times, sizeof(times), "%Y/%m/%d %H:%M:%S", std::localtime(&timet));
            
            std::cout << "[ end ] " << times << " --- " << log << " : " << out << " (" << diff << " ms)" << std::endl << std::endl;
        }
    };


}
#endif
