/*!
 *  \file parser.h
 *
 *  \copyright Copyright (c) 2014 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */

#ifndef sequence_parsers_h
#define sequence_parsers_h

#include <boost/algorithm/string/trim.hpp>
#include "sequence.h"
#include "reader.h"

namespace seq
{
    template<typename Reader>
    class base_parser
    {
        public:
            typedef Reader reader_type;

        private:
            reader_type reader_;
            typename reader_traits<reader_type>::position_type pos_ { 0 };

        protected:
            explicit base_parser (const char * filename) {
                reader_.open (filename);
            }

            std::string readline () {
                std::string line;
                char c;

                c = reader_traits<reader_type>::get (reader_, pos_++);
                while (c != '\n' and c != '\0') {
                    line += c;
                    c = reader_traits<reader_type>::get (reader_, pos_++);
                }

                return line;
            }
    };

    template<typename Parser>
    class sequence_iterator;

    template<typename Reader>
    class fasta_parser : private base_parser<Reader>
    {
        public:
            typedef typename base_parser<Reader>::reader_type reader_type;

        public:
            typedef fasta_string sequence_type;

            explicit fasta_parser (const char * filename) : base_parser<reader_type>(filename) { }

            fasta_string next () {
                std::string hdr, seq;

                hdr = base_parser<reader_type>::readline ();
                seq = base_parser<reader_type>::readline ();

                return fasta_string (std::move(seq), std::move (hdr));
            }

            sequence_iterator<fasta_parser> begin ();
            sequence_iterator<fasta_parser> end ();
    };

    template<typename Reader>
    class fastq_parser : private base_parser<Reader>
    {
        public:
            typedef typename base_parser<Reader>::reader_type reader_type;

        public:
            typedef fastq_string sequence_type;

            explicit fastq_parser (const char * filename) : base_parser<reader_type>(filename) { }

            fastq_string next () {
                std::string hdr, seq, opt, qual;

                hdr = base_parser<reader_type>::readline ();
                seq = base_parser<reader_type>::readline ();
                opt = base_parser<reader_type>::readline ();
                qual = base_parser<reader_type>::readline ();

                if (!seq.empty ()) {
                    boost::algorithm::trim (seq);
                    boost::algorithm::trim (qual);

                    assert (hdr[0] == '@');
                    assert (seq.size () != 0);
                    assert (opt[0] == '+');
                    assert (qual.size () != 0);
                }

                return fastq_string (std::move (seq), std::move (hdr), std::move (opt), std::move (qual));
            }

            sequence_iterator<fastq_parser> begin ();
            sequence_iterator<fastq_parser> end ();
    };

    template<typename Parser>
    class parser_traits
    {
        public:
            typedef Parser                              parser_type;
            typedef typename parser_type::sequence_type sequence_type;

            static sequence_type next (parser_type & p) {
                return p.next ();
            }
    };

    template<typename Parser>
    class sequence_iterator : public std::iterator<std::forward_iterator_tag, typename parser_traits<Parser>::sequence_type>
    {
        public:
            typedef typename parser_traits<Parser>::parser_type   parser_type;
            typedef typename parser_traits<Parser>::sequence_type sequence_type;

        private:
             parser_type * parser_;
             sequence_type read_;

        public:
            sequence_iterator () = default;
            sequence_iterator (parser_type * p) : parser_ (p) {
                read_ = parser_traits<parser_type>::next (*parser_);
            }
            sequence_iterator (sequence_iterator &&) = default;

            ~sequence_iterator () = default;

            sequence_iterator & operator = (sequence_iterator &&) = default;
            sequence_iterator & operator = (const sequence_iterator &) = default;

            bool operator == (const sequence_iterator & other) noexcept {
                return parser_ == other.parser_;
            }

            bool operator != (const sequence_iterator & other) noexcept {
                return !(this->operator == (other));
            }

            sequence_type operator * () {
                return read_;
            }

            sequence_type * operator -> () {
                return &read_;
            }

            sequence_iterator & operator ++ () {
                read_ = parser_traits<parser_type>::next (*parser_);

                if (read_.get ().empty ()) {
                    parser_ = nullptr;
                }

                return *this;
            }

            sequence_iterator & operator ++ (int) {
                auto it = *this;

                (void) this->operator ++ ();

                return it;
            }
    };

    template<typename Reader>
    sequence_iterator<fasta_parser<Reader>> fasta_parser<Reader>::begin () {
        return sequence_iterator<fasta_parser<Reader>> (this);
    }

    template<typename Reader>
    sequence_iterator<fasta_parser<Reader>> fasta_parser<Reader>::end () {
        return sequence_iterator<fasta_parser<Reader>> ();
    }

    template<typename Reader>
    sequence_iterator<fastq_parser<Reader>> fastq_parser<Reader>::begin () {
        return sequence_iterator<fastq_parser<Reader>> (this);
    }

    template<typename Reader>
    sequence_iterator<fastq_parser<Reader>> fastq_parser<Reader>::end () {
        return sequence_iterator<fastq_parser<Reader>> ();
    }

    using fasta_stream_parser = fasta_parser<allocating_reader<stream_source>>;
    using fastq_stream_parser = fastq_parser<allocating_reader<stream_source>>;    
}

#endif

