/*!
 *  \file ofmpm.h
 *
 *  \copyright Copyright (c) 2016 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */


#ifndef ofmpm_hpp
#define ofmpm_hpp

#include <iostream>
#include <vector>
#include <algorithm>
#include <numeric>
#include <random>
#include <atomic>
#include <limits>
#include <iomanip>

namespace seq
{
    namespace details
    {
        /*!
         * \brief This class implements the accessor traits that enables mapping between a container and an OFMPM
         */
        template <class Accessor>
        class accessor_traits
        {
        public:
            
            //! \brief Convert from a type, i.e., a contaner's value_type
            typedef typename Accessor::from_type from_type;

            //! \brief Convert to a type, i.e., the OFMPM value_type
            typedef typename Accessor::to_type to_type;

            //! \brief Converts a type in a container to an OFMPM one
            static to_type convert(from_type v)
            {
                return Accessor::convert(v);
            }

            //! \brief Compatibility conversion used in the (re)distribution
            static to_type convert(to_type v)
            {
                return v;
            }

            //! \brief Tells whether type in an OFMPM is occupied or empty
            static bool is_occupied(to_type v)
            {
                return Accessor::is_occupied(v);
            }

            //! \brief Tells whether type in an OFMPM is occupied or empty
            static constexpr bool is_occupied(from_type v)
            {
                return true;
            }
            
            //! \brief Marks an item slot as empty
            static void mark_empty(to_type &v)
            {
                Accessor::mark_empty(v);
            }
            
            //! \brief Merges two items, returning the results
            static to_type merge(to_type v, to_type w)
            {
                return Accessor::merge(v, w);
            }
            
            //! \brief Compares two items, returning if v <= w
            static bool compare (to_type v, to_type w)
            {
                return Accessor::compare(v, w);
            }

            //! \brief Handy function for debugging initial types
            static std::string fromtype_string(from_type v)
            {
                return Accessor::fromtype_string(v);
            }

            //! \brief Handy function for debugging storage types
            static std::string totype_string(to_type v)
            {
                return Accessor::totype_string(v);
            }
            
        };
        
        /*!
         * \brief This class implements a limited Oredered File Maintenance Packet Memory
         */
        template <class T, class Accessor>
        class ofmpm
        {
            //! \brief This class implements an iterator on a OFMPM data
            class ofmpm_iterator : public std::iterator<std::input_iterator_tag, T>
            {
                friend class ofmpm;
                
                //! \brief The OFMPM class reference
                ofmpm<T, Accessor> &parent_;
                
                //! \brief The initial offset
                std::size_t idx_;

                //! \brief Limits of the current OFMPM data
                const std::size_t limit_;

            public:
                
                //! \brief Standard-friendly definition of difference type
                typedef std::size_t difference_type;

                //! \brief Standard-friendly definition of referenced type
                typedef T value_type;
                
                //! \brief No default constructor allowed
                ofmpm_iterator() = delete;
                
                //! \brief Constructs an interator on a OFMPM, with an offset
                ofmpm_iterator(ofmpm<T, Accessor> &parent, std::size_t idx) : parent_(parent), idx_(idx), limit_(parent_.numbuckets_ * parent_.bucketsize_)
                {
                    // NOP
                }
                
                //! \brief Finds the next non-empty item (pre-increment)
                ofmpm_iterator& operator++()
                {
                    std::size_t last = idx_;
                    
                    // Return out of boundary index if we cannot go further
                    if (idx_ == parent_.end_ || idx_++ >= limit_)
                    {
                        idx_ = parent_.end_;
                    }
                    else
                    {
                        while (!accessor_traits<Accessor>::is_occupied(parent_.data_[idx_]) && idx_ < limit_) ++idx_;
                        
                        if (idx_ == limit_) idx_ = last + 1;
                    }

                    return *this;
                };

                //! \brief Finds the next non-empty item (post-increment)
                ofmpm_iterator operator++(int)
                {
                    auto p = *this;
                    
                    ++(*this);
                    
                    return p;
                }
                
                //! \brief Finds the previous non-empty item (pre-decrement)
                ofmpm_iterator& operator--()
                {
                    // Return out of boundary index if we cannot go further
                    if (idx_ == 0)
                    {
                        *this = parent_.end();
                    }
                    else
                    {
                        --idx_;
                        
                        while (!accessor_traits<Accessor>::is_occupied(parent_.data_[idx_]) && idx_ > 0) --idx_;
                        
                        if (idx_ == 0 && !accessor_traits<Accessor>::is_occupied(parent_.data_[idx_])) *this = parent_.end();
                    }
                    
                    return *this;
                };
                
                //! \brief Finds the previous non-empty item (post-decrement)
                ofmpm_iterator operator--(int)
                {
                    auto p = *this;
                    
                    --(*this);
                    
                    return p;
                }

                //! \brief Returns the difference between two iterators
                difference_type operator-(ofmpm_iterator it)
                {
                    return idx_ - it.idx_;
                }
                
                //! \brief Access an element of the OFMPM
                value_type& operator*()
                {
                    return parent_.data_[idx_];
                };
                
                //! \brief Returns the index inside the OFMPM
                std::size_t operator&()
                {
                    return idx_;
                }
                
                //! \brief Compares two iterators, returns if they're different
                bool operator!=(const ofmpm_iterator it)
                {
                    return it.idx_ != idx_;
                }

                //! \brief Compares two iterators, returns if they're equal
                bool operator==(const ofmpm_iterator it)
                {
                    return it.idx_ == idx_;
                }
                
                //! \brief Swaps two iterators
                void swap(ofmpm_iterator &other)
                {
                    if (&parent_ != &other.parent_)
                        throw std::runtime_error("Can only swap iterators referring to the same OFMPM");
                    
                    std::swap(idx_, other.idx_);
                }

                //! \brief Assigns an iterators
                ofmpm_iterator& operator=(const ofmpm_iterator &other)
                {
                    if (&parent_ != &other.parent_)
                        throw std::runtime_error("Can only assign iterators referring to the same OFMPM");
                    
                    idx_ = other.idx_;
                    
                    return *this;
                }
            };
            
            
        public:
            
            //! \brief Create a OFMPM
            template <class Container>
            ofmpm(const Container& c, std::size_t minitems = 1000000,
                                      std::size_t minbucketsize = 1024) :
                                      minbucketsize_(minbucketsize),
                                      numbuckets_(numbuckets(c.size())),
                                      bucketsize_(bucketsize(c.size())),
                                      full(false)
            {
                // We need a minimum number of items...
                if (c.size() < minitems)
                    throw std::domain_error("A ofmpm can only be constructed from a container of at least " + std::to_string(minitems) + " items");

                // ... And we need a sorted container
                if (!std::is_sorted(c.begin(), c.end()))
                    throw std::domain_error("The initial container must be a sorted collection of items");
            
                // Allocate space for items
                data_ = new T[numbuckets_ * bucketsize_];

                // Allocate space for densities
                density_.resize(numbuckets_, 0);
                
                // Generate random gaps
                generategaps(bucketsize_ - c.size() / numbuckets_);
                
                // Distribute the initial container's items
                redistribute<decltype(std::begin(c)), true>(T(), std::begin(c), std::end(c), 0, numbuckets_);
            }
            
            // REMOVE THIS
            void dump()
            {
                std::cout << "+++ n buckets = " << numbuckets_ << " size of buckets = " << bucketsize_ << std::endl;
                
                // Dump
                if (true)
                {
                    for (std::size_t b = 0; b < numbuckets_; b++)
                    {
                        std::cout << ">>> BUCKET " << b << " DENSITY " << density_[b] << std::endl;
                        
                        for (std::size_t i = 0; i < bucketsize_; i++)
                        {
                            std::cout << "b = " << b << " i = " << i << " o = " << b * bucketsize_ + i << " value = " <<  accessor_traits<Accessor>::totype_string(data_[b * bucketsize_ + i]) << std::endl;
                        }
                    }
                }
            }
            
            //! \brief Deallocate memory
            ~ofmpm()
            {
                delete[] data_;
            }
            
            //! \brief Returns the first non-empty item of a bucket
            ofmpm_iterator begin(std::size_t b = 0)
            {
                if (b >= numbuckets_)
                    throw std::domain_error("Begin iterator bucket index " + std::to_string(b) + " out of bounds (max index " + std::to_string(numbuckets_) + ")");

                std::size_t idx = b * bucketsize_;

                const std::size_t limit = numbuckets_ * bucketsize_;
                
                while (!accessor_traits<Accessor>::is_occupied(data_[idx]) && idx < limit) ++idx;
                
                return ofmpm_iterator(*this, idx);
            }
            
            //! \brief Returns the past-the-end iterator of a given bucket
            ofmpm_iterator end(std::size_t b = end_)
            {
                const std::size_t limit = numbuckets_ * bucketsize_;
                
                std::size_t idx = (b == end_) ? limit : (b + 1) * bucketsize_;
                
                // Return out of boundary index if we cannot go further
                if (idx > limit || idx == 0)
                    return ofmpm_iterator(*this, limit);
                
                while (!accessor_traits<Accessor>::is_occupied(data_[--idx]) && idx > 0);
                
                return ++ofmpm_iterator(*this, idx);
            }
            
            //! \brief Inserts an element into a bucket
            void insert(T v, std::size_t b)
            {
                if (b >= numbuckets_)
                    throw std::domain_error("Insertion index " + std::to_string(b) + " out of bounds (max index " + std::to_string(numbuckets_) + ")");
                
                // Left/Right iterators, with extremes of the bucket
                ofmpm_iterator it = begin(b), ti = end(b), l = it, r = ti;
                
                std::cout << "begin " << &it << " end " << &ti << std::endl;
                
                // Find leftmost iterator
                while (l != ti && accessor_traits<Accessor>::compare(*l, v))
                {
                    r = l++;
                }
                
                // Left and right are swapped in this implementation, so swap them back
                l.swap(r);
                
                // Check if we found the target item
                if (accessor_traits<Accessor>::compare(v, *l) && accessor_traits<Accessor>::compare(*l, v))
                {
                    std::cout << "FOUND " << std::endl;
                    
                    data_[&l] = accessor_traits<Accessor>::merge(*l, v);

                    return;
                }
                
                std::cout << "left " << &l << " right " << &r << std::endl;
                
                // Free space between l/r iterators
                std::size_t s = 0;
                
                // Insert in the middle
                if (r != ti && l != ti)
                {
                    std::cout << "@MIDDLE" << std::endl;
                    s = r - l - 1;
                }

                // Insert at the end
                if (r == ti && l != ti)
                {
                    std::cout << "@END" << std::endl;
                    s = (b + 1) * bucketsize_ - 1 - &l;
                }

                // Insert at the beginning
                if (r != ti && l == ti)
                {
                    std::cout << "@BEGINNING" << std::endl;
                    s = &r - b * bucketsize_;
                }
                
                std::cout << "free space " << s << std::endl;
                
                // If we have sufficient space, insert it and exit
                if (s >= 1)
                {
                    // Insert in the middle, in absence of a better strategy
                    std::size_t idx = ((l == ti) ? b * bucketsize_ : &l) + ((s == 1) ? s : s >> 1);
                    
                    data_[idx] = v;
                    
                    return;
                }

                // See if we can move things around to accomodate the new item
                // Remember, from this point on we have ZERO free space
                
                // Used to update densities
                auto checkdensity = [this, b]()
                {
                    // If we have no space left, find the current densities
                    if (density_[b] > bucketsize_ * 0.75)
                    {
                        // Must redistribute now, but we have space to proceed
                        full = true;
                    }
                    
                    // Increase density
                    density_[b]++;
                };
                
                // We can move only on the left
                if (l != ti && r == ti)
                {
                    // Save the current right boundary
                    r = l;
                    
                    // Find the first EMPTY slot
                    while (accessor_traits<Accessor>::is_occupied(*l)) l.idx_--;
                    
                    std::cout << "<<<<< " << &l << "--" << &r << std::endl;
                    
                    while (l.idx_ != r.idx_)
                    {
                        data_[&l] = data_[&l + 1];
                        
                        ++(l.idx_);
                    }
                    
                    data_[r.idx_] = v;
                    
                    checkdensity();
                    
                    return;
                }

                // We can move only on the right
                if (l == ti && r != ti)
                {
                    std::cout << ">>>>>" << std::endl;
                }
                
                // We can move both on the right and the left
                if (l != ti && r != ti)
                {
                    std::cout << "<<*>>" << std::endl;
                }
            }
            
            
        private:
        
            //! \brief Compute the integer logarithm in base 2
            std::size_t l2(std::size_t v)
            {
                // The result
                std::size_t l = 0;
                
                while (v > 1)
                {
                    v >>= 1;
                    ++l;
                }
                
                return l;
            }
            
            //! \brief Compute the bucket size for a given number of items
            std::size_t bucketsize(std::size_t n)
            {
                return std::max(l2(n), 10*l2(n));
                //return std::max(l2(n), minbucketsize_);
            }

            //! \brief Compute the bucket size for a given number of items
            std::size_t numbuckets(std::size_t n)
            {
                return 1 << (l2(n / bucketsize(n)) + 1);
            }
            
            //! \brief Generate a sequence of 'ngaps' random offsets (0..bucketsize_)
            void generategaps(std::size_t ngaps)
            {
                // Allocate gap offsets
                gaps_.resize(bucketsize_);
                
                // Create a sequence of offsets
                std::iota(gaps_.begin(), gaps_.end(), 0);
                
                std::random_device rd;
                std::mt19937 g(rd());
                
                // Randomize
                std::shuffle(gaps_.begin(), gaps_.end(), g);
                
                gaps_.resize(ngaps);
                
                std::sort(gaps_.begin(), gaps_.end());
            }
            
            //! \brief Return the next occupied item
            inline std::size_t next(std::size_t index)
            {
                const std::size_t limit = numbuckets_ * bucketsize_;
                
                // Return out of boundary index if we cannot go further
                if (index == end_ || index++ >= limit)
                    return end_;
                
                while (!accessor_traits<Accessor>::is_occupied(data_[index]) && index < limit) ++index;
                
                return index == limit ? end_ : index;
            }

            //! \brief Return the previous occupied item, maybe useless, possibly it has UB
            inline std::size_t prev(std::size_t index)
            {
                const std::size_t limit = numbuckets_ * bucketsize_;

                // Return out of boundary index if we cannot go further
                if (index > limit || index == 0)
                    return end_;
                
                while (!accessor_traits<Accessor>::is_occupied(data_[--index]) && index > 0);

                return index;
            }

            //! \brief Redistribute elements starting from a bucket
            template <class Iterator, bool initial = false>
            void redistribute(T first, Iterator ibegin, Iterator iend, std::size_t bucketbegin, std::size_t nbuckets)
            {
                if (bucketbegin >= numbuckets_ || nbuckets > numbuckets_)
                    throw std::domain_error("Paramerters out of range (begin = " + std::to_string(bucketbegin) + ", num buckets = " + std::to_string(numbuckets_) + ")");

                // Total number of items
                const std::size_t s = initial ? std::distance(ibegin, iend) :
                                                std::accumulate(density_.begin() + bucketbegin, density_.begin() + bucketbegin + nbuckets, 0) + accessor_traits<Accessor>::is_occupied(first);
                
                // Quotient and remainder for computing the number of items per bucket
                std::size_t q = s / numbuckets_;
                std::size_t r = s - q * numbuckets_;

                // Index in the bucket, bucket index, gap index, target items in bucket
                std::size_t i = 0, b = bucketbegin, g = 0, t = 0;
                
                std::size_t idx = 0;
                
                // Should we insert remainders?
                bool h = r > 0;
                
                // Number of items to insert in the current bucket
                t = q + (r > 0);
                
                // Current item to be inserted
                auto p = ibegin;
                
                // Initial distribution
                while (p != iend)
                {
                    // Check for gaps
                    if ((g < gaps_.size()) && (i == gaps_[g]))
                    {
                        // If we have a remainder and we didn't insert a remainder, then skip the gap
                        if (r > 0 && h)
                        {
                            data_[b * bucketsize_ + i++] = accessor_traits<Accessor>::convert(*p++);
                            t--;
                            r--;
                            
                            // Global index
                            idx++;
                            
                            // Skip this gap when inserting a remainder
                            g++;
                            
                            // No more remainders
                            h = false;
                            
                            // Increase density
                            density_[b] += 1;
                            
                            continue;
                        }
                        
                        accessor_traits<Accessor>::mark_empty(data_[b * bucketsize_ + i++]);
                        
                        // Use next gap
                        g++;
                        
                        // Global index
                        idx++;
                        
                        continue;
                    }
                    
                    // If we need to insert an element
                    if (t > 0)
                    {
                        data_[b * bucketsize_ + i++] = accessor_traits<Accessor>::convert(*p++);
                        t--;
                        
                        // Global index
                        idx++;
                        
                        // Increase density
                        density_[b] += 1;
                        
                        continue;
                    }
                    else // Elements to insert are finished
                    {
                        // If we're inside the bucket, clear
                        if (i < bucketsize_)
                        {
                            accessor_traits<Accessor>::mark_empty(data_[b * bucketsize_ + i++]);
                            idx++;
                        }
                        else // Bucket finished, go to the next
                        {
                            i = 0;
                            g = 0;
                            t = q + (r > 0);
                            h = r > 0;
                            ++b;
                        }
                    }
                    continue;
                }
                
                // If we still have spaces to fill, clear
                if (idx < bucketsize_ * numbuckets_)
                {
                    while (idx < bucketsize_ * numbuckets_)
                    {
                        accessor_traits<Accessor>::mark_empty(data_[idx++]);
                    }
                }
            }
            
            //! \brief The storage for the OFMPM
            T* data_;
            
            //! \brief The random list of offsets used to insert gaps into buckets
            std::vector<std::size_t> gaps_;
            
            //! \brief The minimum bucket size
            const std::size_t minbucketsize_;

            //! \brief The actual bucket size
            std::size_t bucketsize_;

            //! \brief The number of buckets
            std::size_t numbuckets_;

            //! \brief The number of buckets
            std::vector<std::size_t> density_;
            
            //! \brief The end-of-buffer index
            const static std::size_t end_ = std::numeric_limits<std::size_t>::max();
            
            //! \brief Must be rearranged and redistributed
            bool full;
        };
        
    }
}

#endif
