/*!
 *  \file kmer.h
 *
 *  \copyright Copyright (c) 2014 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */

#ifndef sequence_kmer_h
#define sequence_kmer_h

#include <type_traits>
#include <iostream>
#include <utility>
#include <unordered_map>
#include <stdexcept>
#include <tbb/concurrent_hash_map.h>

#include "common.h"
#include "sequence.h"

namespace seq
{
    /*!
     * This is a base class for storing kmers, and it shouldn't be used except for
     * testing purposes
     */
    template <class Sequence>
    class kmerlist_vector
    {
    public:
        
        //! \brief The container used for storage
        typedef std::vector<Sequence> container_type;
        
        //! \brief The storage type
        typedef typename container_traits<container_type>::value_type value_type;
        
        //! \brief The constructor of a simple kmer list
        kmerlist_vector(std::size_t size = 1000000)
        {
            list_.reserve(size);
        }
        
        //! \brief Inserts a kmer into the list
        value_type& insert(value_type&& kmer)
        {
            list_.push_back(kmer);
            
            return list_.back();
        }
        
        //! \brief Returns the kmer in the list
        value_type& at(std::size_t i)
        {
            return list_.at(i);
        }

        //! \brief Returns the constant kmer in the list
        const value_type& at(std::size_t i) const
        {
            return list_.at(i);
        }

        //! \brief Returns the actual constant kmer list
        const container_type& get() const
        {
            return list_;
        }

        //! \brief Returns the actual kmer list
        container_type& get()
        {
            return list_;
        }

    private:
        
        //! \brief The header of a read sequence
        container_type list_;
    };
    
    template <class Sequence>
    class kmerlist_hashmap
    {
    public:
        
        //! \brief The container used for storage
        typedef std::unordered_map<typename Sequence::container_type, Sequence> container_type;
        
        //! \brief The storage type
        typedef Sequence value_type;
        
        //! \brief The constructor of a simple kmer list
        kmerlist_hashmap(std::size_t size = 1000000)
        {
            list_.reserve(size);
        }
        
        //! \brief Inserts a kmer into the list
        value_type& insert(value_type&& kmer)
        {
            auto &p = sequence_traits<value_type>::get(kmer);
            list_[p] = kmer;
            
            return list_[p];
        }
        
        //! \brief Returns the constant value of a mapped kmer
        const value_type& at(const container_type& c) const
        {   
            return list_.at(c);
        }

        //! \brief Returns the value of a mapped kmer
        value_type& at(const container_type& c)
        {
            return list_.at(c);
        }

        //! \brief Returns the actual constant kmer list
        const container_type& get() const
        {
            return list_;
        }

        //! \brief Returns the actual kmer list
        container_type& get()
        {
            return list_;
        }

        //! \brief Returns the number of kmers (unique) in the list
        const std::size_t size() const
        {
            return list_.size();
        }
        
    private:
        
        //! \brief The header of a read sequence
        container_type list_;
    };
    
    
    template <class Sequence>
    class kmerlist_concurrent_hashmap
    {
    public:
        
        //! \brief The container used for storage
        typedef tbb::concurrent_hash_map<typename Sequence::container_type, Sequence> container_type;
        
        //! \brief The storage type
        typedef Sequence value_type;
        
        //! \brief The constructor of a simple kmer list
        kmerlist_concurrent_hashmap(std::size_t size = 1000000) : list_(size)
        {
            // NOP
        }
        
        //! \brief Inserts a kmer into the list
        value_type& insert(value_type&& kmer)
        {
            typename container_type::accessor accessor;
            list_.insert(accessor, kmer.get());
            accessor->second = kmer;
            return accessor->second;
        }
        
        //! \brief Returns the constant value of a mapped kmer
        const value_type& at(const container_type& c) const
        {
            typename container_type::const_accessor accessor;
            if (!list_.find(accessor, c))
                throw std::out_of_range("kmerlist_concurrent_hashmap::at");

            return accessor->second;
        }
        
        //! \brief Returns the value of a mapped kmer
        value_type& at(const container_type& c)
        {
            typename container_type::accessor accessor;
            if (!list_.find(accessor, c))
                throw std::out_of_range("kmerlist_concurrent_hashmap::at");
            
            return accessor->second;
        }
        
        //! \brief Returns the actual constant kmer list
        const container_type& get() const
        {
            return list_;
        }
        
        //! \brief Returns the actual kmer list
        container_type& get()
        {
            return list_;
        }
        
        //! \brief Returns the number of kmers (unique) in the list
        const std::size_t size() const
        {
            return list_.size();
        }
        
    private:
        
        //! \brief The header of a read sequence
        container_type list_;
    };
    
    /*!
     * This class counts sequences updating their count property
     */
    class counter
    {
    public:
        
        //! \brief Generic counter
        template <class Sequence, class Kmerlist>
        void operator()(Sequence &s, Kmerlist &list)
        {
            auto target = list.get().find(s.get());
            
            if (target != list.get().end())
                s.count() = target->second.count() + 1;
            else
                s.count() = 1;
            
            list.insert(std::move(s));
        }
        
        //! \brief Concurrent
        template <class Sequence, class Kmer>
        void operator()(Sequence &s, kmerlist_concurrent_hashmap<Kmer> &list)
        {
            typename kmerlist_concurrent_hashmap<Kmer>::container_type::accessor accessor;
            list.get().insert(accessor, s.get());
            std::size_t c = accessor->second.count();
            accessor->second = s;
            accessor->second.count() = c + 1;
        }
        
        //! \brief Concurrent paired-ended fwd/rev sequences from FASTQ
        void operator()(fastq_string &s, kmerlist_concurrent_hashmap<paired_revfwd> &list)
        {
            typename paired_revfwd::quality_function fn;
            
            auto seq = s.get();
            paired_revfwd tmp(std::move(seq));

            typename kmerlist_concurrent_hashmap<paired_revfwd>::container_type::accessor accessor;
            list.get().insert(accessor, tmp.get());
            tmp.quality()    = fn(s.quality());
            tmp.count()      = accessor->second.count() + 1;
            accessor->second = std::move(tmp);
        }

        //! \brief Concurrent paired-ended fwd/rev sequences
        void operator()(paired_revfwd &s, kmerlist_concurrent_hashmap<paired_revfwd> &list)
        {
            auto seq = s.get();
            paired_revfwd tmp(std::move(seq));
            
            typename kmerlist_concurrent_hashmap<paired_revfwd>::container_type::accessor accessor;
            list.get().insert(accessor, tmp.get());
            tmp.quality()    = s.quality();
            tmp.count()      = accessor->second.count() + 1;
            accessor->second = std::move(tmp);
        }
    };
        
    //! \brief Generic kmer-izer, using two loops
    template<class Read, class Kmerlist, class Fn>
    void kmerize(Read &read, Kmerlist &list, std::size_t klength, Fn fn)
    {
        typedef typename sequence_traits<Read>::container_type read_container;
        typedef typename container_traits<Kmerlist>::value_type::container_type kmer_container;
        typedef typename container_traits<Kmerlist>::value_type kmer_type;
        
        const std::size_t maxlen = container_traits<read_container>::size(read.get());
        
        read_container &r = sequence_traits<Read>::get(read);
        
        for (std::size_t i = 0; i < maxlen - klength + 1; i++)
        {
            kmer_type kmer;
            
            kmer_container &k = sequence_traits<kmer_type>::get(kmer);
            
            container_traits<kmer_container>::resize(k, klength, ' ');
            
            for (std::size_t j = 0; j < klength; j++)
                container_traits<kmer_container>::at(k, j) = container_traits<read_container>::at(r, i + j);
            
            details::convert_property(read, kmer, i, klength);
            
            fn(kmer, list);
        }
    }

    //! \brief Specialization for a concurrent hashmap
    template<class Read, class Kmer, class Fn>
    void kmerize(Read &read, kmerlist_concurrent_hashmap<Kmer> &list, std::size_t klength, Fn fn)
    {
        typedef typename sequence_traits<Read>::container_type read_container;
        typedef typename sequence_traits<Kmer>::container_type kmer_container;
        typedef Kmer kmer_type;
        
        const std::size_t maxlen = container_traits<read_container>::size(read.get());
        
        read_container &r = sequence_traits<Read>::get(read);
        
        for (std::size_t i = 0; i < maxlen - klength + 1; i++)
        {
            kmer_container k;// = sequence_traits<kmer_type>::get(kmer);
            
            container_traits<kmer_container>::resize(k, klength, ' ');
            
            for (std::size_t j = 0; j < klength; j++)
                container_traits<kmer_container>::at(k, j) = container_traits<read_container>::at(r, i + j);
            
            kmer_type kmer(std::move(k));
            
            details::convert_property(read, kmer, i, klength);
            
            fn(kmer, list);
        }
    }
    
    //! \brief This function dumps a list of a generic sequence to a stream in a CSV format
    template <class Kmerlist, class Stream>
    void list_dump(const Kmerlist &list, Stream &s)
    {
        
        constexpr std::size_t psize = std::tuple_size<typename Kmerlist::value_type::properties_type>::value;
        
        s << "Sequence, ";
        for (std::size_t i = 0; i < psize; i++)
            s << "Property " << std::to_string(i) << (i == psize - 1 ? "" : ",");
        s << std::endl;
        
        for(const auto &p : list.get())
        {
            auto &q = p.second;
            s << p.second.get() << ", ";
            details::dump_properties(s, q.properties());
            s << std::endl;
        }
    }

    //! \brief This function dumps a concurrent list of FASTQ sequences to a stream in a CSV format
    template <class Stream>
    void list_dump(const kmerlist_concurrent_hashmap<fastq_string> &list, Stream &s)
    {
        s << "FASTQ Sequence, Header, Optional, Quality" << std::endl;
        
        for(const auto &p : list.get())
        {
            auto &q = p.second;
            s << p.second.get() << ", ";
            details::dump_properties(s, q.properties());
            s << std::endl;
        }
    }

    //! \brief This function dumps a concurrent list of FASTQ sequences to a stream in a CSV format
    template <class Stream>
    void list_dump(const kmerlist_concurrent_hashmap<paired_revfwd> &list, Stream &s)
    {
        s << "FWD/REV Sequence, Quality, Reverse" << std::endl;
        
        for(const auto &p : list.get())
        {
            auto &q = p.second;
            s << p.second.get() << ", ";
            details::dump_properties(s, q.properties());
            s << std::endl;
        }
    }

}


#endif
