## Data filepaths ##################
worldpop_tiff = "mwi_ppp_2020.tif"
poly_gpkg = "../shps/polygons/study_polygons.gpkg"
####################################

import os
import json
import yaml
import matplotlib
from mpl_toolkits.axes_grid1 import make_axes_locatable
import matplotlib.pyplot as plt
import geopandas as gpd
import pandas as pd
import seaborn as sns

import rasterio
from rasterio.plot import show
from rasterio.mask import mask
import pycrs

sns.set(palette="Paired", font_scale=0.65,  rc={"legend.loc":'upper right'})

pd.set_option('display.max_rows', 1000)
pd.set_option('display.max_columns', 1000)
pd.set_option('display.width', 180)

worldpop_malawi = rasterio.open(worldpop_tiff)

chikwawa_poly = gpd.read_file(poly_gpkg, layer="chikwawa")
chileka_poly = gpd.read_file(poly_gpkg, layer="chileka")
ndirande_poly = gpd.read_file(poly_gpkg, layer="ndirande")

def clip_raster_by_poly(raster, poly, out_tif):
    "Clip raster by polygon."
    coords = [json.loads(poly.to_json())['features'][0]['geometry']]
    out_img, out_transform = mask(raster, shapes=coords, crop=True)
    out_meta = raster.meta.copy()
    epsg_code = int(worldpop_malawi.crs.data['init'][5:])
    out_meta.update({"driver": "GTiff", 
                     "height": out_img.shape[1], 
                     "width": out_img.shape[2], 
                     "transform": out_transform,
                     "crs": pycrs.parse.from_epsg_code(epsg_code).to_proj4()})

    with rasterio.open(out_tif, "w", **out_meta) as dest:
        dest.write(out_img)

def sum_pixel_values(raster_file, band, null_value_to_ignore):
    "Sum pixel values of raster."
    raster = rasterio.open(raster_file)
    band_arr = raster.read(band)
    band_arr = band_arr[band_arr != null_value_to_ignore]

    return round(sum(band_arr)), band_arr

def plot_pop(tif, arr, poly, title, out_file):
    """Plot raster pop."""
    fig, ax = plt.subplots(1, figsize=(7,9), tight_layout=True, dpi=300)
    show(rasterio.open(tif), cmap='Reds', title=title, ax=ax)  
    divider = make_axes_locatable(ax)
    cax = divider.append_axes('right', size='5%', pad=0.05)
    norm = matplotlib.colors.Normalize(vmin=min(arr), vmax=max(arr))
    matplotlib.colorbar.ColorbarBase(cax, cmap=matplotlib.cm.Reds, norm=norm)
    poly.plot(ax=ax, facecolor=[0,0,0,0], edgecolor=[0,0,0,1])
    fig.savefig(out_file, dpi=300)
    plt.close()

if not os.path.isfile("mwi_ppp_2020_ndirande.tif"):
    print("Clipping Malawi Pop Data!")
    clip_raster_by_poly(worldpop_malawi, ndirande_poly, "mwi_ppp_2020_ndirande.tif")
    print("Done Ndirande!")
    clip_raster_by_poly(worldpop_malawi, chileka_poly, "mwi_ppp_2020_chileka.tif")
    print("Done Chileka!")
    clip_raster_by_poly(worldpop_malawi, chikwawa_poly, "mwi_ppp_2020_chikwawa.tif")
    print("Done Chikwawa!")

ndirande_pop, ndirande_arr = sum_pixel_values("mwi_ppp_2020_ndirande.tif", 1, -99999.)
chileka_pop, chileka_arr = sum_pixel_values("mwi_ppp_2020_chileka.tif", 1, -99999.)
chikwawa_pop, chikwawa_arr = sum_pixel_values("mwi_ppp_2020_chikwawa.tif", 1, -99999.)

plot_pop("mwi_ppp_2020_ndirande.tif", ndirande_arr, ndirande_poly, "Ndirande WorldPop 2020 (100 m)", "wp_ndirande.pdf")
plot_pop("mwi_ppp_2020_chileka.tif", chileka_arr, chileka_poly, "Chileka WorldPop 2020 (100 m)", "wp_chileka.pdf")
plot_pop("mwi_ppp_2020_chikwawa.tif", chikwawa_arr, chikwawa_poly, "Chikwawa WorldPop 2020 (100 m)", "wp_chikwawa.pdf")

WorldPopEstimates = {}
WorldPopEstimates['ndirande'] = ndirande_pop
WorldPopEstimates['chileka'] = chileka_pop
WorldPopEstimates['chikwawa'] = chikwawa_pop

with open("world_pop_numbers.yml", 'w') as f:
    yaml.dump(WorldPopEstimates, f)
