#!/usr/bin/env python

"""Model Script"""
import os
import sys
import pickle
from shutil import copyfile
import numpy as np
import matplotlib.pyplot as plt
import tables as h5
import pandas as pd
from scipy import sparse
import itertools
import geopandas as gpd
from scipy.spatial.distance import pdist, squareform
from shapely.geometry import Point


class Model():
    """Class to setup/parameterise the model"""
    def __init__(self, site, num_days, ticks_per_day, model_dir=''):
        self.site = site
        self.networks_dir = f"../data/networks/{site}/"
        self.save_dir = f"../data/models/{site}/{model_dir}/"

        self.model_filename = self.save_dir+'model_output.h5'

        if not os.path.isdir(self.save_dir):
            os.makedirs(self.save_dir)

        # copy file - general_data
        copyfile(self.networks_dir+"master_pop.h5", self.model_filename)

        # get number of people
        file = h5.open_file(self.model_filename, mode='r')
        self.num_people = len(file.root.demography.people)
        file.close()

        self.num_days = int(num_days)
        self.ticks_per_day = int(ticks_per_day)

    def build_model(self):
        # merge any networks as necessary
        self.merge_networks()

        self.schedule()


    def schedule(self):

        # hourly ticks
        total_ticks = self.ticks_per_day*self.num_days

        schedule = {}

        tick = 0
        day = 1
        day_tick = 0

        while tick < total_ticks:
            schedule[tick] = {'day': day,
                              'day_tick': day_tick,
                              'tick_type': 'Home',
                              'network': 'home_net'}

            # USE COMPARISONS WITH DAY_TICK AND DAY TO ADD EXTRA SCHEDULE DATA (E.G. WORK / MARKET / SCHOOL)

            # print(f"Day: {day}, day tick: {day_tick}, total tick: {tick}, schedule: {schedule[tick]}")

            tick += 1
            day_tick += 1

            if tick % self.ticks_per_day == 0:
                day_tick = 0
                day += 1

        pickle.dump(schedule, open(self.save_dir+'schedule.p', 'wb'))

    def merge_networks(self):
        """
        Copies networks from the networks folder, merges (if required), and
        saves in the model folder
        """
        copyfile(self.networks_dir+"houses_net.npz", self.save_dir+'houses_net.npz')
        copyfile(self.networks_dir+"proximity.npz", self.save_dir+'proximity.npz')


    def merge_two_networks(self, network1, network2):
        """
        Merges two networks together
        """
        connected = np.unique(network1.indices)

        combined = network2.copy()

        for i in connected:
            # set rows to zero
            combined.data[combined.indptr[i]:combined.indptr[i+1]] = 0
            combined.data[np.where(combined.indices==i)[0]] = 0

        combined = combined + network1

        return combined
