//==========================================================================
//  AIDA Detector description implementation 
//--------------------------------------------------------------------------
// Copyright (C) Organisation europeenne pour la Recherche nucleaire (CERN)
// All rights reserved.
//
// For the licensing terms see $DD4hepINSTALL/LICENSE.
// For the list of contributors see $DD4hepINSTALL/doc/CREDITS.
//
// Author     : M.Frank
//
//==========================================================================
//
// Specialized generic detector constructor
// 
//==========================================================================

// Framework include files
#include "DD4hep/DetFactoryHelper.h"
#include "DDCMS/DDCMSPlugins.h"

using namespace std;
using namespace dd4hep;
using namespace dd4hep::cms;

static long algorithm(Detector& /* description */,
                      ParsingContext& ctxt,
                      xml_h e,
                      SensitiveDetector& /* sens */)
{
  typedef vector<double> vecdouble;
  Namespace      ns(ctxt, e, true);
  AlgoArguments  args(ctxt, e);
  string    mother       = args.parentName();
  string    genMat       = args.str("GeneralMaterial");           //General material name
  double    detectorTilt = args.dble("DetectorTilt");             //Detector Tilt
  double    layerL       = args.dble("LayerL");                   //Length of the layer

  double    radiusLo     = args.dble("RadiusLo");                 //Radius for detector at lower level
  int       stringsLo    = args.integer("StringsLo");             //Number of strings      ......
  string    detectorLo   = args.str("StringDetLoName");           //Detector string name   ......

  double    radiusUp     = args.dble("RadiusUp");                 //Radius for detector at upper level
  int       stringsUp    = args.integer("StringsUp");             //Number of strings      ......
  string    detectorUp   = args.str("StringDetUpName");           //Detector string name   ......

  double    cylinderT    = args.dble("CylinderThickness");        //Cylinder thickness
  double    cylinderInR  = args.dble("CylinderInnerRadius");      //Cylinder inner radius
  string    cylinderMat  = args.str("CylinderMaterial");          //Cylinder material
  double    MFRingInR    = args.dble("MFRingInnerRadius");        //Inner Manifold Ring Inner Radius 
  double    MFRingOutR   = args.dble("MFRingOuterRadius");        //Outer Manifold Ring Outer Radius 
  double    MFRingT      = args.dble("MFRingThickness");          //Manifold Ring Thickness
  double    MFRingDz     = args.dble("MFRingDeltaz");             //Manifold Ring Half Lenght
  string    MFIntRingMat = args.str("MFIntRingMaterial");         //Manifold Ring Material
  string    MFExtRingMat = args.str("MFExtRingMaterial");         //Manifold Ring Material

  double    supportT     = args.dble("SupportThickness");         //Cylinder barrel CF skin thickness

  string    centMat      = args.str("CentRingMaterial");          //Central rings  material
  vecdouble centRing1par = args.vecDble("CentRing1");             //Central rings parameters
  vecdouble centRing2par = args.vecDble("CentRing2");             //Central rings parameters

  string    fillerMat    = args.str("FillerMaterial");            //Filler material
  double    fillerDz     = args.dble("FillerDeltaz");             //Filler Half Length

  string    ribMat       = args.str("RibMaterial");               //Rib material
  vecdouble ribW         = args.vecDble("RibWidth");              //Rib width
  vecdouble ribPhi       = args.vecDble("RibPhi");                //Rib Phi position

  vecdouble dohmListFW   = args.vecDble("DOHMListFW");            //DOHM/AUX positions in #strings FW
  vecdouble dohmListBW   = args.vecDble("DOHMListBW");            //DOHM/AUX positions in #strings BW

  double    dohmtoMF            = args.dble("DOHMtoMFDist");      //DOHM Distance to MF
  double    dohmCarrierPhiOff   = args.dble("DOHMCarrierPhiOffset"); //DOHM Carrier Phi offset wrt horizontal
  string    dohmPrimName        = args.str("StringDOHMPrimName"); //DOHM Primary Logical Volume name
  string    dohmAuxName         = args.str("StringDOHMAuxName");  //DOHM Auxiliary Logical Volume name

  string    dohmCarrierMaterial = args.str("DOHMCarrierMaterial");//DOHM Carrier Material
  string    dohmCableMaterial   = args.str("DOHMCableMaterial");  //DOHM Cable Material
  double    dohmPrimL           = args.dble("DOHMPRIMLength");    //DOHM PRIMary Length
  string    dohmPrimMaterial    = args.str("DOHMPRIMMaterial");   //DOHM PRIMary Material
  double    dohmAuxL            = args.dble("DOHMAUXLength");     //DOHM AUXiliary Length
  string    dohmAuxMaterial     = args.str("DOHMAUXMaterial");    //DOHM AUXiliary Material

  string    pillarMaterial      = args.str("PillarMaterial");     //Pillar Material

  double    fwIntPillarDz       = args.dble("FWIntPillarDz");     //Internal pillar parameters
  double    fwIntPillarDPhi     = args.dble("FWIntPillarDPhi");  
  vecdouble fwIntPillarZ        = args.vecDble("FWIntPillarZ");  
  vecdouble fwIntPillarPhi      = args.vecDble("FWIntPillarPhi");  
  double    bwIntPillarDz       = args.dble("BWIntPillarDz");  
  double    bwIntPillarDPhi     = args.dble("BWIntPillarDPhi");  
  vecdouble bwIntPillarZ        = args.vecDble("BWIntPillarZ");  
  vecdouble bwIntPillarPhi      = args.vecDble("BWIntPillarPhi");  

  double    fwExtPillarDz       = args.dble("FWExtPillarDz");        //External pillar parameters
  double    fwExtPillarDPhi     = args.dble("FWExtPillarDPhi");  
  vecdouble fwExtPillarZ        = args.vecDble("FWExtPillarZ");    
  vecdouble fwExtPillarPhi      = args.vecDble("FWExtPillarPhi");  
  double    bwExtPillarDz       = args.dble("BWExtPillarDz");  
  double    bwExtPillarDPhi     = args.dble("BWExtPillarDPhi");  
  vecdouble bwExtPillarZ        = args.vecDble("BWExtPillarZ");    
  vecdouble bwExtPillarPhi      = args.vecDble("BWExtPillarPhi");  
  
  LogDebug("TIBGeom") << "Parent " << mother
      << " NameSpace " << ns.name
      << " General Material " << genMat;
  LogDebug("TIBGeom") << "Lower layer Radius " << radiusLo
      << " Number " << stringsLo << " String " << detectorLo;
  LogDebug("TIBGeom") << "Upper layer Radius "<< radiusUp
      << " Number " << stringsUp << " String " << detectorUp;
  LogDebug("TIBGeom") << "Cylinder Material/thickness " << cylinderMat << " " << cylinderT 
      << " Rib Material " << ribMat << " at " << ribW.size() << " positions with width/phi";
  for (unsigned int i = 0; i < ribW.size(); i++)  {
    LogDebug("TIBGeom") << "\tribW[" << i << "] = " <<  ribW[i] 
        << "\tribPhi[" << i << "] = " << ribPhi[i]/CLHEP::deg;
  }
  LogDebug("TIBGeom") << "DOHM Primary " << " Material " << dohmPrimMaterial << " Length " << dohmPrimL;
  LogDebug("TIBGeom") << "DOHM Aux     " << " Material " << dohmAuxMaterial << " Length " << dohmAuxL;
  for (double i : dohmListFW) {
    if (i>0.) LogDebug("TIBGeom") << "DOHM Primary at FW Position " << i;
    if (i<0.) LogDebug("TIBGeom") << "DOHM Aux     at FW Position " << -i;
  }
  for (double i : dohmListBW) {
    if (i>0.) LogDebug("TIBGeom") << "DOHM Primary at BW Position " << i;
    if (i<0.) LogDebug("TIBGeom") << "DOHM Aux     at BW Position " << -i;
  }
  LogDebug("TIBGeom") << "FW Internal Pillar [Dz, DPhi] " << fwIntPillarDz << ", " << fwIntPillarDPhi; 
  for (unsigned int i=0; i<fwIntPillarZ.size(); i++) {
    if( fwIntPillarPhi[i]>0. ) { 
      LogDebug("TIBGeom") << " at positions [z, phi] " << fwIntPillarZ[i] << " " << fwIntPillarPhi[i];
    }
  }
  LogDebug("TIBGeom") << "BW Internal Pillar [Dz, DPhi] " << bwIntPillarDz << ", " << bwIntPillarDPhi; 
  for (unsigned int i=0; i<bwIntPillarZ.size(); i++) {
    if( bwIntPillarPhi[i]>0. ) { 
      LogDebug("TIBGeom") << " at positions [z, phi] " << bwIntPillarZ[i] << " " << bwIntPillarPhi[i];
    }
  }
  LogDebug("TIBGeom") << "FW External Pillar [Dz, DPhi] " << fwExtPillarDz << ", " << fwExtPillarDPhi; 
  for (unsigned int i=0; i<fwExtPillarZ.size(); i++) {
    if( fwExtPillarPhi[i]>0. ) { 
      LogDebug("TIBGeom") << " at positions [z, phi] " << fwExtPillarZ[i] << " " << fwExtPillarPhi[i];
    }
  }
  LogDebug("TIBGeom") << "BW External Pillar [Dz, DPhi] " << bwExtPillarDz << ", " << bwExtPillarDPhi; 
  for (unsigned int i=0; i<bwExtPillarZ.size(); i++) {
    if( bwExtPillarPhi[i]>0. ) { 
      LogDebug("TIBGeom") << " at positions [z, phi] " << bwExtPillarZ[i] << " " << bwExtPillarPhi[i];
    }
  }

  string idName = mother;
  double rmin = MFRingInR;
  double rmax = MFRingOutR;
  Solid  solid = ns.addSolidNS(idName,Tube(rmin, rmax, 0.5*layerL));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << genMat << " from 0 to " << CLHEP::twopi/CLHEP::deg 
      << " with Rin " << rmin << " Rout " << rmax << " ZHalf " << 0.5*layerL;
  Volume layer = ns.addVolumeNS(Volume(idName, solid, ns.material(genMat)));

  //Internal layer first
  double rin  = rmin+MFRingT;
  //  double rout = 0.5*(radiusLo+radiusUp-cylinderT);
  double rout = cylinderInR;
  string name = idName + "Down";
  solid = ns.addSolidNS(name,Tube(rin, rout, 0.5*layerL));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << genMat << " from 0 to " << CLHEP::twopi/CLHEP::deg 
      << " with Rin " << rin << " Rout " << rout 
      << " ZHalf " << 0.5*layerL;
  Volume layerIn = ns.addVolumeNS(Volume(name, solid, ns.material(genMat)));
  layer.placeVolume(layerIn, 1); // copyNr=1 !
  LogDebug("TIBGeom") << layerIn.name()
      << " number 1 positioned in " << layer.name()
      << " at (0,0,0) with no rotation";

  double rposdet = radiusLo;
  double dphi    = CLHEP::twopi/stringsLo;
  Volume detIn   = ns.volume(detectorLo);
  for (int n = 0; n < stringsLo; n++) {
    double phi    = (n+0.5)*dphi;
    double phix   = phi - detectorTilt + 90*CLHEP::deg;
    double theta  = 90*CLHEP::deg;
    double phiy   = phix + 90.*CLHEP::deg;
    Rotation3D rotation = make_rotation3D(theta, phix, theta, phiy, 0., 0.);
    Position   trdet(rposdet*cos(phi), rposdet*sin(phi), 0);
    layerIn.placeVolume(detIn, n+1, Transform3D(rotation,trdet)); // copyNr=n+1
    LogDebug("TIBGeom") << detIn.name() << " number " << n+1 << " positioned in " 
        << layerIn.name() << " at " << trdet << " with "
        << rotation;
  }
  //Now the external layer
  rin  = cylinderInR + cylinderT;
  rout = rmax-MFRingT;
  name = idName + "Up";
  solid = ns.addSolidNS(name,Tube(rin, rout, 0.5*layerL));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << genMat << " from 0 to " << CLHEP::twopi/CLHEP::deg 
      << " with Rin " << rin << " Rout " << rout
      << " ZHalf " << 0.5*layerL;
  Volume layerOut = ns.addVolumeNS(Volume(name, solid, ns.material(genMat)));
  layer.placeVolume(layerOut, 1); // CopyNr 1 
  LogDebug("TIBGeom") << layerOut.name() 
      << " number 1 positioned in " << layer.name() 
      << " at (0,0,0) with no rotation";

  rposdet = radiusUp;
  dphi    = CLHEP::twopi/stringsUp;
  Volume detOut = ns.volume(detectorUp);
  for (int n = 0; n < stringsUp; n++) {
    double phi    = (n+0.5)*dphi;
    double phix   = phi - detectorTilt - 90*CLHEP::deg;
    double theta  = 90*CLHEP::deg;
    double phiy   = phix + 90.*CLHEP::deg;
    Rotation3D rotation = make_rotation3D(theta, phix, theta, phiy, 0., 0.);
    Position   trdet(rposdet*cos(phi), rposdet*sin(phi), 0);
    layerOut.placeVolume(detOut, n+1, Transform3D(rotation,trdet));
    LogDebug("TIBGeom") << "DDTIBLayerAlgo test " << detectorUp
        << " number " << n+1 << " positioned in " 
        << layerOut.name() << " at " << trdet << " with "
        << rotation;
  }

  //
  // Inner cylinder, support wall and ribs
  //
  // External skins
  rin   = cylinderInR;
  rout  = cylinderInR+cylinderT;
  name  = idName + "Cylinder";
  solid = ns.addSolidNS(name, Tube(rin, rout, 0.5*layerL));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << cylinderMat << " from 0 to " 
      << CLHEP::twopi/CLHEP::deg << " with Rin " << rin 
      << " Rout " << rout << " ZHalf " << 0.5*layerL;
  Volume cylinder = ns.addVolumeNS(Volume(name, solid, ns.material(cylinderMat)));
  layer.placeVolume(cylinder, 1); // CopyNr = 1
  LogDebug("TIBGeom") << cylinder.name() 
      << " number 1 positioned in " << layer.name()
      << " at (0,0,0) with no rotation";
  //
  // inner part of the cylinder
  //
  rin  += supportT;
  rout -= supportT;
  name  = idName + "CylinderIn";
  solid = ns.addSolidNS(name, Tube(rin, rout, 0.5*layerL));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of "
      << genMat << " from 0 to " << CLHEP::twopi/CLHEP::deg 
      << " with Rin " << rin << " Rout " << rout 
      << " ZHalf " << 0.5*layerL;
  Volume cylinderIn = ns.addVolumeNS(Volume(name, solid, ns.material(genMat)));
  cylinder.placeVolume(cylinderIn, 1);
  LogDebug("TIBGeom") << cylinderIn.name() 
      << " number 1 positioned in " << cylinder.name() 
      << " at (0,0,0) with no rotation";
  //
  // Filler Rings
  //
  name = idName + "Filler";
  solid = ns.addSolidNS(name,Tube(rin, rout, fillerDz));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << fillerMat << " from " << 0. << " to "
      << CLHEP::twopi/CLHEP::deg << " with Rin " << rin 
      << " Rout " << rout << " ZHalf "  << fillerDz;
  Volume cylinderFiller = ns.addVolumeNS(Volume(name,solid,ns.material(fillerMat)));
  cylinderIn.placeVolume(cylinderFiller, 1, Position(0.0, 0.0, 0.5*layerL-fillerDz)); // copyNr 1
  cylinderIn.placeVolume(cylinderFiller, 2, Position(0.0, 0.0,-0.5*layerL+fillerDz)); // copyNr 2
  LogDebug("TIBGeom") << "DDTIBLayerAlgo test " << cylinderFiller.name()
      << " number 1" << " positioned in " 
      << cylinderIn.name() << " at " << Position(0.0, 0.0, 0.5*layerL-fillerDz)
      << " number 2" << " positioned in " 
      << cylinderIn.name() << " at " << Position(0.0, 0.0,-0.5*layerL+fillerDz);
  //
  // Ribs
  //
  Material matrib = ns.material(ribMat);
  for (size_t i = 0; i < ribW.size(); i++) {
    name = idName + "Rib" + std::to_string(i);
    double width = 2.*ribW[i]/(rin+rout);
    double dz    = 0.5*layerL-2.*fillerDz;
    double _rmi  = std::min(rin+0.5*CLHEP::mm,rout-0.5*CLHEP::mm);
    double _rma  = std::max(rin+0.5*CLHEP::mm,rout-0.5*CLHEP::mm);
    solid = ns.addSolidNS(name,Tube(_rmi,_rma,dz,-0.5*width, width));
    LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
        << ribMat << " from " << -0.5*width/CLHEP::deg <<" to "
        << 0.5*width/CLHEP::deg << " with Rin " 
        << rin+0.5*CLHEP::mm << " Rout " 
        << rout-0.5*CLHEP::mm << " ZHalf "  << dz;
    Volume cylinderRib = ns.addVolumeNS(Volume(name, solid, matrib));
    double phix   = ribPhi[i];
    double theta  = 90*CLHEP::deg;
    double phiy   = phix + 90.*CLHEP::deg;
    Rotation3D rotation = make_rotation3D(theta, phix, theta, phiy, 0., 0.);
    Position tran(0, 0, 0);
    cylinderIn.placeVolume(cylinderRib, 1, Transform3D(rotation,tran));// copyNr=1
    LogDebug("TIBGeom") << cylinderRib.name()
        << " number 1" << " positioned in " 
        << cylinderIn.name() << " at " << tran << " with " 
        << rotation;
  }
  //
  //Manifold rings
  //
  // Inner ones first
  rin   = MFRingInR;
  rout  = rin + MFRingT;
  name  = idName + "InnerMFRing";
  solid = ns.addSolidNS(name,Tube(rin, rout, MFRingDz));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << MFIntRingMat << " from 0 to " 
      << CLHEP::twopi/CLHEP::deg << " with Rin " << rin 
      << " Rout " << rout << " ZHalf " << MFRingDz;
  Volume inmfr = ns.addVolumeNS(Volume(name, solid, ns.material(MFIntRingMat)));
  layer.placeVolume(inmfr, 1, Position(0.0, 0.0, -0.5*layerL+MFRingDz)); // Copy Nr=1
  layer.placeVolume(inmfr, 2, Position(0.0, 0.0, +0.5*layerL+MFRingDz)); // Copy Nr=2
  LogDebug("TIBGeom") << inmfr.name() 
      << " number 1 and 2 positioned in " << layer.name()
      << " at (0,0,+-" << 0.5*layerL-MFRingDz << ") with no rotation";

  // Outer ones
  rout = MFRingOutR;
  rin  = rout - MFRingT;
  name = idName + "OuterMFRing";
  solid= ns.addSolidNS(name,Tube(rin, rout, MFRingDz));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << MFExtRingMat << " from 0 to " 
      << CLHEP::twopi/CLHEP::deg << " with Rin " << rin 
      << " Rout " << rout << " ZHalf " << MFRingDz;

  Volume outmfr = ns.addVolumeNS(Volume(name, solid, ns.material(MFExtRingMat)));
  layer.placeVolume(outmfr, 1, Position(0.0, 0.0, -0.5*layerL+MFRingDz)); // CopyNr=1
  layer.placeVolume(outmfr, 2, Position(0.0, 0.0, +0.5*layerL+MFRingDz)); // CopyNr=2
  LogDebug("TIBGeom") << outmfr.name() 
      << " number 1 and 2 positioned in " << layer.name()
      << " at (0,0,+-" << 0.5*layerL-MFRingDz 
      << ") with no rotation";
  //
  //Central Support rings
  //
  // Ring 1
  double centZ  = centRing1par[0];
  double centDz = 0.5*centRing1par[1];
  rin  = centRing1par[2];
  rout = centRing1par[3];
  name = idName + "CentRing1";
  solid = ns.addSolidNS(name,Tube(rin, rout, centDz));

  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << centMat << " from 0 to " << CLHEP::twopi/CLHEP::deg 
      << " with Rin " << rin << " Rout " << rout 
      << " ZHalf " << centDz;

  Volume cent1 = ns.addVolumeNS(Volume(name, solid, ns.material(centMat)));
  layer.placeVolume(cent1,1, Position(0.0, 0.0, centZ)); // Copy Nr = 1
  LogDebug("TIBGeom") << cent1.name() << " positioned in " << layer.name()
      << " at (0,0," << centZ << ") with no rotation";
  // Ring 2
  centZ  = centRing2par[0];
  centDz = 0.5*centRing2par[1];
  rin  = centRing2par[2];
  rout = centRing2par[3];
  name = idName + "CentRing2";
  solid = ns.addSolidNS(name, Tube(rin, rout, centDz));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of " 
      << centMat << " from 0 to " << CLHEP::twopi/CLHEP::deg 
      << " with Rin " << rin << " Rout " << rout 
      << " ZHalf " << centDz;

  Volume cent2 = ns.addVolumeNS(Volume(name, solid, ns.material(centMat)));
  layer.placeVolume(cent2, 1, Position(0e0,0e0,centZ)); // copyNr=1
  LogDebug("TIBGeom") << cent2.name() 
      << " positioned in " << layer.name()
      << " at (0,0," << centZ << ") with no rotation";
  //
  ////// DOHM
  //
  // Preparing DOHM Carrier solid
  //
  name = idName + "DOHMCarrier";
  double dohmCarrierRin   = MFRingOutR - MFRingT;
  double dohmCarrierRout  = MFRingOutR;
  double dohmCarrierDz    = 0.5*(dohmPrimL+dohmtoMF);
  double dohmCarrierZ     = 0.5*layerL-2.*MFRingDz-dohmCarrierDz;

  solid = ns.addSolidNS(name,Tube(dohmCarrierRin, dohmCarrierRout, 
                                  dohmCarrierPhiOff, dohmCarrierDz, 
                                  180.*CLHEP::deg-2.*dohmCarrierPhiOff));
  LogDebug("TIBGeom") << solid.name() << " Tubs made of "
      << dohmCarrierMaterial << " from "
      << dohmCarrierPhiOff << " to " 
      << 180.*CLHEP::deg-dohmCarrierPhiOff << " with Rin "
      << dohmCarrierRin << " Rout " << MFRingOutR << " ZHalf " 
      << dohmCarrierDz;
  
  // Define FW and BW carrier logical volume and
  // place DOHM Primary and auxiliary modules inside it
  dphi = CLHEP::twopi/stringsUp;

  Rotation3D dohmRotation;
  double dohmR = 0.5*(dohmCarrierRin+dohmCarrierRout);

  for (int j = 0; j<4; j++) {
    vector<double> dohmList;
    Position tran;
    string rotstr;
    Rotation3D rotation;
    int dohmCarrierReplica=0;
    int placeDohm = 0;

    switch (j){
    case 0:
      name = idName + "DOHMCarrierFW";
      dohmList = dohmListFW;
      tran = Position(0., 0., dohmCarrierZ);
      rotstr = idName + "FwUp";
      rotation = Rotation3D();
      dohmCarrierReplica = 1;
      placeDohm=1;
      break;
    case 1:
      name = idName + "DOHMCarrierFW";
      dohmList = dohmListFW;
      tran = Position(0., 0., dohmCarrierZ);
      rotstr = idName + "FwDown";
      rotation = make_rotation3D(90.*CLHEP::deg, 180.*CLHEP::deg, 90.*CLHEP::deg,270.*CLHEP::deg, 0.,0.);
      dohmCarrierReplica = 2;
      placeDohm=0;
      break;
    case 2:
      name = idName + "DOHMCarrierBW";
      dohmList = dohmListBW;
      tran = Position(0., 0., -dohmCarrierZ);
      rotstr = idName + "BwUp";
      rotation = make_rotation3D(90.*CLHEP::deg, 180.*CLHEP::deg, 90.*CLHEP::deg, 90.*CLHEP::deg, 180.*CLHEP::deg, 0.);
      dohmCarrierReplica = 1;
      placeDohm=1;
      break;
    case 3:
      name = idName + "DOHMCarrierBW";
      dohmList = dohmListBW;
      tran = Position(0., 0., -dohmCarrierZ);
      rotstr = idName + "BwDown";
      rotation = make_rotation3D(90.*CLHEP::deg, 0., 90.*CLHEP::deg, 270.*CLHEP::deg, 180.*CLHEP::deg, 0.);
      dohmCarrierReplica = 2;
      placeDohm=0;
      break;
    }

    Volume dohmCarrier = ns.addVolumeNS(Volume(name,solid,ns.material(dohmCarrierMaterial)));
    int primReplica = 0;
    int auxReplica = 0;
#if 0
    for ( size_t i = 0; i < placeDohm*dohmList.size(); i++ )   {
      double phi   = (std::abs(dohmList[i])+0.5-1.)*dphi;
      double phix  = phi + 90*CLHEP::deg;
      double theta = 90*CLHEP::deg;
      double phiy  = phix + 90.*CLHEP::deg;
      dohmRotation = make_rotation3D(theta, phix, theta, phiy, 0., 0.);

      int    dohmReplica = 0;
      double dohmZ = 0.;
      Volume dohm;

      if(dohmList[i]<0.) {
        // Place a Auxiliary DOHM
        dohm  = ns.volume(dohmAuxName);
        dohmZ = dohmCarrierDz - 0.5*dohmAuxL - dohmtoMF;
        primReplica++;
        dohmReplica = primReplica;
      } else {
        // Place a Primary DOHM
        dohm = ns.volume(dohmPrimName);
        dohmZ = dohmCarrierDz - 0.5*dohmPrimL - dohmtoMF;
        auxReplica++;
        dohmReplica = auxReplica;
      }
      Position dohmTrasl(dohmR*cos(phi), dohmR*sin(phi), dohmZ);
      dohmCarrier.placeVolume(dohm,dohmReplica,Transform3D(dohmRotation,dohmTrasl));
      LogDebug("TIBGeom") << dohm.name() 
          << " replica " << dohmReplica << " positioned in " 
          << dohmCarrier.name() << " at " << dohmTrasl << " with "
          << dohmRotation;
    }
#else
    if ( placeDohm || primReplica || auxReplica || dohmR>0e0 )  {} // Avoid warnings
    LogWarn("TIBGeom") << "DOOHM placement sucks for Geant4. ERASED!";
#endif
    layer.placeVolume(dohmCarrier, dohmCarrierReplica, Transform3D(rotation,tran));// copyNr = dohmCarrierReplica
    LogDebug("TIBGeom") << "DDTIBLayerAlgo test "
        << dohmCarrier.name() << " positioned in " << mother
        << " replica " << dohmCarrierReplica << " at "
        << tran << " with " << rotation;
  }
  //
  ////// PILLARS
  for (int j = 0; j<4; j++) {    
    vector<double> pillarZ;
    vector<double> pillarPhi;
    double pillarDz=0, pillarDPhi=0, pillarRin=0, pillarRout=0;
    
    switch (j){
    case 0:
      name = idName + "FWIntPillar";
      pillarZ    = fwIntPillarZ;
      pillarPhi  = fwIntPillarPhi;
      pillarRin  = MFRingInR;
      pillarRout = MFRingInR + MFRingT;
      pillarDz   = fwIntPillarDz;
      pillarDPhi = fwIntPillarDPhi;
      break;
    case 1:
      name = idName + "BWIntPillar";
      pillarZ    = bwIntPillarZ;
      pillarPhi  = bwIntPillarPhi;
      pillarRin  = MFRingInR;
      pillarRout = MFRingInR + MFRingT;
      pillarDz   = bwIntPillarDz;
      pillarDPhi = bwIntPillarDPhi;
      break;
    case 2:
      name = idName + "FWExtPillar";
      pillarZ    = fwExtPillarZ;
      pillarPhi  = fwExtPillarPhi;
      pillarRin  = MFRingOutR - MFRingT;
      pillarRout = MFRingOutR;
      pillarDz   = fwExtPillarDz;
      pillarDPhi = fwExtPillarDPhi;
      break;
    case 3:
      name = idName + "BWExtPillar";
      pillarZ    = bwExtPillarZ;
      pillarPhi  = bwExtPillarPhi;
      pillarRin  = MFRingOutR - MFRingT;
      pillarRout = MFRingOutR;
      pillarDz   = bwExtPillarDz;
      pillarDPhi = bwExtPillarDPhi;
      break;
    }
    
    solid = ns.addSolidNS(name,Tube(pillarRin, pillarRout, pillarDz, -pillarDPhi, 2.*pillarDPhi));    
    Volume Pillar = ns.addVolumeNS(Volume(name,solid,ns.material(pillarMaterial)));
    LogDebug("TIBGeom") << solid.name() << " Tubs made of " << pillarMaterial << " from "
        << -pillarDPhi << " to " << pillarDPhi << " with Rin "
        << pillarRin << " Rout " << pillarRout << " ZHalf "  
        << pillarDz;
    Position   pillarTran;
    Rotation3D pillarRota;
    int pillarReplica = 0;
    for (unsigned int i=0; i<pillarZ.size(); i++) {
      if( pillarPhi[i]>0. ) {
        pillarTran = Position(0., 0., pillarZ[i]);
        pillarRota = make_rotation3D(90.*CLHEP::deg, pillarPhi[i], 90.*CLHEP::deg, 90.*CLHEP::deg+pillarPhi[i], 0., 0.);
        layer.placeVolume(Pillar,i,Transform3D(pillarRota,pillarTran)); // copyNr i
        LogDebug("TIBGeom") << Pillar.name() << " positioned in " 
            << mother << " at "
            << pillarTran << " with " << pillarRota 
            << " copy number " << pillarReplica;
        pillarReplica++;
      }
    }
  }
  return 1;
}

// first argument is the type from the xml file
DECLARE_DDCMS_DETELEMENT(DDCMS_track_DDTIBLayerAlgo,algorithm)
