//==========================================================================
//  AIDA Detector description implementation 
//--------------------------------------------------------------------------
// Copyright (C) Organisation europeenne pour la Recherche nucleaire (CERN)
// All rights reserved.
//
// For the licensing terms see $DD4hepINSTALL/LICENSE.
// For the list of contributors see $DD4hepINSTALL/doc/CREDITS.
//
//==========================================================================
//
//  Simple program to dump the complete DetElement hierarchy
// 
//  Author     : F.Gaede, CERN/DESY
//  Date       : 07 Nov 2014
//==========================================================================

// Framework include files
#include "DD4hep/Detector.h"
#include "DD4hep/DetType.h"
#include "DD4hep/DetectorSelector.h"
#include "DD4hep/DD4hepUnits.h"

#include "DDRec/Surface.h"
#include "DDRec/DetectorSurfaces.h"
#include "DDRec/DetectorData.h"
#include "DDRec/SurfaceHelper.h"

// C/C++ include files
#include <list>


using namespace std ;
using namespace dd4hep ;
using namespace dd4hep::detail;
using namespace dd4hep::rec;


//=============================================================================
static void printDetectorData( DetElement det ){

  try{ 
    FixedPadSizeTPCData* d = det.extension<FixedPadSizeTPCData>() ; 
    std::cout << *d ;
  } catch(...){}
  try{ 
    ZPlanarData* d = det.extension<ZPlanarData>() ; 
    std::cout << *d ;
  } catch(...){}
  try{ 
    ZDiskPetalsData* d = det.extension<ZDiskPetalsData>() ; 
    std::cout << *d ;
  } catch(...){}
  try{ 
    ConicalSupportData* d = det.extension<ConicalSupportData>() ; 
    std::cout << *d ;
  } catch(...){}
  try{ 
    LayeredCalorimeterData* d = det.extension<LayeredCalorimeterData>() ; 
    std::cout << *d ;
  } catch(...){}

}

static void printDetectorSets( std::string name, unsigned int includeFlag,  unsigned int excludeFlag=DetType::IGNORE ){

  Detector& description = Detector::getInstance();
  const std::vector<DetElement>& dets = DetectorSelector(description).detectors( includeFlag, excludeFlag ) ;
  std::cout << " " << name  ;
  for(int i=0,N=dets.size();i<N;++i)  
    std::cout << dets[i].name() << ", " ;
  std::cout << endl ;
} 


//=============================================================================

static int invoke_dump_detector(int argc, char** argv ){
    
  if( argc < 2 ) {
    std::cout << " usage: dumpdetector compact.xml [-s]" 
	      << "  -d :  only print DetectorData objects " 
	      << "  -s :  also print surfaces " 
	      << std::endl ;

    exit(1) ;
  }
  
  std::string inFile =  argv[1] ;


  bool printDetData = ( argc>2 && !strcmp( argv[2] , "-d" ) );

  bool printSurfaces = ( argc>2 && !strcmp( argv[2] , "-s" ) );


  Detector& description = Detector::getInstance();

  description.fromCompact( inFile );

  DetElement world = description.world() ;


  std::cout << "############################################################################### "  << std::endl  ;
  
  Header h = description.header() ;

  std::cout << " detector model : " <<  h.name()  << std::endl 
	    << "    title : "  << h.title() << std::endl 
	    << "    author : " << h.author() << std::endl 
	    << "    status : " << h.status() << std::endl ;


  // print a few sets of detectors (mainly to demonstrate the usage of the detector types )

  printDetectorSets( " barrel trackers : " , ( DetType::TRACKER | DetType::BARREL ) , ( DetType::VERTEX) ) ; 
  printDetectorSets( " endcap trackers : " , ( DetType::TRACKER | DetType::ENDCAP ) , ( DetType::VERTEX) ) ; 

  printDetectorSets( " vertex barrel trackers : " , ( DetType::TRACKER | DetType::BARREL | DetType::VERTEX) ) ; 
  printDetectorSets( " vertex endcap trackers : " , ( DetType::TRACKER | DetType::ENDCAP | DetType::VERTEX) ) ; 

  printDetectorSets( " barrel calorimeters : " , ( DetType::CALORIMETER | DetType::BARREL ) ) ; 
  printDetectorSets( " endcap calorimeters : " , ( DetType::CALORIMETER | DetType::ENDCAP ) ) ; 

  // everything that is not TRACKER or CALORIMETER
  printDetectorSets( " other detecors : " , ( DetType::IGNORE ) , ( DetType::CALORIMETER | DetType::TRACKER ) ) ; 


  if( printDetData ){

    dd4hep::Detector::HandleMap dets = description.detectors() ;

    for( dd4hep::Detector::HandleMap::const_iterator it = dets.begin() ; it != dets.end() ; ++it ){
      
      DetElement det = it->second ;

      std::cout << " ---------------------------- " << det.name() << " ----------------------------- " << std::endl ;

      DetType type( det.typeFlag() ) ;
      
      std::cout << " ------     " << type << std:: endl ;

      printDetectorData( det ) ;

    }
    
    std::cout << "############################################################################### "  << std::endl  ;

    return 0;
  }




  dd4hep::Detector::HandleMap sensDet = description.sensitiveDetectors() ;


  std::cout << "############################################################################### "  << std::endl  
            << "     sensitive detectors:     " << std::endl ;

  for( dd4hep::Detector::HandleMap::const_iterator it = sensDet.begin() ; it != sensDet.end() ; ++it ){

    SensitiveDetector sDet = it->second ;
    std::cout << "     " << it->first << " : type = " << sDet.type() << std::endl ;
  }


  std::cout << "############################################################################### "  << std::endl  << std::endl  ;
	

  //------------------ breadth first tree traversal ---------
  std::list< DetElement > dets ;
  std::list< DetElement > daugs ; 
  std::list< DetElement > gdaugs ; 
  daugs.emplace_back( world ) ;
  while( ! daugs.empty() ) {
    for( std::list< DetElement >::iterator li=daugs.begin() ; li != daugs.end() ; ++li ){
      DetElement dau = *li ;
      DetElement::Children chMap = dau.children() ;
      for ( DetElement::Children::const_iterator it=chMap.begin() ; it != chMap.end() ; ++it ){
        DetElement de = (*it).second ;
        gdaugs.emplace_back( de ) ;
      }  
    }
    dets.splice( dets.end() , daugs ) ;
    daugs.splice( daugs.end() , gdaugs ) ;
  }
  //------------------ end tree traversal ---------
  

  for (  std::list< DetElement >::const_iterator it=dets.begin() ; it != dets.end() ; ++it ){

    DetElement de = (*it) ;
    
    DetElement mother = de.parent() ;
    unsigned parentCount = 0 ;
    while( mother.isValid() ) {
      mother = mother.parent() ;
      ++parentCount ;
    } 

    SurfaceHelper surfMan( de ) ;

    const SurfaceList& sL = surfMan.surfaceList() ;

    std::cout << "DetElement: " ;
    
    for(unsigned i=0 ; i < parentCount ; ++i ) std::cout << "\t" ;

    std::cout << de.name() << "[ path: "<< de.placementPath ()  <<  "] (id: " << de.id() << ") - sens type : " << description.sensitiveDetector( de.name() ).type() << "\t surfaces : " <<  ( sL.empty() ? 0 : sL.size()  ) << std::endl ;


    //    printDetectorData( de ) ;

    if( printSurfaces ){
      for( SurfaceList::const_iterator sit = sL.begin() ; sit != sL.end() ; ++sit ){
	const ISurface* surf =  *sit ;
	std::cout << " ------------------------- " 
		  << " surface: "  << *surf         << std::endl
		  << " ------------------------- "  << std::endl ;
      }
    }
  }

  std::cout << "############################################################################### "  << std::endl  << std::endl  ;

  //	FixedPadSizeTPCData* tpc = tpcDE.extension<FixedPadSizeTPCData>() ;

  return 0;
}


int main(int argc, char** argv ){
  try {
    return invoke_dump_detector(argc,argv);
  }
  catch(const std::exception& e)  {
    std::cout << "Got uncaught exception: " << e.what() << std::endl;
  }
  catch (...)  {
    std::cout << "Got UNKNOWN uncaught exception." << std::endl;
  }
  return EINVAL;    
}

//=============================================================================
