% Mott ring fragmentation model
%
% (c) Joseph Robson, University of Manchester, 2022
% Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
% Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
% Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation 
% and/or other materials provided with the distribution.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ?AS IS? AND ANY EXPRESS OR IMPLIED WARRANTIES, 
% INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
% IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
% OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
% OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
% OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
%
% Note: This code is provided without support.
%
%
function [fragall,sf,x0] = frag_numerical_new(gamma)
%
% Numerical fragmentation model
% Based on Mott 
%
% 1-D rod
%
%
%
% Runs lots of times
reps = 1000;
fragall = [];
for j = 1:reps
sr = 0.5 .* 3.1623e03; % strain rate - chosen to give best fit to Wesenberg
% sr = 3.1e3 %strain rate to match initial results made with incorrect rho
Y = 240e6; % flow stress (6061-T6 yield)
rho = 2700; % density (Note - this was 1e3 too big in earlier model version!)
sfa = 0.1; % minimum strain to failure
fluc = 0.1; % fluctation in strain to failure
bins = 100; % number of bins
allbins = linspace(1,bins,bins); % all bins
activebins = linspace(1,bins,bins); % all bins can fracture at start
tcrack = zeros(1,bins); % crack formation times
cpos = zeros(1,bins); % crack positions
xa = zeros(1,bins); % initial relaxation zone 
relaxbinsu = [];
fracbins =  [];
%
% Geometry
%
l = 0.4; % length of rod (m) 40cm corresponds to circumference of Wesenberg ring
binspace = l./bins; % spacing between bins
%sf_dist = allbins .* (fluc./bins); % strain to failure increases linearly
%sf_dist = log(allbins) .* fluc; % strains to failure clustered at max log
%gamma = 20; % Mott expression - gamma for Fe
fp = exp(gamma .* (1-(allbins*(0.1/bins)))); % fp = exp(gamma .* (1-(allbins*(0.1/bins)))) is good
%plot(fp)
%fp = wblrnd(1,weib,1,bins); % randomly distributed numbers following Weibull (shape = 12)
sf_dist = ((max(fp)-fp)/(max(max(fp)-fp))) .* (fluc);
p = randperm(bins);
sf = sfa + (sf_dist(p));
%
%
%wave = sin(linspace(0,4.*pi)) * fluc;
%sf = sfa + sf_dist(p) + wave; % fluctuations with wave
%
%sf(bins/2) = 0.5.*sfa; % Single defect with half strain to failure
%sf(160) = 0.5.*sfa;
%sf(500) = 0.501.*sfa;
%sf(840) = 0.499.*sfa;
%sf(87) = 0.498.*sfa;
%sf = (randn(1,bins)*fluc) + sfa; % randomized strain to failure,
%normalized random distribution - fluc = 0.01 (standard deviation)
%sf = (rand(1,bins) * fluc) + sfa; % randomized strain to failure fluc = 0.1
%sf = ones(1,bins) .* sfa;
%sf(5) = 0.4;
strainflag = 0; % determines if straining still in progress
t = 0;
dt = (1./3.1623e03)*1e-5; % timestep (s) - reduce to check for convergence
while strainflag == 0
    t = t + dt;
    strain = sr .* t;
    newfracvec = find(strain>sf(activebins));
    newfrac = activebins(newfracvec);
    if length(newfrac)>1
        disp(['Too many fractures in step. Newfrac ' num2str(newfrac)])
    end
    fracbins = [fracbins allbins(newfrac)];
    tcrack(allbins(newfrac)) = t; % initial time of new cracks
    cpos(fracbins) = 1;
    if isempty(fracbins)
        %disp(['Strain ' num2str(strain)])
    else
        %
        % Growth of relaxation zone from each crack
        %
        xa(fracbins) = (((2 .* Y .* (t - tcrack(fracbins)))./(rho .* sr)).^0.5);
        %
        % x in units of bin spacing
        %
        x =  round(xa./binspace);
        relaxbins = [];
        for i=1:bins % loop to work out which bins are relaxed
            if x(i) ~= 0; % if bin has relaxation field around it
                if min(i-x(i))<1 & max(i+x(i))>bins % all bins are relaxed
                    relaxbins = allbins;
                else
                    if min(i-x(i))<1 %relaxation field exceeds bar LHS                        
                        relaxbins = [relaxbins 1:i+x(i)]; % Relax all bins to RHS
                        toprelax = abs(i-x(i)); % wrap around relaxation to top bins
                        relaxbins = [relaxbins (bins-toprelax):bins]; % relax top bins
                    else
                        if max(i+x(i))>bins %relaxation field exceeds bar RHS
                            relaxbins = [relaxbins i-x(i):bins]; % Relax all bins to LHS
                            bottomrelax = (i+x(i))-bins;
                            relaxbins = [relaxbins 1:bottomrelax]; % relax bottom bins
                        else
                            relaxbins = [relaxbins i-x(i):i+x(i)];    
                        end
                    end
                end
            end
        end
        relaxbinsu = unique([relaxbins fracbins]); % remove duplicated from relaxbins
        activebins = allbins;
        activebins(relaxbinsu) = []; % remove bins that are relaxed
        if isequal(relaxbinsu,allbins) % fully relaxed
            strainflag = 1;
        end
    end
       % disp(['Strain ' num2str(strain)])
       % disp(['Active bins ' num2str(activebins)])
       % disp(['Relaxed bins ' num2str(relaxbinsu)])
       % disp(['Fracture pos ' num2str(cpos)])
end
% Post process to capture fragment distribution
fragl = []; % frac length vector
count = 0;
%
% To avoid end effects, only count frags completely in bar
%
minfrag = find(cpos==1, 1 );
maxfrag = find(cpos==1, 1, 'last' );
for i=minfrag+1:maxfrag-1
    count = count + 1;
    if cpos(i)==1 % a fracture point
        fragl = [fragl count];
        count = 0;
    end
end
%if min(fragl)==1
%    disp(['stop'])
%end
fragall = [fragall fragl];
end
%
% Characteristic length (from Mott)
%
x0 = sqrt((2 .* Y)./(rho .* gamma)) .* (1./sr);
xscale = binspace./x0;
fragalls = fragall .* xscale;
end

