%========================================================================
%
% Modified from reference: Shaner et al. Energy Environ. Sci.,2018, 11, 914
%
% AIM: Calculate reliability with flexible input of supply side 
%
%      Default: Supply_sum = Demand_sum * GenRatio
%
%      1) Ratio of Supply_sum to Demand_sum (i.e., GenRatio) can be personalzied 
%
%      2) Wind/Solar share in total generation (i.e., sf_fract) can be personalzied
%
%      3) Storage (In units of time of mean demand (i.e, storage_size) can be personalzied
%
%========================================================================


function [Power_rlb_sum,Num_Wb,Num_PV,Battery_size,rlb_day] = Power_Storage_Liu(CF_Wb    ,CF_PV    ,Demand,...
                                                                        GenRatio ,sf_fract ,storage_days)

% % Input

% CF_Wb                % Time series of unit Wind power supply
% CF_PV                % Time series of unit PV power supply
% Demand               % Time series of power demand 
% GenRatio             % Ratio of total supply to total demand
% sf_fract             % Solar fraction of total generation 
% storage_days         % Units: days of mean demand; 

% % Output

% Power_rlb_sum        % Power  Reliability (% of total demand met by generated power) 
% Binary_rlb_sum       % Binary Reliability (% of total time steps met by generated power)
% Num_Wb               % Number of CF_PV installed capacity
% Num_PV               % Number of CF_PV installed capacity
% Battery_size         % Units: power; equal to storage_size * mean daily demand

Power_rlb_sum  = [];

Binary_rlb_sum = [];

% Length of wind&solar time series
days_length = length(CF_Wb);

% Battery size
Battery_size= mean(Demand)*storage_days; 

% Prescribe Wind/Solar generation ratio

Supply_sum  = sum(Demand)*GenRatio;

Num_Wb      = Supply_sum.*(1-sf_fract)./sum(CF_Wb); 

Num_PV      = Supply_sum.*sf_fract./sum(CF_PV);

CF_Wb_one   = Num_Wb.*CF_Wb;

CF_PV_one   = Num_PV.*CF_PV;

   % Supply - demand at each time step
    
   Supply  = CF_Wb_one + CF_PV_one;
    
   Power   = Supply - Demand;

     % Initialize arrays for collection reliabilities
     
     binary_rlb = zeros(1,days_length); % will collect wheather each time step can met demand or not
                                        % 0 - reliable; 1 - unreliable
                                     
     power_rlb  = zeros(1,days_length); % will collect amount of power unable to be delivered at each time step
                                        % 0 - Demand 100% met; 1 - unmet Demand size
     
     battery_state  = zeros(1,days_length+1);% initialize array; need + 1 because first entry is initial state    
     
          % Option 1 : Initial state of storage  is empty 
                %   battery_state(1,1) = 0;
          
          % Option 2 : Initial state of storage  is full
                %   battery_state(1,1) = Battery_size;
    
     % Loop through all time steps
     for t = 1 : days_length 
    
         battery_next_state = battery_state(1,t) + Power(t); % Starting point fo next battery state;
           % will be bounded by empty and full battery below

         % Storage Discharging, Demand > Generation
         if Power(t) < 0  
        
            if battery_next_state < 0  % if power demand cannot be met by battery
                  
                 binary_rlb(1,t) = 1; % record that hour as unable to meet demand
                 
                 power_rlb(1,t)  = abs(battery_next_state); % add amount of power that 
                 % cannot be delivered to running sum 
                 
            else % if battery could meet demand   
             
                 battery_state(1,t+1) = battery_next_state;% Next battery state equal the current state minus
                 % discharged amount
            end
         
         % Storage Charging, Demand < Generation    
         else
           
          battery_state(1,t+1) = min(battery_next_state,Battery_size); % either something less than full or full
        
         end
         
     end
     
     % holder for biniary reliability over all time (%)
     Binary_rlb_sum = [1 - sum(binary_rlb)./days_length].*100;

     % holder for power reliability over all time (%)
     Power_rlb_sum  = [1 - sum(power_rlb)./sum(Demand)].*100;
     
     % holder for daily power reliability
     rlb_day = 1 - power_rlb'./Demand;
 
     % If Demand is zero constant, reliability do not make sense and thus
         % be assigned as Nan value
     if sum(Demand) == 0
        Binary_rlb_sum = nan;
        Power_rlb_sum  = nan;
     end
     
end










