//ORIGINAL
// Produce Random Excursions in a 3D isosceles Wedge with angles (alpha, beta) and the associated CRT graph
#include <vector>
#include <iostream>
#include <time.h>
#include <queue>
#include <sstream>
#include <fstream>
#include <string>
#include <stack>
#include <algorithm>
#include <complex>
#include <iomanip>

#include "random.h"

#include "boost/program_options.hpp"

// specify pseudo-random number generator
typedef Xoshiro256PlusPlus RNG;

// to import numerical harmonic function
class HarmonicFunction {
public:
    virtual double operator()(double theta, double phi) = 0;
    virtual double operator()(double x, double y, double z) = 0;
};

// assuming startangle is uniform in [0,2pi) the following function returns an angle theta and an angle phi that is sampled from the distribution h(x-x0).

std::vector<double> sampleangle(HarmonicFunction * harmonic, double alpha, double beta, double epsilon, double xcenter, double ycenter, double zcenter, RNG & rng)
{
    double xincrement = 0.0;
    double yincrement = 0.0;
    double zincrement = 0.0;
    double N = 1.0;

    double maxvalue;
    
	// give a crude upper bound on the harmonic function

       maxvalue = N * (*harmonic)(xcenter, ycenter, zcenter);
    
	while(true){
		double xi = random_normal(rng, 0.0, 1.0); //choose in normal dist
        double yi = random_normal(rng, 0.0, 1.0);
        double zi = random_normal(rng, 0.0, 1.0);
        xincrement = epsilon * (xi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
        yincrement = epsilon * (yi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
        zincrement = epsilon * (zi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
                
        // Accept or Reject move
 
        double value = N * (*harmonic)(xincrement + xcenter, yincrement + ycenter, zincrement + zcenter);
 
    
        double rand = uniform_real(rng);
        if( value > rand * maxvalue && maxvalue > 0.0)
        {

            break;
        }
    }
    std::vector<double> increment_dist = {xcenter + xincrement, ycenter + yincrement,zcenter + zincrement};
    return increment_dist;
}

// Uniform sampling of a point of radius = radius
std::vector<double> randominsphere(double radius, RNG & rng)
{
    double xi = random_normal(rng, 0.0, 1.0); //choose in normal dist
    double yi = random_normal(rng, 0.0, 1.0);
    double zi = random_normal(rng, 0.0, 1.0);
    double xincrement = radius * (xi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
    double yincrement = radius * (yi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
    double zincrement = radius * (zi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
    std::vector<double> increment_rand =  {xincrement, yincrement, zincrement};
    return increment_rand;
}


//VECTOR OPERATIONS & Geometrical computations
//Norm
double norm( std::vector<double> v )
{
    int size = v.size();
    // double accum;
    double accum = 0.0;
    for (int i = 0; i < size; ++i) {
        accum += v[i] * v[i];
    }

    double norm_v = sqrt(accum);
    return norm_v;
}

// Sum
std::vector<double> sum( std::vector<double> u, std::vector<double> v )
{
    int size = v.size();
    std::vector<double> uplusv(size);
    for (int i = 0; i < size; ++i) {
        uplusv[i] = u[i] + v[i];
    }
    return uplusv;
}

//Subtraction
std::vector<double> rest( std::vector<double> u, std::vector<double> v )
{
    int size = v.size();
    std::vector<double> uminusv(size);
    for (int i = 0; i < size; ++i) {
        uminusv[i] = u[i] - v[i];
    }
    return uminusv;
}

//Map from wedge to octant 
void mapwalktooctant(double alpha, double beta, std::vector<std::vector<double> > & walk)
{
    // the linear mapping;p;-[ to the quadrant is given by the matrix {{ 1/cosb, -cota/cosb, 0 }, {0, 1/cosb*sina, 0}, {-tanb, -tanb(csca - cota), 1}}
    double cota = std::cos(alpha) / std::sin(alpha);
    double tanb = std::sin(beta) / std::cos(beta);
    for(int i=0,endi=walk.size();i<endi;++i)
    {
        
        walk[i] = {(walk[i][0] / (std::cos(beta))) - (walk[i][1] * (cota / std::cos(beta))), walk[i][1] / (std::cos(beta) * std::sin(alpha)), walk[i][2] - (walk[i][0] * tanb) + (walk[i][1] * (tanb*cota - tanb/std::sin(alpha)))};

    }
}


//Rotation
// this is the matrix that rotates \tilde{x_1} to x_1. It maps a vector u into v , i.e. R (u/normu) = (v/normv)
std::vector<double> Rotation( std::vector<double> u, std::vector<double> v )
{
    double normu = std::sqrt(std::pow(u[0],2) + std::pow(u[1],2) +  std::pow(u[2],2));
    double normv = std::sqrt(std::pow(v[0],2) + std::pow(v[1],2) +  std::pow(v[2],2));
    double normuxv = std::sqrt(std::pow(u[0]*v[1] - u[1]*v[0],2) + std::pow(u[2]*v[0] - u[0]*v[2],2) +  std::pow(u[1]*v[2] - u[2]*v[1],2));
    double udotv = u[0]*v[0] +  u[1]*v[1] +  u[2]*v[2];
    std::vector <double> R (9);
    R[0] = 1/(normu * normv) * (udotv + ((normu * normv - udotv)/std::pow(normuxv,2)) * std::pow(u[2]*v[1] - u[1]*v[2],2) );
    R[1] = 1/(normu * normv) * (u[1]*v[0] - u[0]*v[1] + ((normu * normv - udotv)/std::pow(normuxv,2)) * (- u[2]*v[1] + u[1]*v[2]) * (u[2]*v[0] - u[0]*v[2]));
    R[2] = 1/(normu * normv) * (u[2]*v[0] - u[0]*v[2] + ((normu * normv - udotv)/std::pow(normuxv,2)) * (- u[1]*v[0] + u[0]*v[1]) * (-u[2]*v[1] + u[1]*v[2]));
    R[3] = 1/(normu * normv) * (u[0]*v[1] - u[1]*v[0] + ((normu * normv - udotv)/std::pow(normuxv,2)) * (- u[2]*v[1] + u[1]*v[2]) * (u[2]*v[0] - u[0]*v[2]));
    R[4] = 1/(normu * normv) * (udotv + ((normu * normv - udotv)/std::pow(normuxv,2)) * std::pow(u[2]*v[0] - u[0]*v[2],2));
    R[5] = 1/(normu * normv) * (u[2]*v[1] - u[1]*v[2] + ((normu * normv - udotv)/std::pow(normuxv,2)) * (- u[1]*v[0] + u[0]*v[1]) * (u[2]*v[0] - u[0]*v[2]));
    R[6] = 1/(normu * normv) * (u[0]*v[2] - u[2]*v[0] + ((normu * normv - udotv)/std::pow(normuxv,2)) * (- u[1]*v[0] + u[0]*v[1]) * (u[1]*v[2] - u[2]*v[1]));
    R[7] = 1/(normu * normv) * (u[1]*v[2] - u[2]*v[1] + ((normu * normv - udotv)/std::pow(normuxv,2)) * (- u[1]*v[0] + u[0]*v[1]) * (u[2]*v[0] - u[0]*v[2]));
    R[8] = 1/(normu * normv) * (udotv + ((normu * normv - udotv)/std::pow(normuxv,2)) * std::pow(u[1]*v[0] - u[0]*v[1],2));
    return R;
}

//Distance to boundary 
double conedistance(double alpha, double beta, double x, double y, double z)
{
    double normplane = std::sqrt((std::pow(std::cos(beta),4) * std::pow(std::sin(alpha),2)) + (std::pow(std::sin(0.5 * alpha),2) * std::pow(std::sin(2 * beta),2)));
    double distance1 = y; //Distance to plane x
    double distance2 = (-(x * std::cos(beta) * std::sin(beta) * std::sin(alpha)) - (y * std::cos(beta) * std::sin(beta) * (1 - std::cos(alpha))) + (z * std::pow(std::cos(beta),2) * std::sin(alpha))) / normplane; //Distance to the plane at angle beta
    double distance3 = std::sqrt((std::pow(x,2) + std::pow(y,2)) * ( std::pow(std::cos(std::atan(y / x)) - std::cos(alpha),2) + std::pow(std::sin(std::atan(y / x)) - std::sin(alpha),2)  )); //Distance to the plane of angle alpha
    
    return std::min(std::min(distance1 , distance2) , distance3);
}

//Sample a point in the unit sphere to start the whole simulations
double boundarytheta(double alpha, double beta, double ph)
{
    double thetaboundary = std::atan((std::cos(beta)/std::sin(beta)) / (std::cos(ph) + (1- std::cos(alpha)) * (std::sin(ph) / std::sin(alpha))));
    
    return thetaboundary;
}


//////////////// GENERATING WALK //////////////////

bool generateWalk( double alpha, double beta, HarmonicFunction * harmonic, double xstart, double ystart, double zstart, double stepsize, int min, int max, RNG & rng, std::vector<std::vector<double> > & walk )
{
    //minimal distance to the origin to reach
    double target = 2 * stepsize;
    int pivotindex = 0;
    //this one -> FIRST ENTRY OF ARRAY
    std::vector<double> pivotpoint = {xstart, ystart, zstart};
    
    //biggest ball contained in the wedge
    double spheresize = conedistance(alpha, beta, xstart, ystart, zstart);// - stepsize/4;
    walk.push_back(pivotpoint);
    
    while( norm(walk.back()) > target )
    {
        //start a uniform random walk from 'pivotpoint'
        walk.push_back( sum( walk.back() , randominsphere(stepsize,rng) ));
       
       // measure how far from 'pivotpoint' the random walk is
        std::vector<double> diff = rest( walk.back() , pivotpoint );
        double distance = norm( diff );

        if( distance >= spheresize )
        {
        	//record until walk reaches boundary of ball around 'pivotpoint'
            walk.back()[0] = pivotpoint[0] + (spheresize/distance) * (diff[0]);
            walk.back()[1] = pivotpoint[1] + (spheresize/distance) * (diff[1]);
            walk.back()[2] = pivotpoint[2] + (spheresize/distance) * (diff[2]);
            
            //useful definition
            double xcenter = pivotpoint[0];
            double ycenter = pivotpoint[1];
            double zcenter = pivotpoint[2];

           //sample point in sphere of radius 'spheresize' center at 'pivotpoint' contained in the wedge 
            std::vector<double> increment = sampleangle(harmonic, alpha, beta, spheresize,xcenter,ycenter,zcenter,rng);
            
            //Rotate the step using the transformation 'Rotation' that sends walk.back-pivotpoint to increment-pivotpoint
            // i.e. R[positionbefore -> positionafter] (walk - pivotpoint) = rotated walk
            // This map mantains the norm!
            
               //useful names
               std::vector<double> positionbefore = rest( walk.back() , pivotpoint );
  
                std::vector<double> positionafter = rest( increment , pivotpoint );

                //Apply ROTATION MATRIX
                std::vector<double> R = Rotation(positionbefore , positionafter);
         
                for(int i = pivotindex + 1, endi = walk.size();i<endi;++i)
                {
                	//record in vector 'walk'
                    walk[i] = {pivotpoint[0] + R[0] * (walk[i][0] - pivotpoint[0]) + R[1] * (walk[i][1] - pivotpoint[1]) + R[2] * (walk[i][2] - pivotpoint[2]), pivotpoint[1] + R[3] * (walk[i][0] - pivotpoint[0]) + R[4] * (walk[i][1] - pivotpoint[1]) + R[5] * (walk[i][2] - pivotpoint[2]), pivotpoint[2] + R[6] * (walk[i][0] - pivotpoint[0]) + R[7] * (walk[i][1] - pivotpoint[1]) + R[8] * (walk[i][2] - pivotpoint[2]) };
                }
            
            //relabel to repeat iteration 
            pivotindex = walk.size()-1;
            pivotpoint = walk.back();
            double xnew = walk.back()[0];
            double ynew = walk.back()[1];
            double znew = walk.back()[2];
            spheresize = conedistance(alpha, beta, xnew, ynew, znew);
            //}
        }
        
        if( static_cast<int>(walk.size()) >= max )
        {
            return false;
        }
    }
       
    return static_cast<int>(walk.size()) >= min;
    
}




//////////////////////////////////// CRT MATING ////////////////////////////////

// divide the walk (x-coordinates, y-coordinates and z)
// into "vertices" segments of approximately equal size, and determine the
// non-trivial adjacency between these segments

void determineadjacency(std::vector<std::vector<double> > & walk, int k, int vertices, std::vector<std::vector<int> > & adjZ)
{
    std::stack<std::pair<int,double> > infimums;
    
    // starting point of first bin
    int binstart = 0;
    for(int i=0;i<vertices;++i)
    {
        // determine endpoint of the i'th bin such that it contains a fraction 1/vertices of steps
        int binend = (i+1)*walk.size() / vertices;
        
        // determine the minimum in the i'th bin
        double min = 10000.0;
        for(int j = binstart; j < binend; ++j)
        {
            min = walk[j][k];
 
        }
        
        // compare to the previous minima
        while( !infimums.empty() && infimums.top().second > min )
        {
            infimums.pop();
            if( !infimums.empty() )
            {
                adjZ[i].push_back(infimums.top().first);
                adjZ[infimums.top().first].push_back(i);
            }
        }
        infimums.push( std::pair<int,double>(i,min) );
        
        binstart = binend;
    }
}

////////////////////////////// NEIGHBORNS  in the entire graph/////////////

void determineadjacency(std::vector<std::vector<double> > & walk, int vertices, std::vector<std::vector<int> > & adj)
{
    adj.resize(vertices);
    for(int k=0;k<3;++k)
    {
        // add left neighbour
        for(int i=0;i<vertices;++i)
        {
            adj[i].push_back((i+vertices-1)%vertices);
        }
        // add neighbours from first walk
        determineadjacency(walk,k,vertices,adj);
        
        // add right neighbour
        for(int i=0;i<vertices;++i)
        {
            adj[i].push_back((i+1)%vertices);
        }
        
    }

}

////////////////////////////// DISTANCES ////////////////////////////

// use the adjacency information to determine all graph distances to the vertex "start"
// and insert these distances into the histogram
void determinedistances(std::vector<std::vector<int> > & adj, int start, std::vector<uint64_t> & histogram)
{
    std::queue<int> queue;
    // set the distance of all unexplored vertices to -1
    std::vector<int> distance(adj.size(),-1);
    distance[start] = 0;
    histogram[0]++;
    queue.push(start);
    
    while( !queue.empty() )
    {
        int current = queue.front();
        queue.pop();
        int nextdist = distance[current]+1;
        
        // examine neighbours
        for(int i=0,endi=adj[current].size();i<endi;++i)
        {
            int nbr = adj[current][i];
            if( distance[ nbr ] == -1 )
            {
                // unexplored neighbour
                distance[ nbr ] = nextdist;
                queue.push( nbr );
                
                // if distance does not exceed maximum, record it
                if( nextdist < static_cast<int>(histogram.size()) )
                {
                    histogram[nextdist]++;
                }
            }
        }
    }
}

/////////////////////// SAMPLE ANGLE IN UNIT SPHERE TO START ///////////////////////
//  SAMPLING A POINT IN THE UNIT SPHERE according ro Harmonic 
// here i need to choose random normal variables
std::vector<double> RandomAngle(double alpha, double beta, HarmonicFunction * harmonic, RNG & rng)
{
    std::vector<double> angle;
    double phi, theta, uniform, prob, xsample, ysample, zsample;
    //double N = (std::pow(1 / std::sqrt(2), 1+M_PI/alpha)) / (*harmonic)((0.5 * M_PI - beta)/2, 0.5 * alpha );
    double N = 1.0;
    do{
        double xi = random_normal(rng, 0.0, 1.0); //choose in normal dist
        double yi = random_normal(rng, 0.0, 1.0);
        double zi = random_normal(rng, 0.0, 1.0);
        xsample = (xi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
        ysample = (yi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
        zsample = (zi / (std::sqrt(std::pow(xi,2) + std::pow(yi,2) +  std::pow(zi,2))));
        phi = ( std::atan2(ysample, xsample)) ;
        //phi = std::atan(ysample / xsample);
        theta = std::acos(zsample);
        uniform = uniform_real(rng);
        prob = N * (*harmonic)(theta,phi);

    } while( (prob * prob > uniform) || (prob <= 0.0) || (0.0 >= phi) || (phi >= alpha) || (((M_PI / 2) - beta) <= theta) || (theta <= 0.0) );
    angle = {phi, theta};
    return angle;
}

////////////////////////// HARMONIC FUNCTION ////////////////////////////////

// Here I can put the exact harmonic function, if available
class SimpleHarmonicFunction : public HarmonicFunction {
public:
    SimpleHarmonicFunction(double alpha) : alpha_(alpha) {}
    double operator()(double theta,double phi)
    {
        // to do
        return 1.0;
    }
    double operator()(double x, double y, double z)
    {
        // to do
        return 1.0;
    }
private:
    double alpha_;
};

//IMPORTED HAMORNIC FUNCTION ////// Note: Be careful with the format in which the data is. The corresponding Mathematica notebook should give the data in correct order

class ImportedFunction : public HarmonicFunction {
public:
    ImportedFunction() {}
    bool import(std::string filename) {
        std::ifstream file(filename);
        
        if( !file )
        {
            // cannot open file
            return false;
        }
        //file data order
        file >> thetasize_ >> phisize_ >> thetamax_ >> phimax_ >> exponent_;
        //size of bins 
        thetabin_ = thetamax_ / (thetasize_ - 1);

        phibin_ = phimax_ / (phisize_ - 1);

        
        int numvalues = thetasize_*phisize_;
        
        
        values_.resize(numvalues);

        
        for(int i=0;i<numvalues;++i)
        {
            file >> values_[i];
        }
        return true;
    }
    
    double value(int i,int j)
    {
        return values_[i * phisize_ + j];
    }
    
    double operator()(double theta,double phi)
    {
   
        if( theta <= 0.0 || theta >= thetamax_ || phi <= 0.0 || phi >= phimax_ )
        {
            return 0.0;
        }
                       
        int thetabin = theta / thetabin_;
        int phibin = phi / phibin_;

        double x =  (theta/thetabin_) - thetabin;
        double y = (phi/phibin_) - phibin;
        

        //Near the boundary in the (\theta, \phi) plane, interpolation can fail due to numerical artifacts. with this part of the code we make sure the harmonic function vanishes in the boundary
        
        //Boundary in ThetaMax
        if( value(thetabin+1,phibin  ) <= 0.0 && value(thetabin+1,phibin+1) <= 0.0)
        {
            
        double xp = x;
       
        double val = 0.0;
        //std::cout << val << "\n";
            val += ((xp-x)/xp)*(1-y) * value(thetabin,  phibin  );
            //std::cout << value(thetabin,  phibin  ) << " " << val <<  "\n";
            val +=    (x/xp) *(1-y) * value(thetabin+1,phibin  );
            //std::cout << value(thetabin+1,phibin  ) << " " << val <<  "\n";
            val += ((xp-x)/xp)*   y  * value(thetabin,  phibin+1);
            //std::cout << value(thetabin,  phibin+1) << " " << val <<  "\n";
            val +=    (x/xp) *   y  * value(thetabin+1,phibin+1);
            //std::cout << value(thetabin+1,phibin+1) << " " << val <<  "\n";
            return val;
        }
        
        //Bdy in PhiMax
        if(value(thetabin,  phibin+1) <= 0.0 && value(thetabin+1,  phibin+1) <= 0.0)
        {

            
            double yp = y;
            //std::cout << "yp" << " " << yp << "\n";
            double val = 0.0;
            val += (1-x)*((yp-y)/yp) * value(thetabin,  phibin  );
            //std::cout << value(thetabin,  phibin  ) << " " << val <<  "\n";
            val +=    x *((yp-y)/yp) * value(thetabin+1,phibin  );
            //std::cout << value(thetabin+1,phibin  ) << " " << val <<  "\n";
            val += (1-x)*  (y/yp)  * value(thetabin,  phibin+1);
            //std::cout << value(thetabin,  phibin+1) << " " << val <<  "\n";
            val +=    x * (y/yp)  * value(thetabin+1,phibin+1);
            //std::cout << value(thetabin+1,phibin+1) << " " << val <<  "\n";
            return val;
            
        }
        
        //Bdy in PhiMin = 0
        if(value(thetabin,  phibin  ) <= 0.0 && value(thetabin+1,phibin  ) <= 0.0)
        {
            //std::cout << "close to bdy " << "\n";
            
            double yp = y;
            //std::cout << "yp" << " " << yp << "\n";
            double val = 0.0;
            val += (1-x)*((1-y)/yp) * value(thetabin,  phibin  );
            //std::cout << value(thetabin,  phibin  ) << " " << val <<  "\n";
            val +=    x *((1-y)/yp) * value(thetabin+1,phibin  );
            //std::cout << value(thetabin+1,phibin  ) << " " << val <<  "\n";
            val += (1-x)*  (y-yp/yp)  * value(thetabin,  phibin+1);
            //std::cout << value(thetabin,  phibin+1) << " " << val <<  "\n";
            val +=    x * (y-yp/yp)  * value(thetabin+1,phibin+1);
            //std::cout << value(thetabin+1,phibin+1) << " " << val <<  "\n";
            return val;
        }
        
        //Rest of the bulk: No interpolations correction necessary
        else{
            // use bilinear interpolation
            double val = 0.0;
            //std::cout << val << "\n";
            val += (1-x)*(1-y) * value(thetabin,  phibin  );
            //std::cout << value(thetabin,  phibin  ) << " " << val <<  "\n";
            val +=    x *(1-y) * value(thetabin+1,phibin  );
            //std::cout << value(thetabin+1,phibin  ) << " " << val <<  "\n";
            val += (1-x)*   y  * value(thetabin,  phibin+1);
            //std::cout << value(thetabin,  phibin+1) << " " << val <<  "\n";
            val +=    x *   y  * value(thetabin+1,phibin+1);
            //std::cout << value(thetabin+1,phibin+1) << " " << val <<  "\n";
            return val;
        }
    }
 
    double operator()(double x, double y, double z)
    {
        double r = std::sqrt((x*x)+(y*y)+(z*z));
        
        double theta = std::acos(z / r);
        double phi = std::atan2(y,x);
        
        double val = operator()(theta,phi) * std::pow( r, exponent_);
        
        return val;
    }
private:
    double thetamax_;
    double phimax_;
    double exponent_;
    int thetasize_;
    int phisize_;
    double thetabin_;
    double phibin_;
    std::vector<double> values_;
};

//////////////////////////////////////////////////////////////GENERATIN ALL
int main(int argc, char **argv)
{
    // initialize random number generator
    Xoshiro256PlusPlus rng(getseed());
    
    // parameters
    double alpha;
    double beta;
    double stepsize;
    double xstart;
    double ystart;
    double zstart;
    bool excursion;
    int max;
    int min;
    int numvertices;
    int maxdist;
    int samples, persample;
    std::string harmonicfilename;

    // get command line arguments (this is the only place where BOOST is used)
    namespace po = boost::program_options;
    po::options_description desc("Allowed options");
    desc.add_options()
        ("help", "produce help message")
        ("alpha,a", po::value<double>(&alpha)->default_value(M_PI/2), "the opening angle of the cone from the XZ-plane (between 0 and pi)")
        ("beta,b", po::value<double>(&beta)->default_value(0.0), "the opening angle of the cone from the XY-plane (between 0 and pi/2)")
        ("stepsize,s", po::value<double>(&stepsize)->default_value(0.01), "stepsize")
        ("xstart,x", po::value<double>(&xstart)->default_value(1.0), "x-coordinate of starting point")
        ("ystart,y", po::value<double>(&ystart)->default_value(1.0), "y-coordinate of starting point")
        ("zstart,z", po::value<double>(&zstart)->default_value(1.0), "z-coordinate of starting point")
        ("excursion,e", "perform excursion from origin" )
        ("max,m", po::value<int>(&max)->default_value(1000000000), "maximum number of steps")
        ("min,u", po::value<int>(&min)->default_value(0), "minimum number of steps")
        ("numvertices,n", po::value<int>(&numvertices)->default_value(0), "number of vertices")
        ("maxdist,d", po::value<int>(&maxdist)->default_value(0), "maximal distance for histogram")
        ("samples,r", po::value<int>(&samples)->default_value(1), "number of samples")
        ("persample,p", po::value<int>(&persample)->default_value(1), "measurements per sample")
        ("import,i", po::value<std::string>(&harmonicfilename), "filename for harmonic function")
    ;
    po::variables_map vm;
    po::store(po::parse_command_line(argc,argv,desc),vm);
    po::notify(vm);
    if( vm.count("help") ) {
        std::cout << desc << "\n";
        return 1;
    }

////// check input parameters ////////

///ANGLES
   if( alpha <= 0.0 || alpha >= M_PI || beta < 0.0 || beta >= (M_PI / 2) )
   {
       std::cout << "Angles not in range!\n";
       return 1;
   }

   excursion = vm.count("excursion");
    
//// HARMONIC FUNCTION
   HarmonicFunction * harmonic;
   

   if( vm.count("import") )
   {
       ImportedFunction * importharm = new ImportedFunction;
       
       if( !importharm->import(harmonicfilename) )
       {
           std::cout << "Error reading harmonic function\n";
           return 1;
       }
       
       harmonic = importharm;
   }

    
    if( (*harmonic)(0.5 * ((M_PI / 2) - beta), 0.5 * alpha) == 0.0 && (*harmonic)(0.3 * ((M_PI / 2) - beta), 0.5 * alpha) == 0.0 )
    {
        std::cout << "Harmonic = 0!\n";
        return 1;
    }
    
//// STARTING POINT
    if( !excursion && (std::atan(ystart / xstart) <= 0.0 || std::atan(ystart / xstart) >= alpha || std::acos(zstart / (std::sqrt(xstart*xstart+ystart*ystart+zstart*zstart))) <= 0.0 || std::acos(zstart / (std::sqrt(xstart*xstart+ystart*ystart+zstart*zstart))) >= ((M_PI / 2) - beta) || conedistance(alpha, beta, xstart, ystart, zstart) < stepsize ) )
   {
       std::cout << "Invalid starting point: " << xstart << " " << ystart << " " << zstart << " " << "distance =" << " " << conedistance(alpha, beta, xstart, ystart, zstart) << "\n";
       return 1;
   }


    /// Histogram
    
   std::vector<uint64_t> histogram(maxdist+1,0);
    
   for(int n=0;n<samples;++n)
   {
    //Generate excursion
       if( excursion )
       {
        // start at distance one and random angular coordinate,
           // making sure that we are not too close to the boundary
           double theta, phi;
           do{
               std::vector<double> phitheta = RandomAngle(alpha,beta,harmonic,rng);
               phi = phitheta[0];
               theta = phitheta[1];
               xstart = std::cos(phi) * std::sin(theta);
               ystart = std::sin(phi) * std::sin(theta);
               zstart = std::cos(theta);
           } while( conedistance(alpha,beta,xstart, ystart, zstart) < stepsize );
       }

    // generate first walk from unit sphere to origin
       std::vector<std::vector<double> > walk;
       while( !generateWalk(alpha,beta,harmonic,xstart,ystart,zstart,stepsize,min,max,rng,walk) )
       {
        // generation did not succeed within bounds (min,max)
           walk.clear();
       }
    
       if( excursion )
       {
    //generate second walk from unit sphere to origin
           std::vector<std::vector<double> > walk2;
           
           while( !generateWalk(alpha,beta,harmonic,xstart,ystart,zstart,stepsize,min,max,rng,walk2) )
           {
               walk2.clear();
               
           }
       // reverse the first walk
           std::reverse( walk.begin(), walk.end() );
       // delete the last point (which is start)
           walk.pop_back();
       // and append the second
           walk.insert( walk.end(), walk2.begin(), walk2.end() );
           
       }

    // perform appropriate linear transformation to map the uncorrelated BE in the wedge to a correlated BE in the octant
       mapwalktooctant(alpha,beta,walk);
       
       if( numvertices > 0 )
               {
                   std::vector<std::vector<int> > adj;
                   determineadjacency(walk,numvertices,adj);
                   
                   if( maxdist > 0 )
                   {
                       // measure distances several times and populate histogram
                       for(int k=0;k<persample;++k)
                       {
                           // select a random vertex as starting point
                           int start = uniform_int(rng,numvertices);
                           determinedistances(adj,start,histogram);
                       }
                   } else
                   {
                       // output adjacency information
                       for(int i=0;i<numvertices;++i)
                       {
                           std::cout << i << ": ";
                           for(int j=0,endj=adj[i].size();j<endj;++j)
                           {
                               std::cout << (j>0?" ":"") << adj[i][j];
                           }
                           std::cout << "\n";
                       }
                   }
               } else {
                   // output walk
                   for(int i=0,endi=walk.size();i<endi;++i)
                   {
                       std::cout << std::fixed << walk[i][0] << " " << walk[i][1] << " " << walk[i][2] << "\n";
                   }
               }
           }
           
           if( maxdist > 0 )
           {
               // output histogram
               for(int i=0;i<=maxdist;++i)
               {
                   std::cout << histogram[i] << "\n";
               }
           }
           
           return 0;
       }

