// Produce Random Excursions in a 2D wedge of opening angle (alpha) and the associated CRT map
#include <vector>
#include <iostream>
#include <time.h>
#include <queue>
#include <sstream>
#include <fstream>
#include <string>
#include <stack>
#include <algorithm>
#include <complex>
#include <iomanip>

#include "random.h"

#include "boost/program_options.hpp"

// specify pseudo-random number generator
typedef Xoshiro256PlusPlus RNG;

// value of the harmonic function -Im(z^{-\pi / \alpha})
double harmonicfunction(double alpha, std::complex<double> z)
{
	double exponent = - M_PI / alpha;
	return - std::imag(std::pow(z,exponent));
}

// return e^{i theta}
std::complex<double> expi(double theta)
{
	return std::complex<double>( std::cos(theta), std::sin(theta) ); 
}

// assuming startangle is uniform in [0,2pi) the following function returns an angle theta that is sampled
// from the distribution h(alpha, z + r e^{i (startangle +theta)}) dtheta/(2pi *h(alpha,z)).
double sampleangle(double alpha, double radius, std::complex<double> center, double startangle, RNG & rng)
{
	double exponent = -M_PI / alpha;
	double angleincrement = 0.0;

	// give a crude upper bound on the harmonic function
	double maxvalue = std::pow(std::abs(center) - radius, exponent );
	
	while(true){
		double value = harmonicfunction(alpha, center + radius * expi( startangle + angleincrement ) );
		if( value > uniform_real(rng) * maxvalue )
		{
			break;
		}
		angleincrement = uniform_real(rng, 0.0, 2 * M_PI );
	}
	return angleincrement;
}

// a uniform sampling of a complex number of modulus = radius
std::complex<double> randomincircle(double radius, RNG & rng)
{
	return radius * expi(uniform_real(rng,0,2*M_PI) );
}

// distance of z to the boundary of a cone of opening angle alpha
double conedistance(double alpha, std::complex<double> z)
{
	return std::min( std::imag(z), - std::imag( expi(-alpha) * z ) );
}

// argument of the complex number z
double argument(std::complex<double> z)
{
	return std::atan2( std::imag(z), std::real(z) );
}

// Stores in "walk" a random path in the complex plane from "start" to the disk of radius "target" around the
// origin while remaining in the cone of opening angle "alpha". The steps have length "stepsize" (and random
// orientation). The function returns true if the number of steps is between "min" and "max".
bool generateWalk( double alpha, std::complex<double> start, double stepsize, int min, int max, RNG & rng, std::vector<std::complex<double> > & walk)
{
        double target = 2 * stepsize;
	int pivotindex = 0;
	std::complex<double> pivotpoint = start;
	double disksize = conedistance(alpha, start) - stepsize;
	walk.push_back(start);
	
	while( std::abs(walk.back()) > target )
	{
		walk.push_back( walk.back() + randomincircle(stepsize,rng) );
		
		std::complex<double> diff = walk.back() - pivotpoint;
		double distance = std::abs( diff );
		if( distance >= disksize )
		{
			double angle = argument(diff);
			double angleincrement = sampleangle(alpha,distance,pivotpoint,angle,rng);
			
			if( std::abs(angleincrement) > 0.00001 )
			{
				std::complex<double> factor = expi( angleincrement );
				for(int i = pivotindex + 1, endi = walk.size();i<endi;++i)
				{
					walk[i] = pivotpoint + factor*(walk[i] - pivotpoint);
				} 
			}
			
			pivotindex = walk.size()-1;
			pivotpoint = walk.back();
			disksize = conedistance(alpha, walk.back()) - stepsize;
		}
		
		if( static_cast<int>(walk.size()) >= max )
		{
			return false;
		}
	}
	
	return static_cast<int>(walk.size()) >= min;
}

// perform linear mapping on the walk from the cone of opening angle "alpha" to 
// the quarter plane (sending bisector of the cone to the diagonal)
void mapwalktoquadrant(double alpha, std::vector<std::complex<double> > & walk)
{
	double sin = std::sin(alpha);
	double cos = std::cos(alpha);
	double cot = cos / sin;
	
	// the linear mapping to the quadrant is given by the matrix {{ 1, -cot }, {0, 1/sin}}	
	for(int i=0,endi=walk.size();i<endi;++i)
	{
		walk[i].real(walk[i].real() - cot * walk[i].imag());
		walk[i].imag(walk[i].imag() / sin);
	}
}

// divide the walk (x-coordinates if realpart == true, and y-coordinates otherwise) 
// into "vertices" segments of approximately equal size, and determine the
// non-trivial adjacency between these segments
void determineadjacency(std::vector<std::complex<double> > & walk, int vertices, bool realpart, std::vector<std::vector<int> > & adj)
{
	std::stack<std::pair<int,double> > infimums;
	
	// starting point of first bin
	int binstart = 0;
	for(int i=0;i<vertices;++i)
	{
		// determine endpoint of the i'th bin such that it contains a fraction 1/vertices of steps
		int binend = (i+1)*walk.size() / vertices;
		
		// determine the minimum in the i'th bin
		double min = 10000.0;
		for(int j = binstart; j < binend; ++j)
		{
			if( realpart && walk[j].real() < min )
			{
				min = walk[j].real();
			} else if( !realpart && walk[j].imag() < min )
			{
				min = walk[j].imag();
			}
		}
		
		// compare to the previous minima
		while( !infimums.empty() && infimums.top().second > min )
		{
			infimums.pop();
			if( !infimums.empty() )
			{
				adj[i].push_back(infimums.top().first);
				adj[infimums.top().first].push_back(i);
			}
		}
		infimums.push( std::pair<int,double>(i,min) );
		
		binstart = binend;
	}
}

void determineadjacency(std::vector<std::complex<double> > & walk, int vertices, std::vector<std::vector<int> > & adj)
{
	adj.resize(vertices);
	
	// add left neighbour
	for(int i=0;i<vertices;++i)
	{
		adj[i].push_back((i+vertices-1)%vertices); 
	}
	// add neighbours from first walk
	determineadjacency(walk,vertices,true,adj);
	// add right neighbour
	for(int i=0;i<vertices;++i)
	{
		adj[i].push_back((i+1)%vertices); 
	}
	// add neighbours from second walk	
	determineadjacency(walk,vertices,false,adj);
}

// use the adjacency information to determine all graph distances to the vertex "start"
// and insert these distances into the histogram
void determinedistances(std::vector<std::vector<int> > & adj, int start, std::vector<uint64_t> & histogram)
{
	std::queue<int> queue;
	// set the distance of all unexplored vertices to -1
	std::vector<int> distance(adj.size(),-1);
	distance[start] = 0;
	histogram[0]++;
	queue.push(start);
	
	while( !queue.empty() )
	{
		int current = queue.front();
		queue.pop();
		int nextdist = distance[current]+1;
		
		// examine neighbours
		for(int i=0,endi=adj[current].size();i<endi;++i)
		{
			int nbr = adj[current][i];
			if( distance[ nbr ] == -1 )
			{
				// unexplored neighbour
				distance[ nbr ] = nextdist;
				queue.push( nbr );
				
				// if distance does not exceed maximum, record it
				if( nextdist < static_cast<int>(histogram.size()) )
				{
					histogram[nextdist]++;
				}
			}
		}
	}
}

// sample random real number in (0,alpha) with density (2/alpha)*sin(theta*pi/alpha)^2
double randomSineSquared(double alpha, RNG & rng)
{
	double theta, uniform, sine;
	do{
		theta = uniform_real(rng,0.0,alpha);
		uniform = uniform_real(rng);
		sine = std::sin( theta * M_PI / alpha );
	} while( sine * sine > uniform );
	return theta;
}

int main(int argc, char **argv)
{
	// initialize random number generator
	Xoshiro256PlusPlus rng(getseed());
    
	// parameters
	double alpha;
	double stepsize;
	std::complex<double> start;
        double startx, starty;
	bool excursion;
	int max;
	int min;
	int numvertices;
	int maxdist;
	int samples, persample;

	// get command line arguments (this is the only place where BOOST is used)
	namespace po = boost::program_options;
	po::options_description desc("Allowed options");
	desc.add_options()
		("help", "produce help message")
		("alpha,a", po::value<double>(&alpha)->default_value(M_PI/2), "the opening angle of the cone (between 0 and pi)")
		("stepsize,s", po::value<double>(&stepsize)->default_value(0.01), "stepsize")
		("startx,x", po::value<double>(&startx)->default_value(1.0), "x-coordinate of starting point")
		("starty,y", po::value<double>(&starty)->default_value(1.0), "y-coordinate of starting point")	
		("excursion,e", "perform excursion from origin" )
		("max,m", po::value<int>(&max)->default_value(1000000000), "maximum number of steps")
		("min,u", po::value<int>(&min)->default_value(0), "minimum number of steps")
		("numvertices,n", po::value<int>(&numvertices)->default_value(0), "number of vertices")	
		("maxdist,d", po::value<int>(&maxdist)->default_value(0), "maximal distance for histogram")
		("samples,r", po::value<int>(&samples)->default_value(1), "number of samples")
		("persample,p", po::value<int>(&persample)->default_value(1), "measurements per sample")
	;	
	po::variables_map vm;
	po::store(po::parse_command_line(argc,argv,desc),vm);
	po::notify(vm);
	if( vm.count("help") ) {
		std::cout << desc << "\n";
		return 1;
	}
	
	
	////// check input parameters ////////
	
	start.real(startx);
    start.imag(starty);
	
	if( alpha <= 0.0 || alpha >= M_PI )
	{
		std::cout << "Alpha not in range!\n";
		return 1;
	}
	
	excursion = vm.count("excursion");
	
	if( !excursion && (argument(start) <= 0.0 || argument(start) >= alpha || conedistance(alpha,start) < stepsize ) )
	{
		std::cout << "Invalid starting point: arg = " << argument(start) << ", distance = " << conedistance(alpha,start) << "\n";
		return 1;
	}
	
	//////////////////////////////////////
	
	
	// histogram that will hold the distance measurements
	std::vector<uint64_t> histogram(maxdist+1,0);
	
	for(int n=0;n<samples;++n)
	{
        if( excursion )
        {
            // start at distance one and random angular coordinate,
            // making sure that we are not too close to the boundary
            double theta;
            do{
                theta = randomSineSquared(alpha,rng);
                start = expi(theta);
            } while( conedistance(alpha,start) < stepsize );
        }
        
		// generate a walk
		std::vector<std::complex<double> > walk;
		while( !generateWalk(alpha,start,stepsize,min,max,rng,walk) )
		{
			// generation did not succeed within bounds (min,max)
			walk.clear();
		}
		
		if( excursion )
		{
			// in case of an excursion generate a second walk with same starting point
			std::vector<std::complex<double> > walk2;
			while( !generateWalk(alpha,start,stepsize,min,max,rng,walk2) )
			{
				walk2.clear();
			}
			// reverse the first walk
			std::reverse( walk.begin(), walk.end() );
			// delete the last point (which is start)
			walk.pop_back();
			// and append the second
			walk.insert( walk.end(), walk2.begin(), walk2.end() );
		}
		
		// perform appropriate linear transformation
		mapwalktoquadrant(alpha,walk);
		
		if( numvertices > 0 )
		{
			std::vector<std::vector<int> > adj;
			determineadjacency(walk,numvertices,adj);
			
			if( maxdist > 0 )
			{
				// measure distances several times and populate histogram
				for(int k=0;k<persample;++k)
				{
					// select a random vertex as starting point
					int start = uniform_int(rng,numvertices);
					determinedistances(adj,start,histogram);
				}
			} else
			{
				// output adjacency information
				for(int i=0;i<numvertices;++i)
				{
					std::cout << i << ": ";
					for(int j=0,endj=adj[i].size();j<endj;++j)
					{
						std::cout << (j>0?" ":"") << adj[i][j];
					}
					std::cout << "\n";
				}
			}
		} else {
			// output walk (in the quadrant)
			for(int i=0,endi=walk.size();i<endi;++i)
			{
				std::cout << std::fixed << std::real(walk[i]) << " " << std::imag(walk[i]) << "\n";
			}
		}
	}
	
	if( maxdist > 0 )
	{
		// output histogram
		for(int i=0;i<=maxdist;++i)
		{
			std::cout << histogram[i] << "\n";
		}
	}
	
    return 0;
}
