char *template_data[] = {
"!**********************************************************************************************************************************",
"!> ## ModuleName",
"!! The ModuleName and ModuleName_Types modules make up a template for creating user-defined calculations in the FAST Modularization",
"!! Framework. ModuleName_Types will be auto-generated by the FAST registry program, based on the variables specified in the",
"!! ModuleName_Registry.txt file.",
"!!",
"! ..................................................................................................................................",
"!! ## LICENSING",
"!! Copyright (C) 2012-2013, 2015-2016  National Renewable Energy Laboratory",
"!!",
"!!    This file is part of ModuleName.",
"!!",
"!! Licensed under the Apache License, Version 2.0 (the \"License\");",
"!! you may not use this file except in compliance with the License.",
"!! You may obtain a copy of the License at",
"!!",
"!!     http://www.apache.org/licenses/LICENSE-2.0",
"!!",
"!! Unless required by applicable law or agreed to in writing, software",
"!! distributed under the License is distributed on an \"AS IS\" BASIS,",
"!! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.",
"!! See the License for the specific language governing permissions and",
"!! limitations under the License.",
"!**********************************************************************************************************************************",
"MODULE ModuleName",
"",
"   USE ModuleName_Types",
"   USE NWTC_Library",
"",
"   IMPLICIT NONE",
"",
"   PRIVATE",
"",
"   TYPE(ProgDesc), PARAMETER  :: ModName_Ver = ProgDesc( 'ModuleName', '', '' ) !< module date/version information",
"",
"",
"   ! ..... Public Subroutines ...................................................................................................",
"",
"   PUBLIC :: ModName_Init                           !  Initialization routine",
"   PUBLIC :: ModName_End                            !  Ending routine (includes clean up)",
"",
"   PUBLIC :: ModName_UpdateStates                   !  Loose coupling routine for solving for constraint states, integrating",
"                                                    !    continuous states, and updating discrete states",
"   PUBLIC :: ModName_CalcOutput                     !  Routine for computing outputs",
"",
"   PUBLIC :: ModName_CalcConstrStateResidual        !  Tight coupling routine for returning the constraint state residual",
"   PUBLIC :: ModName_CalcContStateDeriv             !  Tight coupling routine for computing derivatives of continuous states",
"   PUBLIC :: ModName_UpdateDiscState                !  Tight coupling routine for updating discrete states",
"",
"   PUBLIC :: ModName_JacobianPInput                 ! Routine to compute the Jacobians of the output(Y), continuous - (X), discrete -",
"                                                    !   (Xd), and constraint - state(Z) functions all with respect to the inputs(u)",
"   PUBLIC :: ModName_JacobianPContState             ! Routine to compute the Jacobians of the output(Y), continuous - (X), discrete -",
"                                                    !   (Xd), and constraint - state(Z) functions all with respect to the continuous",
"                                                    !   states(x)",
"   PUBLIC :: ModName_JacobianPDiscState             ! Routine to compute the Jacobians of the output(Y), continuous - (X), discrete -",
"                                                    !   (Xd), and constraint - state(Z) functions all with respect to the discrete",
"                                                    !   states(xd)",
"   PUBLIC :: ModName_JacobianPConstrState           ! Routine to compute the Jacobians of the output(Y), continuous - (X), discrete -",
"                                                    !   (Xd), and constraint - state(Z) functions all with respect to the constraint",
"                                                    !   states(z)",
"   PUBLIC :: ModName_GetOP                          ! Routine to get the operating-point values for linearization (from data structures to arrays)",
"",
"CONTAINS",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> This routine is called at the start of the simulation to perform initialization steps.",
"!! The parameters are set here and not changed during the simulation.",
"!! The initial states and initial guess for the input are defined.   ",
"SUBROUTINE ModName_Init( InitInp, u, p, x, xd, z, OtherState, y, misc, Interval, InitOut, ErrStat, ErrMsg )",
"!..................................................................................................................................",
"",
"   TYPE(ModName_InitInputType),       INTENT(IN   )  :: InitInp     !< Input data for initialization routine",
"   TYPE(ModName_InputType),           INTENT(  OUT)  :: u           !< An initial guess for the input; input mesh must be defined",
"   TYPE(ModName_ParameterType),       INTENT(  OUT)  :: p           !< Parameters",
"   TYPE(ModName_ContinuousStateType), INTENT(  OUT)  :: x           !< Initial continuous states",
"   TYPE(ModName_DiscreteStateType),   INTENT(  OUT)  :: xd          !< Initial discrete states",
"   TYPE(ModName_ConstraintStateType), INTENT(  OUT)  :: z           !< Initial guess of the constraint states",
"   TYPE(ModName_OtherStateType),      INTENT(  OUT)  :: OtherState  !< Initial other states (logical, etc)",
"   TYPE(ModName_OutputType),          INTENT(  OUT)  :: y           !< Initial system outputs (outputs are not calculated;",
"                                                                    !!   only the output mesh is initialized)",
"   TYPE(ModName_MiscVarType),         INTENT(  OUT)  :: misc        !< Misc variables for optimization (not copied in glue code)",
"   REAL(DbKi),                        INTENT(INOUT)  :: Interval    !< Coupling interval in seconds: the rate that",
"                                                                    !!   (1) ModName_UpdateStates() is called in loose coupling &",
"                                                                    !!   (2) ModName_UpdateDiscState() is called in tight coupling.",
"                                                                    !!   Input is the suggested time from the glue code;",
"                                                                    !!   Output is the actual coupling interval that will be used",
"                                                                    !!   by the glue code.",
"   TYPE(ModName_InitOutputType),      INTENT(  OUT)  :: InitOut     !< Output for initialization routine",
"   INTEGER(IntKi),                    INTENT(  OUT)  :: ErrStat     !< Error status of the operation",
"   CHARACTER(*),                      INTENT(  OUT)  :: ErrMsg      !< Error message if ErrStat /= ErrID_None",
"",
"      ! local variables",
"",
"   INTEGER(IntKi)                                    :: NumOuts     ! number of outputs; would probably be in the parameter type",
"   INTEGER(IntKi)                                    :: ErrStat2    ! local error status",
"   CHARACTER(ErrMsgLen)                              :: ErrMsg2     ! local error message",
"   CHARACTER(*), PARAMETER                           :: RoutineName = 'ModName_Init'",
"",
"      !! Initialize variables",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"   NumOuts = 2",
"",
"",
"      ! Initialize the NWTC Subroutine Library",
"",
"   call NWTC_Init( )",
"",
"      ! Display the module information",
"",
"   call DispNVD( ModName_Ver )",
"",
"",
"      ! Define parameters here:",
"",
"   p%DT  = Interval",
"",
"",
"      ! Define initial system states here:",
"",
"   x%DummyContState           = 0.0_ReKi",
"   xd%DummyDiscState          = 0.0_ReKi",
"   z%DummyConstrState         = 0.0_ReKi",
"   OtherState%DummyOtherState = 0.0_ReKi",
"",
"      ! define optimization variables here:",
"   misc%DummyMiscVar          = 0.0_ReKi",
"",
"      ! Define initial guess for the system inputs here:",
"",
"   u%DummyInput = 0.0_ReKi",
"",
"",
"      ! Define system output initializations (set up mesh) here:",
"   call AllocAry( y%WriteOutput, NumOuts, 'WriteOutput', ErrStat2, ErrMsg2 )",
"      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName) ! set return error status based on local (concatenate errors)",
"      if (ErrStat >= AbortErrLev) return        ! if there are local variables that need to be deallocated, do so before early return",
"      ",
"   y%DummyOutput = 0",
"   y%WriteOutput = 0",
"",
"",
"      ! Define initialization-routine output here:",
"   call AllocAry(InitOut%WriteOutputHdr,NumOuts,'WriteOutputHdr',ErrStat2,ErrMsg2); call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"   call AllocAry(InitOut%WriteOutputUnt,NumOuts,'WriteOutputUnt',ErrStat2,ErrMsg2); call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"      if (ErrStat >= AbortErrLev) return        ! if there are local variables that need to be deallocated, do so before early return",
"",
"   InitOut%WriteOutputHdr = (/ 'Time   ', 'Column2' /)",
"   InitOut%WriteOutputUnt = (/ '(s)',     '(-)'     /)",
"",
"",
"      ! If you want to choose your own rate instead of using what the glue code suggests, tell the glue code the rate at which",
"      !   this module must be called here:",
"",
"    !Interval = p%DT",
"",
"",
"   if (InitInp%Linearize) then",
"",
"      ! If this module does not implement the four Jacobian routines at the end of this template, or the module cannot",
"      ! linearize with the features that are enabled, stop the simulation if InitInp%Linearize is true.",
"",
"      CALL SetErrStat( ErrID_Fatal, 'ModuleName cannot perform linearization analysis.', ErrStat, ErrMsg, RoutineName)",
"",
"      ! Otherwise, if the module does allow linearization, return the appropriate Jacobian row/column names and rotating-frame flags  here:",
"      ! Allocate and set these variables: InitOut%LinNames_y, InitOut%LinNames_x, InitOut%LinNames_xd, InitOut%LinNames_z, InitOut%LinNames_u",
"      ! Allocate and set these variables: InitOut%RotFrame_y, InitOut%RotFrame_x, InitOut%RotFrame_xd, InitOut%RotFrame_z, InitOut%RotFrame_u",
"      ! Allocate and set these variables: InitOut%IsLoad_u, InitOut%DerivOrder_x",
"",
"   end if",
"",
"",
"END SUBROUTINE ModName_Init",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> This routine is called at the end of the simulation.",
"SUBROUTINE ModName_End( u, p, x, xd, z, OtherState, y, misc, ErrStat, ErrMsg )",
"!..................................................................................................................................",
"",
"   TYPE(ModName_InputType),           INTENT(INOUT)  :: u           !< System inputs",
"   TYPE(ModName_ParameterType),       INTENT(INOUT)  :: p           !< Parameters",
"   TYPE(ModName_ContinuousStateType), INTENT(INOUT)  :: x           !< Continuous states",
"   TYPE(ModName_DiscreteStateType),   INTENT(INOUT)  :: xd          !< Discrete states",
"   TYPE(ModName_ConstraintStateType), INTENT(INOUT)  :: z           !< Constraint states",
"   TYPE(ModName_OtherStateType),      INTENT(INOUT)  :: OtherState  !< Other states",
"   TYPE(ModName_OutputType),          INTENT(INOUT)  :: y           !< System outputs",
"   TYPE(ModName_MiscVarType),         INTENT(INOUT)  :: misc        !< Misc variables for optimization (not copied in glue code)",
"   INTEGER(IntKi),                    INTENT(  OUT)  :: ErrStat     !< Error status of the operation",
"   CHARACTER(*),                      INTENT(  OUT)  :: ErrMsg      !< Error message if ErrStat /= ErrID_None",
"",
"      ! local variables",
"   INTEGER(IntKi)                                    :: ErrStat2    ! local error status",
"   CHARACTER(ErrMsgLen)                              :: ErrMsg2     ! local error message",
"   CHARACTER(*), PARAMETER                           :: RoutineName = 'ModName_End'",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"",
"      !! Place any last minute operations or calculations here:",
"",
"",
"      !! Close files here (but because of checkpoint-restart capability, it is not recommended to have files open during the simulation):",
"",
"",
"      !! Destroy the input data:",
"",
"   call ModName_DestroyInput( u, ErrStat2, ErrMsg2 )",
"      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"",
"",
"      !! Destroy the parameter data:",
"",
"   call ModName_DestroyParam( p, ErrStat2, ErrMsg2 )",
"      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"",
"      !! Destroy the state data:",
"",
"   call ModName_DestroyContState(   x,          ErrStat2,ErrMsg2); call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"   call ModName_DestroyDiscState(   xd,         ErrStat2,ErrMsg2); call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"   call ModName_DestroyConstrState( z,          ErrStat2,ErrMsg2); call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"   call ModName_DestroyOtherState(  OtherState, ErrStat2,ErrMsg2); call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"",
"",
"      !! Destroy the output data:",
"",
"   call ModName_DestroyOutput( y, ErrStat2, ErrMsg2 ); call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"",
"   ",
"      !! Destroy the misc data:",
"",
"   call ModName_DestroyMisc( misc, ErrStat2, ErrMsg2 ); call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"",
"",
"END SUBROUTINE ModName_End",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> This is a loose coupling routine for solving constraint states, integrating continuous states, and updating discrete and other ",
"!! states. Continuous, constraint, discrete, and other states are updated to values at t + Interval.",
"SUBROUTINE ModName_UpdateStates( t, n, Inputs, InputTimes, p, x, xd, z, OtherState, misc, ErrStat, ErrMsg )",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                         INTENT(IN   ) :: t               !< Current simulation time in seconds",
"   INTEGER(IntKi),                     INTENT(IN   ) :: n               !< Current step of the simulation: t = n*Interval",
"   TYPE(ModName_InputType),            INTENT(INOUT) :: Inputs(:)       !< Inputs at InputTimes (output from this routine only ",
"                                                                        !!  because of record keeping in routines that copy meshes)",
"   REAL(DbKi),                         INTENT(IN   ) :: InputTimes(:)   !< Times in seconds associated with Inputs",
"   TYPE(ModName_ParameterType),        INTENT(IN   ) :: p               !< Parameters",
"   TYPE(ModName_ContinuousStateType),  INTENT(INOUT) :: x               !< Input: Continuous states at t;",
"                                                                        !!   Output: Continuous states at t + Interval",
"   TYPE(ModName_DiscreteStateType),    INTENT(INOUT) :: xd              !< Input: Discrete states at t;",
"                                                                        !!   Output: Discrete states at t + Interval",
"   TYPE(ModName_ConstraintStateType),  INTENT(INOUT) :: z               !< Input: Constraint states at t;",
"                                                                        !!   Output: Constraint states at t + Interval",
"   TYPE(ModName_OtherStateType),       INTENT(INOUT) :: OtherState      !< Other states: Other states at t;",
"                                                                        !!   Output: Other states at t + Interval",
"   TYPE(ModName_MiscVarType),          INTENT(INOUT) :: misc            !<  Misc variables for optimization (not copied in glue code)",
"   INTEGER(IntKi),                     INTENT(  OUT) :: ErrStat         !< Error status of the operation",
"   CHARACTER(*),                       INTENT(  OUT) :: ErrMsg          !< Error message if ErrStat /= ErrID_None",
"",
"      ! Local variables",
"",
"   TYPE(ModName_ContinuousStateType)                 :: dxdt            ! Continuous state derivatives at t",
"   TYPE(ModName_DiscreteStateType)                   :: xd_t            ! Discrete states at t (copy)",
"   TYPE(ModName_ConstraintStateType)                 :: z_Residual      ! Residual of the constraint state functions (Z)",
"   TYPE(ModName_InputType)                           :: u               ! Instantaneous inputs",
"   ",
"   INTEGER(IntKi)                                    :: ErrStat2        ! local error status",
"   CHARACTER(ErrMsgLen)                              :: ErrMsg2         ! local error message",
"   CHARACTER(*), PARAMETER                           :: RoutineName = 'ModName_UpdateStates'",
"",
"",
"      ! Initialize variables",
"",
"   ErrStat   = ErrID_None           ! no error has occurred",
"   ErrMsg    = ''",
"",
"",
"   ! This subroutine contains an example of how the states could be updated. Developers will",
"   ! want to adjust the logic as necessary for their own situations.",
"",
"",
"",
"   ! Get the inputs at time t, based on the array of values sent by the glue code:",
"",
"   ! before calling ExtrapInterp routine, memory in u must be allocated; we can do that with a copy:",
"   call ModName_CopyInput( Inputs(1), u, MESH_NEWCOPY, ErrStat2, ErrMsg2 )",
"      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"      if ( ErrStat >= AbortErrLev ) then",
"         call cleanup()       ! to avoid memory leaks, we have to destroy the local variables that may have allocatable arrays or meshes",
"         return",
"      end if",
"",
"   call ModName_Input_ExtrapInterp( Inputs, InputTimes, u, t, ErrStat2, ErrMsg2 )  ",
"      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"      if ( ErrStat >= AbortErrLev ) then",
"         call cleanup()",
"         return",
"      end if",
"",
"",
"",
"      ! Get first time derivatives of continuous states (dxdt):",
"",
"   call ModName_CalcContStateDeriv( t, u, p, x, xd, z, OtherState, misc, dxdt, ErrStat2, ErrMsg2 )",
"      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"      if ( ErrStat >= AbortErrLev ) then",
"         call cleanup()",
"         return",
"      end if",
"",
"",
"      ! Update discrete states:",
"      !   Note that xd [discrete state] is changed in ModName_UpdateDiscState() so xd will now contain values at t+Interval",
"      !   We'll first make a copy that contains xd at time t, which will be used in computing the constraint states",
"   call ModName_CopyDiscState( xd, xd_t, MESH_NEWCOPY, ErrStat2, ErrMsg2 )",
"      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"      if ( ErrStat >= AbortErrLev ) then",
"         call cleanup()",
"         return",
"      end if",
"",
"   call ModName_UpdateDiscState( t, n, u, p, x, xd, z, OtherState, misc, ErrStat2, ErrMsg2 )",
"      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"      if ( ErrStat >= AbortErrLev ) then",
"         call cleanup()",
"         return",
"      end if",
"",
"",
"      ! Solve for the constraint states (z) here:",
"",
"      ! Iterate until the value is within a given tolerance.",
"",
"   ! DO ",
"",
"      call ModName_CalcConstrStateResidual( t, u, p, x, xd_t, z, OtherState, misc, Z_Residual, ErrStat2, ErrMsg2 )",
"         call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)",
"         if ( ErrStat >= AbortErrLev ) then",
"            call cleanup()",
"            return",
"         end if",
"",
"      !  z =",
"",
"   ! END DO",
"",
"",
"",
"      ! Integrate (update) continuous states (x) here:",
"",
"   !x = function of dxdt and x",
"",
"",
"      ! Destroy local variables before returning",
"   call cleanup()",
"",
"",
"CONTAINS",
"   SUBROUTINE cleanup()",
"   ! note that this routine inherits all of the data in ModName_UpdateStates",
"",
"",
"   CALL ModName_DestroyInput(       u,          ErrStat2, ErrMsg2)",
"   CALL ModName_DestroyConstrState( Z_Residual, ErrStat2, ErrMsg2)",
"   CALL ModName_DestroyContState(   dxdt,       ErrStat2, ErrMsg2)",
"   CALL ModName_DestroyDiscState(   xd_t,       ErrStat2, ErrMsg2) ",
"",
"   END SUBROUTINE cleanup",
"END SUBROUTINE ModName_UpdateStates",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> This is a routine for computing outputs, used in both loose and tight coupling.",
"SUBROUTINE ModName_CalcOutput( t, u, p, x, xd, z, OtherState, y, misc, ErrStat, ErrMsg )",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                        INTENT(IN   )  :: t           !< Current simulation time in seconds",
"   TYPE(ModName_InputType),           INTENT(IN   )  :: u           !< Inputs at t",
"   TYPE(ModName_ParameterType),       INTENT(IN   )  :: p           !< Parameters",
"   TYPE(ModName_ContinuousStateType), INTENT(IN   )  :: x           !< Continuous states at t",
"   TYPE(ModName_DiscreteStateType),   INTENT(IN   )  :: xd          !< Discrete states at t",
"   TYPE(ModName_ConstraintStateType), INTENT(IN   )  :: z           !< Constraint states at t",
"   TYPE(ModName_OtherStateType),      INTENT(IN   )  :: OtherState  !< Other states at t",
"   TYPE(ModName_MiscVarType),         INTENT(INOUT)  :: misc        !< Misc variables for optimization (not copied in glue code)",
"   TYPE(ModName_OutputType),          INTENT(INOUT)  :: y           !< Outputs computed at t (Input only so that mesh con-",
"                                                                    !!   nectivity information does not have to be recalculated)",
"   INTEGER(IntKi),                    INTENT(  OUT)  :: ErrStat     !< Error status of the operation",
"   CHARACTER(*),                      INTENT(  OUT)  :: ErrMsg      !< Error message if ErrStat /= ErrID_None",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"",
"      ! Compute outputs here:",
"   y%DummyOutput    = 2.0_ReKi",
"",
"   y%WriteOutput(1) = REAL(t,ReKi)",
"   y%WriteOutput(2) = 1.0_ReKi",
"",
"",
"END SUBROUTINE ModName_CalcOutput",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> This is a tight coupling routine for computing derivatives of continuous states.",
"SUBROUTINE ModName_CalcContStateDeriv( t, u, p, x, xd, z, OtherState, misc, dxdt, ErrStat, ErrMsg )",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                        INTENT(IN   )  :: t           !< Current simulation time in seconds",
"   TYPE(ModName_InputType),           INTENT(IN   )  :: u           !< Inputs at t",
"   TYPE(ModName_ParameterType),       INTENT(IN   )  :: p           !< Parameters",
"   TYPE(ModName_ContinuousStateType), INTENT(IN   )  :: x           !< Continuous states at t",
"   TYPE(ModName_DiscreteStateType),   INTENT(IN   )  :: xd          !< Discrete states at t",
"   TYPE(ModName_ConstraintStateType), INTENT(IN   )  :: z           !< Constraint states at t",
"   TYPE(ModName_OtherStateType),      INTENT(IN   )  :: OtherState  !< Other states at t",
"   TYPE(ModName_MiscVarType),         INTENT(INOUT)  :: misc        !< Misc variables for optimization (not copied in glue code)",
"   TYPE(ModName_ContinuousStateType), INTENT(  OUT)  :: dxdt        !< Continuous state derivatives at t",
"   INTEGER(IntKi),                    INTENT(  OUT)  :: ErrStat     !< Error status of the operation",
"   CHARACTER(*),                      INTENT(  OUT)  :: ErrMsg      !< Error message if ErrStat /= ErrID_None",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"",
"      ! Compute the first time derivatives of the continuous states here:",
"",
"   dxdt%DummyContState = 0.0_ReKi",
"",
"",
"END SUBROUTINE ModName_CalcContStateDeriv",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> This is a tight coupling routine for updating discrete states.",
"SUBROUTINE ModName_UpdateDiscState( t, n, u, p, x, xd, z, OtherState, misc, ErrStat, ErrMsg )",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                        INTENT(IN   )  :: t           !< Current simulation time in seconds",
"   INTEGER(IntKi),                    INTENT(IN   )  :: n           !< Current step of the simulation: t = n*Interval",
"   TYPE(ModName_InputType),           INTENT(IN   )  :: u           !< Inputs at t",
"   TYPE(ModName_ParameterType),       INTENT(IN   )  :: p           !< Parameters",
"   TYPE(ModName_ContinuousStateType), INTENT(IN   )  :: x           !< Continuous states at t",
"   TYPE(ModName_DiscreteStateType),   INTENT(INOUT)  :: xd          !< Input: Discrete states at t;",
"                                                                    !!   Output: Discrete states at t + Interval",
"   TYPE(ModName_ConstraintStateType), INTENT(IN   )  :: z           !< Constraint states at t",
"   TYPE(ModName_OtherStateType),      INTENT(IN   )  :: OtherState  !< Other states at t",
"   TYPE(ModName_MiscVarType),         INTENT(INOUT)  :: misc        !< Misc variables for optimization (not copied in glue code)",
"   INTEGER(IntKi),                    INTENT(  OUT)  :: ErrStat     !< Error status of the operation",
"   CHARACTER(*),                      INTENT(  OUT)  :: ErrMsg      !< Error message if ErrStat /= ErrID_None",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"",
"      ! Update discrete states here:",
"",
"   xd%DummyDiscState = 0.0_Reki",
"",
"END SUBROUTINE ModName_UpdateDiscState",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> This is a tight coupling routine for solving for the residual of the constraint state functions.",
"SUBROUTINE ModName_CalcConstrStateResidual( t, u, p, x, xd, z, OtherState, misc, Z_residual, ErrStat, ErrMsg )",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                        INTENT(IN   )  :: t           !< Current simulation time in seconds",
"   TYPE(ModName_InputType),           INTENT(IN   )  :: u           !< Inputs at t",
"   TYPE(ModName_ParameterType),       INTENT(IN   )  :: p           !< Parameters",
"   TYPE(ModName_ContinuousStateType), INTENT(IN   )  :: x           !< Continuous states at t",
"   TYPE(ModName_DiscreteStateType),   INTENT(IN   )  :: xd          !< Discrete states at t",
"   TYPE(ModName_ConstraintStateType), INTENT(IN   )  :: z           !< Constraint states at t (possibly a guess)",
"   TYPE(ModName_OtherStateType),      INTENT(IN   )  :: OtherState  !< Other states at t",
"   TYPE(ModName_MiscVarType),         INTENT(INOUT)  :: misc        !< Misc variables for optimization (not copied in glue code)",
"   TYPE(ModName_ConstraintStateType), INTENT(  OUT)  :: Z_residual  !< Residual of the constraint state functions using",
"                                                                    !!     the input values described above",
"   INTEGER(IntKi),                    INTENT(  OUT)  :: ErrStat     !< Error status of the operation",
"   CHARACTER(*),                      INTENT(  OUT)  :: ErrMsg      !< Error message if ErrStat /= ErrID_None",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"",
"      ! Solve for the residual of the constraint state functions here:",
"",
"   Z_residual%DummyConstrState = 0.0_ReKi",
"",
"END SUBROUTINE ModName_CalcConstrStateResidual",
"",
"",
"!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++",
"! ###### The following four routines are Jacobian routines for linearization capabilities #######",
"! If the module does not implement them, set ErrStat = ErrID_Fatal in ModName_Init() when InitInp%Linearize is .true.",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> Routine to compute the Jacobians of the output (Y), continuous- (X), discrete- (Xd), and constraint-state (Z) functions",
"!! with respect to the inputs (u). The partial derivatives dY/du, dX/du, dXd/du, and dZ/du are returned.",
"SUBROUTINE ModName_JacobianPInput( t, u, p, x, xd, z, OtherState, y, m, ErrStat, ErrMsg, dYdu, dXdu, dXddu, dZdu)",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                                INTENT(IN   )           :: t          !< Time in seconds at operating point",
"   TYPE(ModName_InputType),                   INTENT(IN   )           :: u          !< Inputs at operating point (may change to inout if a mesh copy is required)",
"   TYPE(ModName_ParameterType),               INTENT(IN   )           :: p          !< Parameters",
"   TYPE(ModName_ContinuousStateType),         INTENT(IN   )           :: x          !< Continuous states at operating point",
"   TYPE(ModName_DiscreteStateType),           INTENT(IN   )           :: xd         !< Discrete states at operating point",
"   TYPE(ModName_ConstraintStateType),         INTENT(IN   )           :: z          !< Constraint states at operating point",
"   TYPE(ModName_OtherStateType),              INTENT(IN   )           :: OtherState !< Other states at operating point",
"   TYPE(ModName_OutputType),                  INTENT(IN   )           :: y          !< Output (change to inout if a mesh copy is required);",
"                                                                                    !!   Output fields are not used by this routine, but type is",
"                                                                                    !!   available here so that mesh parameter information (i.e.,",
"                                                                                    !!   connectivity) does not have to be recalculated for dYdu.",
"   TYPE(ModName_MiscVarType),                 INTENT(INOUT)           :: m          !< Misc/optimization variables",
"   INTEGER(IntKi),                            INTENT(  OUT)           :: ErrStat    !< Error status of the operation",
"   CHARACTER(*),                              INTENT(  OUT)           :: ErrMsg     !< Error message if ErrStat /= ErrID_None",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dYdu(:,:)  !< Partial derivatives of output functions (Y) with respect",
"                                                                                    !!   to the inputs (u) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dXdu(:,:)  !< Partial derivatives of continuous state functions (X) with",
"                                                                                    !!   respect to the inputs (u) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dXddu(:,:) !< Partial derivatives of discrete state functions (Xd) with",
"                                                                                    !!   respect to the inputs (u) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dZdu(:,:)  !< Partial derivatives of constraint state functions (Z) with",
"                                                                                    !!   respect to the inputs (u) [intent in to avoid deallocation]",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"",
"   IF ( PRESENT( dYdu ) ) THEN",
"",
"      ! Calculate the partial derivative of the output functions (Y) with respect to the inputs (u) here:",
"",
"      ! allocate and set dYdu",
"",
"   END IF",
"",
"   IF ( PRESENT( dXdu ) ) THEN",
"",
"      ! Calculate the partial derivative of the continuous state functions (X) with respect to the inputs (u) here:",
"",
"      ! allocate and set dXdu",
"",
"   END IF",
"",
"   IF ( PRESENT( dXddu ) ) THEN",
"",
"      ! Calculate the partial derivative of the discrete state functions (Xd) with respect to the inputs (u) here:",
"",
"      ! allocate and set dXddu",
"",
"   END IF",
"",
"   IF ( PRESENT( dZdu ) ) THEN",
"",
"      ! Calculate the partial derivative of the constraint state functions (Z) with respect to the inputs (u) here:",
"",
"      ! allocate and set dZdu",
"",
"   END IF",
"",
"",
"END SUBROUTINE ModName_JacobianPInput",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> Routine to compute the Jacobians of the output (Y), continuous- (X), discrete- (Xd), and constraint-state (Z) functions",
"!! with respect to the continuous states (x). The partial derivatives dY/dx, dX/dx, dXd/dx, and dZ/dx are returned.",
"SUBROUTINE ModName_JacobianPContState( t, u, p, x, xd, z, OtherState, y, m, ErrStat, ErrMsg, dYdx, dXdx, dXddx, dZdx )",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                                INTENT(IN   )           :: t          !< Time in seconds at operating point",
"   TYPE(ModName_InputType),                   INTENT(IN   )           :: u          !< Inputs at operating point (may change to inout if a mesh copy is required)",
"   TYPE(ModName_ParameterType),               INTENT(IN   )           :: p          !< Parameters",
"   TYPE(ModName_ContinuousStateType),         INTENT(IN   )           :: x          !< Continuous states at operating point",
"   TYPE(ModName_DiscreteStateType),           INTENT(IN   )           :: xd         !< Discrete states at operating point",
"   TYPE(ModName_ConstraintStateType),         INTENT(IN   )           :: z          !< Constraint states at operating point",
"   TYPE(ModName_OtherStateType),              INTENT(IN   )           :: OtherState !< Other states at operating point",
"   TYPE(ModName_OutputType),                  INTENT(IN   )           :: y          !< Output (change to inout if a mesh copy is required);",
"                                                                                    !!   Output fields are not used by this routine, but type is",
"                                                                                    !!   available here so that mesh parameter information (i.e.,",
"                                                                                    !!   connectivity) does not have to be recalculated for dYdx.",
"   TYPE(ModName_MiscVarType),                 INTENT(INOUT)           :: m          !< Misc/optimization variables",
"   INTEGER(IntKi),                            INTENT(  OUT)           :: ErrStat    !< Error status of the operation",
"   CHARACTER(*),                              INTENT(  OUT)           :: ErrMsg     !< Error message if ErrStat /= ErrID_None",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dYdx(:,:)  !< Partial derivatives of output functions",
"                                                                                    !!   (Y) with respect to the continuous",
"                                                                                    !!   states (x) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dXdx(:,:)  !< Partial derivatives of continuous state",
"                                                                                    !!   functions (X) with respect to",
"                                                                                    !!   the continuous states (x) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dXddx(:,:) !< Partial derivatives of discrete state",
"                                                                                    !!   functions (Xd) with respect to",
"                                                                                    !!   the continuous states (x) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dZdx(:,:)  !< Partial derivatives of constraint state",
"                                                                                    !!   functions (Z) with respect to",
"                                                                                    !!   the continuous states (x) [intent in to avoid deallocation]",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"",
"",
"   IF ( PRESENT( dYdx ) ) THEN",
"",
"      ! Calculate the partial derivative of the output functions (Y) with respect to the continuous states (x) here:",
"",
"      ! allocate and set dYdx",
"",
"   END IF",
"",
"   IF ( PRESENT( dXdx ) ) THEN",
"",
"      ! Calculate the partial derivative of the continuous state functions (X) with respect to the continuous states (x) here:",
"",
"      ! allocate and set dXdx",
"",
"   END IF",
"",
"   IF ( PRESENT( dXddx ) ) THEN",
"",
"      ! Calculate the partial derivative of the discrete state functions (Xd) with respect to the continuous states (x) here:",
"",
"      ! allocate and set dXddx",
"",
"   END IF",
"",
"   IF ( PRESENT( dZdx ) ) THEN",
"",
"",
"      ! Calculate the partial derivative of the constraint state functions (Z) with respect to the continuous states (x) here:",
"",
"      ! allocate and set dZdx",
"",
"   END IF",
"",
"",
"END SUBROUTINE ModName_JacobianPContState",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> Routine to compute the Jacobians of the output (Y), continuous- (X), discrete- (Xd), and constraint-state (Z) functions",
"!! with respect to the discrete states (xd). The partial derivatives dY/dxd, dX/dxd, dXd/dxd, and dZ/dxd are returned.",
"SUBROUTINE ModName_JacobianPDiscState( t, u, p, x, xd, z, OtherState, y, m, ErrStat, ErrMsg, dYdxd, dXdxd, dXddxd, dZdxd )",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                                INTENT(IN   )           :: t          !< Time in seconds at operating point",
"   TYPE(ModName_InputType),                   INTENT(IN   )           :: u          !< Inputs at operating point (may change to inout if a mesh copy is required)",
"   TYPE(ModName_ParameterType),               INTENT(IN   )           :: p          !< Parameters",
"   TYPE(ModName_ContinuousStateType),         INTENT(IN   )           :: x          !< Continuous states at operating point",
"   TYPE(ModName_DiscreteStateType),           INTENT(IN   )           :: xd         !< Discrete states at operating point",
"   TYPE(ModName_ConstraintStateType),         INTENT(IN   )           :: z          !< Constraint states at operating point",
"   TYPE(ModName_OtherStateType),              INTENT(IN   )           :: OtherState !< Other states at operating point",
"   TYPE(ModName_OutputType),                  INTENT(IN   )           :: y          !< Output (change to inout if a mesh copy is required);",
"                                                                                    !!   Output fields are not used by this routine, but type is",
"                                                                                    !!   available here so that mesh parameter information (i.e.,",
"                                                                                    !!   connectivity) does not have to be recalculated for dYdxd.",
"   TYPE(ModName_MiscVarType),                 INTENT(INOUT)           :: m          !< Misc/optimization variables",
"   INTEGER(IntKi),                            INTENT(  OUT)           :: ErrStat    !< Error status of the operation",
"   CHARACTER(*),                              INTENT(  OUT)           :: ErrMsg     !< Error message if ErrStat /= ErrID_None",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dYdxd(:,:) !< Partial derivatives of output functions",
"                                                                                    !!  (Y) with respect to the discrete",
"                                                                                    !!  states (xd) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dXdxd(:,:) !< Partial derivatives of continuous state",
"                                                                                    !!   functions (X) with respect to the",
"                                                                                    !!   discrete states (xd) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dXddxd(:,:)!< Partial derivatives of discrete state",
"                                                                                    !!   functions (Xd) with respect to the",
"                                                                                    !!   discrete states (xd) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dZdxd(:,:) !< Partial derivatives of constraint state",
"                                                                                    !!   functions (Z) with respect to the",
"                                                                                    !!   discrete states (xd) [intent in to avoid deallocation]",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"",
"   IF ( PRESENT( dYdxd ) ) THEN",
"",
"      ! Calculate the partial derivative of the output functions (Y) with respect to the discrete states (xd) here:",
"",
"      ! allocate and set dYdxd",
"",
"   END IF",
"",
"   IF ( PRESENT( dXdxd ) ) THEN",
"",
"      ! Calculate the partial derivative of the continuous state functions (X) with respect to the discrete states (xd) here:",
"",
"      ! allocate and set dXdxd",
"",
"   END IF",
"",
"   IF ( PRESENT( dXddxd ) ) THEN",
"",
"      ! Calculate the partial derivative of the discrete state functions (Xd) with respect to the discrete states (xd) here:",
"",
"      ! allocate and set dXddxd",
"",
"   END IF",
"",
"   IF ( PRESENT( dZdxd ) ) THEN",
"",
"      ! Calculate the partial derivative of the constraint state functions (Z) with respect to the discrete states (xd) here:",
"",
"      ! allocate and set dZdxd",
"",
"   END IF",
"",
"",
"END SUBROUTINE ModName_JacobianPDiscState",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> Routine to compute the Jacobians of the output (Y), continuous- (X), discrete- (Xd), and constraint-state (Z) functions",
"!! with respect to the constraint states (z). The partial derivatives dY/dz, dX/dz, dXd/dz, and dZ/dz are returned.",
"SUBROUTINE ModName_JacobianPConstrState( t, u, p, x, xd, z, OtherState, y, m, ErrStat, ErrMsg, dYdz, dXdz, dXddz, dZdz )",
"!..................................................................................................................................",
"",
"   REAL(DbKi),                                INTENT(IN   )           :: t          !< Time in seconds at operating point",
"   TYPE(ModName_InputType),                   INTENT(IN   )           :: u          !< Inputs at operating point (may change to inout if a mesh copy is required)",
"   TYPE(ModName_ParameterType),               INTENT(IN   )           :: p          !< Parameters",
"   TYPE(ModName_ContinuousStateType),         INTENT(IN   )           :: x          !< Continuous states at operating point",
"   TYPE(ModName_DiscreteStateType),           INTENT(IN   )           :: xd         !< Discrete states at operating point",
"   TYPE(ModName_ConstraintStateType),         INTENT(IN   )           :: z          !< Constraint states at operating point",
"   TYPE(ModName_OtherStateType),              INTENT(IN   )           :: OtherState !< Other states at operating point",
"   TYPE(ModName_OutputType),                  INTENT(IN   )           :: y          !< Output (change to inout if a mesh copy is required);",
"                                                                                    !!   Output fields are not used by this routine, but type is",
"                                                                                    !!   available here so that mesh parameter information (i.e.,",
"                                                                                    !!   connectivity) does not have to be recalculated for dYdz.",
"   TYPE(ModName_MiscVarType),                 INTENT(INOUT)           :: m          !< Misc/optimization variables",
"   INTEGER(IntKi),                            INTENT(  OUT)           :: ErrStat    !< Error status of the operation",
"   CHARACTER(*),                              INTENT(  OUT)           :: ErrMsg     !< Error message if ErrStat /= ErrID_None",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dYdz(:,:)  !< Partial derivatives of output",
"                                                                                    !!  functions (Y) with respect to the",
"                                                                                    !!  constraint states (z) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dXdz(:,:)  !< Partial derivatives of continuous",
"                                                                                    !!  state functions (X) with respect to",
"                                                                                    !!  the constraint states (z) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dXddz(:,:) !< Partial derivatives of discrete state",
"                                                                                    !!  functions (Xd) with respect to the",
"                                                                                    !!  constraint states (z) [intent in to avoid deallocation]",
"   REAL(R8Ki), ALLOCATABLE, OPTIONAL,         INTENT(INOUT)           :: dZdz(:,:)  !< Partial derivatives of constraint",
"                                                                                    !! state functions (Z) with respect to",
"                                                                                    !!  the constraint states (z) [intent in to avoid deallocation]",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"   IF ( PRESENT( dYdz ) ) THEN",
"",
"         ! Calculate the partial derivative of the output functions (Y) with respect to the constraint states (z) here:",
"",
"      ! allocate and set dYdz",
"",
"   END IF",
"",
"   IF ( PRESENT( dXdz ) ) THEN",
"",
"         ! Calculate the partial derivative of the continuous state functions (X) with respect to the constraint states (z) here:",
"",
"      ! allocate and set dXdz",
"",
"   END IF",
"",
"   IF ( PRESENT( dXddz ) ) THEN",
"",
"         ! Calculate the partial derivative of the discrete state functions (Xd) with respect to the constraint states (z) here:",
"",
"      ! allocate and set dXddz",
"",
"   END IF",
"",
"   IF ( PRESENT( dZdz ) ) THEN",
"",
"         ! Calculate the partial derivative of the constraint state functions (Z) with respect to the constraint states (z) here:",
"",
"      ! allocate and set dZdz",
"",
"   END IF",
"",
"",
"END SUBROUTINE ModName_JacobianPConstrState",
"!----------------------------------------------------------------------------------------------------------------------------------",
"!> Routine to pack the data structures representing the operating points into arrays for linearization.",
"SUBROUTINE ModName_GetOP( t, u, p, x, xd, z, OtherState, y, m, ErrStat, ErrMsg, u_op, y_op, x_op, dx_op, xd_op, z_op )",
"",
"   REAL(DbKi),                           INTENT(IN   )           :: t          !< Time in seconds at operating point",
"   TYPE(ModName_InputType),              INTENT(IN   )           :: u          !< Inputs at operating point (may change to inout if a mesh copy is required)",
"   TYPE(ModName_ParameterType),          INTENT(IN   )           :: p          !< Parameters",
"   TYPE(ModName_ContinuousStateType),    INTENT(IN   )           :: x          !< Continuous states at operating point",
"   TYPE(ModName_DiscreteStateType),      INTENT(IN   )           :: xd         !< Discrete states at operating point",
"   TYPE(ModName_ConstraintStateType),    INTENT(IN   )           :: z          !< Constraint states at operating point",
"   TYPE(ModName_OtherStateType),         INTENT(IN   )           :: OtherState !< Other states at operating point",
"   TYPE(ModName_OutputType),             INTENT(IN   )           :: y          !< Output at operating point",
"   TYPE(ModName_MiscVarType),            INTENT(INOUT)           :: m          !< Misc/optimization variables",
"   INTEGER(IntKi),                       INTENT(  OUT)           :: ErrStat    !< Error status of the operation",
"   CHARACTER(*),                         INTENT(  OUT)           :: ErrMsg     !< Error message if ErrStat /= ErrID_None",
"   REAL(ReKi), ALLOCATABLE, OPTIONAL,    INTENT(INOUT)           :: u_op(:)    !< values of linearized inputs",
"   REAL(ReKi), ALLOCATABLE, OPTIONAL,    INTENT(INOUT)           :: y_op(:)    !< values of linearized outputs",
"   REAL(ReKi), ALLOCATABLE, OPTIONAL,    INTENT(INOUT)           :: x_op(:)    !< values of linearized continuous states",
"   REAL(ReKi), ALLOCATABLE, OPTIONAL,    INTENT(INOUT)           :: dx_op(:)   !< values of first time derivatives of linearized continuous states",
"   REAL(ReKi), ALLOCATABLE, OPTIONAL,    INTENT(INOUT)           :: xd_op(:)   !< values of linearized discrete states",
"   REAL(ReKi), ALLOCATABLE, OPTIONAL,    INTENT(INOUT)           :: z_op(:)    !< values of linearized constraint states",
"",
"",
"      ! Initialize ErrStat",
"",
"   ErrStat = ErrID_None",
"   ErrMsg  = ''",
"",
"   IF ( PRESENT( u_op ) ) THEN",
"",
"   END IF",
"",
"   IF ( PRESENT( y_op ) ) THEN",
"   END IF",
"",
"   IF ( PRESENT( x_op ) ) THEN",
"",
"   END IF",
"",
"   IF ( PRESENT( dx_op ) ) THEN",
"",
"   END IF",
"",
"   IF ( PRESENT( xd_op ) ) THEN",
"",
"   END IF",
"",
"   IF ( PRESENT( z_op ) ) THEN",
"",
"   END IF",
"",
"END SUBROUTINE ModName_GetOP",
"!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++",
"",
"END MODULE ModuleName",
"!**********************************************************************************************************************************",
0L } ;
