! This module is an add on to AeroDyn 15 to allow output of Aerodynamic data at each blade node.
!
!  Copyright 2016   Envision Energy
!
MODULE AeroDyn_AllBldNdOuts_IO

   USE NWTC_Library
   USE NWTC_LAPACK
   USE AeroDyn_Types

   IMPLICIT NONE

   PRIVATE

   PUBLIC   :: AllBldNdOuts_InitOut
   PUBLIC   :: Calc_WriteAllBldNdOutput
   PUBLIC   :: AllBldNdOuts_SetParameters


      ! Parameters related to output length (number of characters allowed in the output data headers):

   INTEGER(IntKi), PARAMETER      :: OutStrLenM1 = ChanLen - 6    ! The NREL allowed channel name length is usually 20.  We are making these of the form B#N##namesuffix


! ===================================================================================================
! NOTE: The following lines of code were generated by a Matlab script called "Write_ChckOutLst.m"
!      using the parameters listed in the "OutListParameters.xlsx" Excel file. Any changes to these
!      lines should be modified in the Matlab script and/or Excel worksheet as necessary.
! ===================================================================================================
! This code was generated by Write_ChckOutLst.m at 02-Mar-2022 11:12:19.


     ! NOTES:
     !    (1) These parameters are in the order stored in "OutListParameters.xlsx"


     ! Blade:

   INTEGER(IntKi), PARAMETER      :: BldNd_VUndx     =  1
   INTEGER(IntKi), PARAMETER      :: BldNd_VUndy     =  2
   INTEGER(IntKi), PARAMETER      :: BldNd_VUndz     =  3
   INTEGER(IntKi), PARAMETER      :: BldNd_Vundxi    =  4
   INTEGER(IntKi), PARAMETER      :: BldNd_Vundyi    =  5
   INTEGER(IntKi), PARAMETER      :: BldNd_Vundzi    =  6
   INTEGER(IntKi), PARAMETER      :: BldNd_VDisx     =  7
   INTEGER(IntKi), PARAMETER      :: BldNd_VDisy     =  8
   INTEGER(IntKi), PARAMETER      :: BldNd_VDisz     =  9
   INTEGER(IntKi), PARAMETER      :: BldNd_STVx      = 10
   INTEGER(IntKi), PARAMETER      :: BldNd_STVy      = 11
   INTEGER(IntKi), PARAMETER      :: BldNd_STVz      = 12
   INTEGER(IntKi), PARAMETER      :: BldNd_VRel      = 13
   INTEGER(IntKi), PARAMETER      :: BldNd_DynP      = 14
   INTEGER(IntKi), PARAMETER      :: BldNd_Re        = 15
   INTEGER(IntKi), PARAMETER      :: BldNd_M         = 16
   INTEGER(IntKi), PARAMETER      :: BldNd_Vindx     = 17
   INTEGER(IntKi), PARAMETER      :: BldNd_Vindy     = 18
   INTEGER(IntKi), PARAMETER      :: BldNd_AxInd     = 19
   INTEGER(IntKi), PARAMETER      :: BldNd_TnInd     = 20
   INTEGER(IntKi), PARAMETER      :: BldNd_Alpha     = 21
   INTEGER(IntKi), PARAMETER      :: BldNd_Theta     = 22
   INTEGER(IntKi), PARAMETER      :: BldNd_Phi       = 23
   INTEGER(IntKi), PARAMETER      :: BldNd_Curve     = 24
   INTEGER(IntKi), PARAMETER      :: BldNd_Cl        = 25
   INTEGER(IntKi), PARAMETER      :: BldNd_Cd        = 26
   INTEGER(IntKi), PARAMETER      :: BldNd_Cm        = 27
   INTEGER(IntKi), PARAMETER      :: BldNd_Cx        = 28
   INTEGER(IntKi), PARAMETER      :: BldNd_Cy        = 29
   INTEGER(IntKi), PARAMETER      :: BldNd_Cn        = 30
   INTEGER(IntKi), PARAMETER      :: BldNd_Ct        = 31
   INTEGER(IntKi), PARAMETER      :: BldNd_Fl        = 32
   INTEGER(IntKi), PARAMETER      :: BldNd_Fd        = 33
   INTEGER(IntKi), PARAMETER      :: BldNd_Mm        = 34
   INTEGER(IntKi), PARAMETER      :: BldNd_Fx        = 35
   INTEGER(IntKi), PARAMETER      :: BldNd_Fy        = 36
   INTEGER(IntKi), PARAMETER      :: BldNd_Fn        = 37
   INTEGER(IntKi), PARAMETER      :: BldNd_Ft        = 38
   INTEGER(IntKi), PARAMETER      :: BldNd_Clrnc     = 39
   INTEGER(IntKi), PARAMETER      :: BldNd_Vx        = 40
   INTEGER(IntKi), PARAMETER      :: BldNd_Vy        = 41
   INTEGER(IntKi), PARAMETER      :: BldNd_GeomPhi   = 42
   INTEGER(IntKi), PARAMETER      :: BldNd_Chi       = 43
   INTEGER(IntKi), PARAMETER      :: BldNd_UA_Flag   = 44
   INTEGER(IntKi), PARAMETER      :: BldNd_UA_x1     = 45
   INTEGER(IntKi), PARAMETER      :: BldNd_UA_x2     = 46
   INTEGER(IntKi), PARAMETER      :: BldNd_UA_x3     = 47
   INTEGER(IntKi), PARAMETER      :: BldNd_UA_x4     = 48
   INTEGER(IntKi), PARAMETER      :: BldNd_UA_x5     = 49
   INTEGER(IntKi), PARAMETER      :: BldNd_Debug1    = 50
   INTEGER(IntKi), PARAMETER      :: BldNd_Debug2    = 51
   INTEGER(IntKi), PARAMETER      :: BldNd_Debug3    = 52
   INTEGER(IntKi), PARAMETER      :: BldNd_CpMin     = 53
   INTEGER(IntKi), PARAMETER      :: BldNd_SgCav     = 54
   INTEGER(IntKi), PARAMETER      :: BldNd_SigCr     = 55
   INTEGER(IntKi), PARAMETER      :: BldNd_Gam       = 56
   INTEGER(IntKi), PARAMETER      :: BldNd_Cl_Static = 57
   INTEGER(IntKi), PARAMETER      :: BldNd_Cd_Static = 58
   INTEGER(IntKi), PARAMETER      :: BldNd_Cm_Static = 59
   INTEGER(IntKi), PARAMETER      :: BldNd_Uin       = 60
   INTEGER(IntKi), PARAMETER      :: BldNd_Uit       = 61
   INTEGER(IntKi), PARAMETER      :: BldNd_Uir       = 62


     ! The maximum number of output channels which can be output by the code.
   INTEGER(IntKi), PARAMETER, PUBLIC  :: BldNd_MaxOutPts       = 62

!End of code generated by Matlab script
! ===================================================================================================

CONTAINS
!----------------------------------------------------------------------------------------------------------------------------------

!----------------------------------------------------------------------------------------------------------------------------------
!> This subroutine populates the headers with the blade node outputs.  The iteration cycle is blade:node:channel (channel iterated
!! fastest).  If this iteration order is changed, it should be changed in the Calc_WriteAllBldNdOutput routine as well.
SUBROUTINE AllBldNdOuts_InitOut( InitOut, p, p_AD, InputFileData, ErrStat, ErrMsg )

   TYPE(RotInitOutputType),      INTENT(INOUT)  :: InitOut                          ! output data
   TYPE(RotParameterType),       INTENT(IN   )  :: p                                ! The rotor parameters
   TYPE(AD_ParameterType),       INTENT(IN   )  :: p_AD                             ! The module parameters
   TYPE(RotInputFile),           INTENT(IN   )  :: InputFileData                    ! All the data in the AeroDyn input file (want Blade Span for channel name)
   INTEGER(IntKi),               INTENT(  OUT)  :: ErrStat                          ! The error status code
   CHARACTER(*),                 INTENT(  OUT)  :: ErrMsg                           ! The error message, if an error occurred

   INTEGER(IntKi)                               :: INDX                             ! Index count within WriteOutput
   INTEGER(IntKi)                               :: IdxBlade                         ! Counter to which blade we are on
   INTEGER(IntKi)                               :: IdxNode                          ! Counter to the blade node we ae on
   INTEGER(IntKi)                               :: IdxChan                          ! Counter to the channel we are outputting.
   CHARACTER(16)                                :: ChanPrefix                       ! Name prefix (AB#N###)
   CHARACTER(6)                                 :: TmpChar                          ! Temporary char array to hold the node digits (3 places only!!!!)
   CHARACTER(*), PARAMETER                      :: RoutineName = ('AllBldNdOuts_InitOut')


         ! Initialize some things
      ErrMsg = ''
      ErrStat = ErrID_None



         ! Warn if we will run into issues with more than 99 nodes.
      IF (p%NumBlNds > 999 ) CALL SetErrStat(ErrID_Severe,'More than 999 blade nodes in use.  Output channel headers will not '// &
            'correctly reflect blade stations beyond 999. Modifications to the variable ChanLen in FAST are required.',ErrStat,ErrMsg,RoutineName)


         ! Populate the header an unit lines for all blades and nodes
         ! First set a counter so we know where in the output array we are in
         ! NOTE: we populate invalid names as well (some names are not valid outputs for certain configurations).  That means we will have zeros in those values.
      INDX = p%NumOuts + 1       ! p%NumOuts is the number of outputs from the normal AeroDyn output.  The WriteOutput array is sized to p%NumOuts + num(AllBldNdOuts)

      DO IdxChan=1,p%BldNd_NumOuts

         DO IdxBlade=1,p%BldNd_BladesOut
            DO IdxNode=1,p%NumBlNds

                  ! Create the name prefix:
               WRITE (TmpChar,'(I3.3)')  IdxNode         ! 3 digit number
               ChanPrefix = 'AB' // TRIM(Num2LStr(IdxBlade)) // 'N' // TRIM(TmpChar) ! // '_' ! note that I added an "AB" to indicate "Aero B1" in case of confusion with structural nodal outputs with the same name
                  ! Now write to the header
               InitOut%WriteOutputHdr(INDX) = trim(ChanPrefix) // p%BldNd_OutParam(IdxChan)%Name
               InitOut%WriteOutputUnt(INDX) = p%BldNd_OutParam(IdxChan)%Units

                  ! Increment the index to the Header arrays
               INDX = INDX + 1

            ENDDO
         ENDDO

      ENDDO

END SUBROUTINE AllBldNdOuts_InitOut

!----------------------------------------------------------------------------------------------------------------------------------
!> This subroutine populates the headers with the blade node outputs.  The iteration cycle is channel:blade:node (node iterated
!! fastest).  If this iteration order is changed, it should be changed in the Calc_WriteAllBldNdOutput routine as well.
!!
!! NOTE: the equations here came from the output section of AeroDyn_IO.f90.  If anything changes in there, it needs to be reflected
!!       here.

SUBROUTINE Calc_WriteAllBldNdOutput( p, p_AD, u, m, m_AD, x, y, OtherState, Indx, iRot, ErrStat, ErrMsg )
   TYPE(RotParameterType),       INTENT(IN   )  :: p                                 ! The rotor parameters
   TYPE(AD_ParameterType),       INTENT(IN   )  :: p_AD                              ! The module parameters
   TYPE(RotInputType),           INTENT(IN   )  :: u                                 ! inputs
   TYPE(RotMiscVarType),         INTENT(IN   )  :: m                                 ! misc variables
   TYPE(AD_MiscVarType),         INTENT(IN   )  :: m_AD                              ! misc variables ! NOTE: temporary
   TYPE(RotContinuousStateType), INTENT(IN   )  :: x                                 ! rotor Continuous states
   TYPE(RotOutputType),          INTENT(INOUT)  :: y                                 ! outputs (updates y%WriteOutput)
   TYPE(RotOtherStateType),      INTENT(IN   )  :: OtherState                        ! other states
   INTEGER,                      INTENT(IN   )  :: Indx                              ! index into m%BEMT_u(Indx) array; 1=t and 2=t+dt (but not checked here)
   INTEGER,                      INTENT(IN   )  :: iRot                              ! Rotor index, needed for OLAF
   INTEGER(IntKi),               INTENT(  OUT)  :: ErrStat                           ! The error status code
   CHARACTER(*),                 INTENT(  OUT)  :: ErrMsg                            ! The error message, if an error occurred

      ! local variables
 
   INTEGER(IntKi)                               :: OutIdx                             ! Index count within WriteOutput
   INTEGER(IntKi)                               :: IdxBlade, iW                     ! Counter to which blade we are on, and Wing
   INTEGER(IntKi)                               :: IdxNode                          ! Counter to the blade node we ae on
   INTEGER(IntKi)                               :: IdxChan                          ! Counter to the channel we are outputting.
   INTEGER(IntKi)                               :: compIndx                         ! index for array component (x,y,z)
   CHARACTER(*), PARAMETER                      :: RoutineName = 'Calc_WriteAllBldNdOutput'
   REAL(ReKi)                                   :: ct, st                  ! cosine, sine of theta
   REAL(ReKi)                                   :: cp, sp                  ! cosine, sine of phi
   real(ReKi)                                   :: M_ph(3,3)               ! Transformation from hub to "blade-rotor-plane": n,t,r (not the same as AeroDyn)
   real(ReKi)                                   :: M_pg(3,3,p%NumBlades)   ! Transformation from global to "blade-rotor-plane" (n,t,r), with same x at hub coordinate system
   real(ReKi)                                   :: psi_hub                 ! Azimuth wrt hub
   real(ReKi)                                   :: Vind_g(3)               ! Induced velocity vector in global coordinates
   real(ReKi)                                   :: Vind_s(3)               ! Induced velocity vector in section coordinates (AeroDyn "x-y")
 

         ! Initialize some things
      ErrMsg = ''
      ErrStat = ErrID_None
      ! NOTE: if no blade outputs, we return
      if (p%BldNd_BladesOut<=0 .or. p%BldNd_NumOuts<=0) then
         return
      endif


         ! Precalculate the M_ph matrix -- no reason to recalculate for each output
      DO IdxBlade=1,p%NumBlades
         psi_hub = TwoPi*(real(IdxBlade-1,ReKi))/real(p%NumBlades,ReKi)
         M_ph(1,1:3) = (/ 1.0_ReKi, 0.0_ReKi    , 0.0_ReKi     /)
         M_ph(2,1:3) = (/ 0.0_ReKi, cos(psi_hub), sin(psi_hub) /)
         M_ph(3,1:3) = (/ 0.0_ReKi,-sin(psi_hub), cos(psi_hub) /)
         M_pg(1:3,1:3,IdxBlade) = matmul(M_ph, u%HubMotion%Orientation(1:3,1:3,1) ) 
      ENDDO


         ! Populate the header an unit lines for all blades and nodes
         ! First set a counter so we know where in the output array we are in
      OutIdx = p%NumOuts + 1       ! p%NumOuts is the number of outputs from the normal AeroDyn output.  The WriteOutput array is sized to p%NumOuts + num(AllBldNdOuts)

      
         ! Case to assign output to this channel and populate based on Indx value (this indicates what the channel is)
         ! Logic and mathematics used here come from Calc_WriteOutput
      DO IdxChan=1,p%BldNd_NumOuts

         SELECT CASE( p%BldNd_OutParam(IdxChan)%Indx )      ! Indx contains the information on what channel should be output
         CASE (0) ! Invalid channel
            ! We still have headers for invalid channels.  Need to account for that
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx ) = 0.0_ReKi
                  OutIdx = OutIdx + 1
               END DO
            END DO

            ! ***** Undisturbed wind velocity in local blade coord system *****
         CASE ( BldNd_VUndx )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(1,:,IdxNode,IdxBlade), u%InflowOnBlade(:,IdxNode,IdxBlade) )
                  OutIdx = OutIdx + 1
               END DO
            END DO
                  
                  
         CASE ( BldNd_VUndy )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(2,:,IdxNode,IdxBlade), u%InflowOnBlade(:,IdxNode,IdxBlade) )
                  OutIdx = OutIdx + 1
               END DO
            END DO
                  
         CASE ( BldNd_VUndz )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(3,:,IdxNode,IdxBlade), u%InflowOnBlade(:,IdxNode,IdxBlade) )
                  OutIdx = OutIdx + 1
               END DO
            END DO
      
      

            ! ***** Disturbed wind velocity in local blade coord system *****
         CASE ( BldNd_VUndxi )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = u%InflowOnBlade(1,IdxNode,IdxBlade)
                  OutIdx = OutIdx + 1
               END DO
            END DO
                  
                  
         CASE ( BldNd_VUndyi )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = u%InflowOnBlade(2,IdxNode,IdxBlade)
                  OutIdx = OutIdx + 1
               END DO
            END DO
                  
         CASE ( BldNd_VUndzi )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = u%InflowOnBlade(3,IdxNode,IdxBlade)
                  OutIdx = OutIdx + 1
               END DO
            END DO
      
      

                  ! ***** Disturbed wind velocity in the local blade coordinate system *****
         CASE ( BldNd_VDisx )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(1,:,IdxNode,IdxBlade), m%DisturbedInflow(:,IdxNode,IdxBlade) )
                  OutIdx = OutIdx + 1
               END DO
            END DO

         CASE ( BldNd_VDisy )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(2,:,IdxNode,IdxBlade), m%DisturbedInflow(:,IdxNode,IdxBlade) )
                  OutIdx = OutIdx + 1
               END DO
            END DO
         
         CASE ( BldNd_VDisz )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(3,:,IdxNode,IdxBlade), m%DisturbedInflow(:,IdxNode,IdxBlade) )
                  OutIdx = OutIdx + 1
               END DO
            END DO

         
               ! ***** Structural translational velocity in the local blade coordinate system ***** 
         CASE ( BldNd_STVx )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(1,:,IdxNode,IdxBlade), u%BladeMotion(IdxBlade)%TranslationVel(:,IdxNode) )
                  OutIdx = OutIdx + 1
               END DO
            END DO
         
         CASE ( BldNd_STVy )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(2,:,IdxNode,IdxBlade), u%BladeMotion(IdxBlade)%TranslationVel(:,IdxNode) )
                  OutIdx = OutIdx + 1
               END DO
            END DO
         
         CASE ( BldNd_STVz )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds
                  y%WriteOutput( OutIdx )  = dot_product( m%WithoutSweepPitchTwist(3,:,IdxNode,IdxBlade), u%BladeMotion(IdxBlade)%TranslationVel(:,IdxNode) )
                  OutIdx = OutIdx + 1
               END DO
            END DO

         
            ! Relative wind speed
         CASE ( BldNd_VRel )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m%BEMT_y%Vrel(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Vrel(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
         
            ! Dynamic pressure
         CASE ( BldNd_DynP )
            if (p_AD%WakeMod /= WakeMod_FVW) then
                DO IdxBlade=1,p%BldNd_BladesOut
                   DO IdxNode=1,p%NumBlNds
                      y%WriteOutput( OutIdx )  = 0.5 * p%airDens * m%BEMT_y%Vrel(IdxNode,IdxBlade)**2
                      OutIdx = OutIdx + 1
                   END DO
                END DO
            else
                DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                   DO IdxNode=1,p%NumBlNds
                      y%WriteOutput( OutIdx )  = 0.5 * p%airDens *  m_AD%FVW%W(iW)%BN_Vrel(IdxNode)**2
                      OutIdx = OutIdx + 1
                   END DO
                END DO
            endif

            ! Reynolds number (in millions)
         CASE ( BldNd_Re )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = p%BEMT%chord(IdxNode,IdxBlade) * m%BEMT_y%Vrel(IdxNode,IdxBlade) / p%KinVisc / 1.0E6
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Re(IdxNode) / 1.0E6
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

            ! Mach number
         CASE ( BldNd_M )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m%BEMT_y%Vrel(IdxNode,IdxBlade) / p%SpdSound
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Vrel(IdxNode) / p%SpdSound
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
 
         
         
            ! Axial and tangential induced wind velocity
         CASE ( BldNd_Vindx )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds 
                     y%WriteOutput( OutIdx )  = - m%BEMT_u(Indx)%Vx(IdxNode,IdxBlade) * m%BEMT_y%axInduction( IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds 
                     y%WriteOutput( OutIdx )  = -m_AD%FVW%W(iW)%BN_UrelWind_s(1,IdxNode) * m_AD%FVW%W(iW)%BN_AxInd(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
                     
         CASE ( BldNd_Vindy )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds 
                     y%WriteOutput( OutIdx )  = m%BEMT_u(Indx)%Vy(IdxNode,IdxBlade) * m%BEMT_y%tanInduction(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds 
                     y%WriteOutput( OutIdx )  =  m_AD%FVW%W(iW)%BN_UrelWind_s(2,IdxNode) * m_AD%FVW%W(iW)%BN_TanInd(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
         

            ! Axial and tangential induction factors
         CASE ( BldNd_AxInd )         
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m%BEMT_y%axInduction(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_AxInd(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_TnInd )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%BEMT_y%tanInduction(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_TanInd(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
         
                  
            ! AoA, pitch+twist angle, inflow angle, and curvature angle
         CASE ( BldNd_Alpha )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = Rad2M180to180Deg( m%BEMT_y%phi(IdxNode,IdxBlade) - m%BEMT_u(Indx)%theta(IdxNode,IdxBlade) )
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_alpha(IdxNode)*R2D
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_Theta )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%BEMT_u(Indx)%theta(IdxNode,IdxBlade)*R2D
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%PitchAndTwist(IdxNode)*R2D
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
         
         CASE ( BldNd_Phi )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%BEMT_y%phi(IdxNode,IdxBlade)*R2D                                            
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  =m_AD%FVW%W(iW)%BN_phi(IdxNode)*R2D
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
         
         CASE ( BldNd_Curve )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%Curve(IdxNode,IdxBlade)*R2D                                            
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
!NOT available in FVW yet
                     y%WriteOutput( OutIdx ) = 0.0_ReKi 
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
         
         
                        ! Lift force, drag force, pitching moment coefficients
         CASE ( BldNd_Cl )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%BEMT_y%Cl(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Cl(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
         
         CASE ( BldNd_Cd )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%BEMT_y%Cd(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Cd(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_Cm )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%BEMT_y%Cm(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Cm(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

            ! Normal force (to plane), tangential force (to plane) coefficients
         CASE ( BldNd_Cx )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%BEMT_y%Cx(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Cx(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_Cy )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m%BEMT_y%Cy(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Cy(IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

            ! Normal force (to chord), and tangential force (to chord) coefficients
         CASE ( BldNd_Cn )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds   
                     ct=cos(m%BEMT_u(Indx)%theta(IdxNode,IdxBlade))
                     st=sin(m%BEMT_u(Indx)%theta(IdxNode,IdxBlade))               
                     y%WriteOutput( OutIdx )  = m%BEMT_y%Cx(IdxNode,IdxBlade)*ct + m%BEMT_y%Cy(IdxNode,IdxBlade)*st
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     ct=cos(m_AD%FVW%W(iW)%PitchAndTwist(IdxNode))    ! cos(theta)
                     st=sin(m_AD%FVW%W(iW)%PitchAndTwist(IdxNode))    ! sin(theta)
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_Cx(IdxNode)*ct + m_AD%FVW%W(iW)%BN_Cy(IdxNode)*st
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_Ct )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds   
                     ct=cos(m%BEMT_u(Indx)%theta(IdxNode,IdxBlade))
                     st=sin(m%BEMT_u(Indx)%theta(IdxNode,IdxBlade))               
                     y%WriteOutput( OutIdx )  = -m%BEMT_y%Cx(IdxNode,IdxBlade)*st + m%BEMT_y%Cy(IdxNode,IdxBlade)*ct
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     ct=cos(m_AD%FVW%W(iW)%PitchAndTwist(IdxNode))    ! cos(theta)
                     st=sin(m_AD%FVW%W(iW)%PitchAndTwist(IdxNode))    ! sin(theta)
                     y%WriteOutput( OutIdx )  = -m_AD%FVW%W(iW)%BN_Cx(IdxNode)*st + m_AD%FVW%W(iW)%BN_Cy(IdxNode)*ct
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif


               ! Lift force, drag force, pitching moment
         CASE ( BldNd_Fl )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds   
                     cp=cos(m%BEMT_y%phi(IdxNode,IdxBlade))
                     sp=sin(m%BEMT_y%phi(IdxNode,IdxBlade))
                     y%WriteOutput( OutIdx )  = m%X(IdxNode,IdxBlade)*cp - m%Y(IdxNode,IdxBlade)*sp
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     cp=cos(m_AD%FVW%W(iW)%BN_phi(IdxNode))
                     sp=sin(m_AD%FVW%W(iW)%BN_phi(IdxNode))
                     y%WriteOutput( OutIdx )  = m%X(IdxNode,IdxBlade)*cp - m%Y(IdxNode,IdxBlade)*sp
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_Fd )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds   
                     cp=cos(m%BEMT_y%phi(IdxNode,IdxBlade))
                     sp=sin(m%BEMT_y%phi(IdxNode,IdxBlade))
                     y%WriteOutput( OutIdx )  = m%X(IdxNode,IdxBlade)*sp + m%Y(IdxNode,IdxBlade)*cp
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds                   
                     cp=cos(m_AD%FVW%W(iW)%BN_phi(IdxNode))
                     sp=sin(m_AD%FVW%W(iW)%BN_phi(IdxNode))
                     y%WriteOutput( OutIdx )  = m%X(IdxNode,IdxBlade)*sp + m%Y(IdxNode,IdxBlade)*cp
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_Mm )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds   
                  y%WriteOutput( OutIdx )  = m%M(IdxNode,IdxBlade)
                  OutIdx = OutIdx + 1
               END DO
            END DO 

            ! Normal force (to plane), tangential force (to plane)
         CASE ( BldNd_Fx )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds   
                  y%WriteOutput( OutIdx )  = m%X(IdxNode,IdxBlade)
                  OutIdx = OutIdx + 1
               END DO
            END DO 

         CASE ( BldNd_Fy )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,p%NumBlNds   
                  y%WriteOutput( OutIdx )  = -m%Y(IdxNode,IdxBlade)
                  OutIdx = OutIdx + 1
               END DO
            END DO 

            ! Normal force (to chord), and tangential force (to chord) per unit length
         CASE ( BldNd_Fn )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds   
                     ct=cos(m%BEMT_u(Indx)%theta(IdxNode,IdxBlade))
                     st=sin(m%BEMT_u(Indx)%theta(IdxNode,IdxBlade))
                     y%WriteOutput( OutIdx )  = m%X(IdxNode,IdxBlade)*ct - m%Y(IdxNode,IdxBlade)*st
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds   
                     ct=cos(m_AD%FVW%W(iW)%PitchAndTwist(IdxNode))    ! cos(theta)
                     st=sin(m_AD%FVW%W(iW)%PitchAndTwist(IdxNode))    ! sin(theta)
                     y%WriteOutput( OutIdx )  = m%X(IdxNode,IdxBlade)*ct - m%Y(IdxNode,IdxBlade)*st
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            endif
         
         CASE ( BldNd_Ft )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds   
                     ct=cos(m%BEMT_u(Indx)%theta(IdxNode,IdxBlade))
                     st=sin(m%BEMT_u(Indx)%theta(IdxNode,IdxBlade))
                     y%WriteOutput( OutIdx )  = -m%X(IdxNode,IdxBlade)*st - m%Y(IdxNode,IdxBlade)*ct
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds   
                     ct=cos(m_AD%FVW%W(iW)%PitchAndTwist(IdxNode))    ! cos(theta)
                     st=sin(m_AD%FVW%W(iW)%PitchAndTwist(IdxNode))    ! sin(theta)
                     y%WriteOutput( OutIdx )  = -m%X(IdxNode,IdxBlade)*st - m%Y(IdxNode,IdxBlade)*ct
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            endif

                        ! Tower clearance (requires tower influence calculation):
         CASE ( BldNd_Clrnc )
            if (.not. allocated(m%TwrClrnc)) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds   
                     y%WriteOutput( OutIdx )  = 0.0_ReKi
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m%TwrClrnc(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            end if
         
         
         CASE ( BldNd_Vx )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m%BEMT_u(Indx)%Vx(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else 
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m_AD%FVW%W(iW)%BN_UrelWind_s(1,IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_Vy )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m%BEMT_u(Indx)%Vy(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else 
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  =  m_AD%FVW%W(iW)%BN_UrelWind_s(2,IdxNode)
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif
                     
         CASE ( BldNd_GeomPhi )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               if (allocated(OtherState%BEMT%ValidPhi)) then
                  DO IdxBlade=1,p%BldNd_BladesOut
                     DO IdxNode=1,p%NumBlNds 
                        if (OtherState%BEMT%ValidPhi(IdxNode,IdxBlade)) then
                           y%WriteOutput( OutIdx ) = 1.0_ReKi - m%BEMT%BEM_weight
                        else
                           y%WriteOutput( OutIdx ) = 1.0_ReKi
                        end if
                        OutIdx = OutIdx + 1
                     END DO
                  END DO 
               else
                  DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                     DO IdxNode=1,p%NumBlNds   
                        y%WriteOutput( OutIdx )  = 1.0_ReKi
                        OutIdx = OutIdx + 1
                     END DO
                  END DO 
               end if
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds   
                     y%WriteOutput( OutIdx )  = 0.0_ReKi ! Not valid for FVW
                     OutIdx = OutIdx + 1
                  END DO
               END DO 
            endif

         CASE ( BldNd_chi )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
                     y%WriteOutput( OutIdx )  = m%BEMT_y%chi(IdxNode,IdxBlade)*R2D
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            else 
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,p%NumBlNds
!NOT available in FVW yet
                     y%WriteOutput( OutIdx ) = 0.0_ReKi
                     OutIdx = OutIdx + 1
                  END DO
               END DO
            endif

         CASE ( BldNd_UA_Flag )
            IF (p_AD%UA_Flag) THEN
               if (p_AD%WakeMod /= WakeMod_FVW) then
                  DO IdxBlade=1,p%BldNd_BladesOut
                     DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                        y%WriteOutput( OutIdx ) = m%BEMT%UA%weight(IdxNode, IdxBlade)
                        OutIdx = OutIdx + 1
                     ENDDO
                  ENDDO
               else
                  DO IdxBlade=1,p%BldNd_BladesOut
                     iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                     DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                        y%WriteOutput( OutIdx ) = m_AD%FVW%W(iW)%m_UA%weight(IdxNode, 1)
                        OutIdx = OutIdx + 1
                     ENDDO
                  ENDDO
               end if
            ELSE
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = 0.0_ReKi
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            END IF
               
      
            CASE ( BldNd_UA_x1, BldNd_UA_x2, BldNd_UA_x3, BldNd_UA_x4, BldNd_UA_x5 )
               IF (allocated(x%BEMT%UA%element)) THEN
                     SELECT CASE( p%BldNd_OutParam(IdxChan)%Indx )
                     CASE (BldNd_UA_x1)
                        compIndx = 1
                     CASE (BldNd_UA_x2)
                        compIndx = 2
                     CASE (BldNd_UA_x3)
                        compIndx = 3
                     CASE (BldNd_UA_x4)
                        compIndx = 4
                     CASE (BldNd_UA_x5)
                        compIndx = 5
                     END SELECT
            
                     !if (p_AD%WakeMod /= WakeMod_FVW) then
                        DO IdxBlade=1,p%BldNd_BladesOut
                           DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                              y%WriteOutput( OutIdx ) = x%BEMT%UA%element(IdxNode, IdxBlade)%x(compIndx)
                              OutIdx = OutIdx + 1
                           ENDDO
                        ENDDO
                     !else
                     !   DO IdxBlade=1,p%BldNd_BladesOut
                     !      iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                     !      DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     !         y%WriteOutput( OutIdx ) = x_AD%FVW%UA(iW)%element(IdxNode, IdxBlade)%x(compIndx)
                     !         OutIdx = OutIdx + 1
                     !      ENDDO
                     !   ENDDO
                     !end if
                        
               ELSE
                  DO IdxBlade=1,p%BldNd_BladesOut
                     DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                        y%WriteOutput( OutIdx ) = 0.0_ReKi
                        OutIdx = OutIdx + 1
                     ENDDO
                  ENDDO
               END IF


         CASE ( BldNd_Debug1 )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                  y%WriteOutput( OutIdx )  = 0.0_ReKi ! something here
                  OutIdx = OutIdx + 1
               END DO
            END DO 

         CASE ( BldNd_Debug2 )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                  y%WriteOutput( OutIdx )  = 0.0_ReKi ! something here
                  OutIdx = OutIdx + 1
               END DO
            END DO 

         CASE ( BldNd_Debug3 )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                  y%WriteOutput( OutIdx )  = 0.0_ReKi ! something here
                  OutIdx = OutIdx + 1
               END DO
            END DO 
            
            
            ! CpMin
         CASE ( BldNd_CpMin )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = m%BEMT_y%Cpmin(IdxNode,IdxBlade)
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
!NOT available in FVW yet
                     y%WriteOutput( OutIdx ) = 0.0_ReKi
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            endif

            ! Cavitation
         CASE ( BldNd_SgCav )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                  y%WriteOutput( OutIdx ) = m%SigmaCavit(IdxNode,IdxBlade)
                  OutIdx = OutIdx + 1
               ENDDO
            ENDDO

         CASE ( BldNd_SigCr )
            DO IdxBlade=1,p%BldNd_BladesOut
               DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                  y%WriteOutput( OutIdx ) = m%SigmaCavitCrit(IdxNode,IdxBlade)
                  OutIdx = OutIdx + 1
               ENDDO
            ENDDO

            ! circulation on blade
         CASE ( BldNd_Gam )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = 0.5_ReKi * p%BEMT%chord(IdxNode,IdxBlade) * m%BEMT_y%Vrel(IdxNode,IdxBlade) * m%BEMT_y%Cl(IdxNode,IdxBlade) ! "Gam" [m^2/s]
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = 0.5_ReKi * p_AD%FVW%W(iW)%chord_LL(IdxNode) * m_AD%FVW%W(iW)%BN_Vrel(IdxNode) * m_AD%FVW%W(iW)%BN_Cl(IdxNode) ! "Gam" [m^2/s]
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            endif


            !================================================
            ! Static portion of Cl, Cd, Cm (ignoring unsteady effects)
            ! Cl_Static
         CASE ( BldNd_Cl_Static )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
!NOT available in BEMT/DBEMT yet
                     y%WriteOutput( OutIdx ) = 0.0_ReKi
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = m_AD%FVW%W(iW)%BN_Cl_Static(IdxNode)
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            endif

            ! Cd_Static
         CASE ( BldNd_Cd_Static )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
!NOT available in BEMT/DBEMT yet
                     y%WriteOutput( OutIdx ) = 0.0_ReKi
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = m_AD%FVW%W(iW)%BN_Cd_Static(IdxNode)
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            endif

            ! Cm_Static
         CASE ( BldNd_Cm_Static )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
!NOT available in BEMT/DBEMT yet
                     y%WriteOutput( OutIdx ) = 0.0_ReKi
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = m_AD%FVW%W(iW)%BN_Cm_Static(IdxNode)
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            endif

 

            !================================================
            ! Inductions in polar rotating hub coordinates
            ! Axial induction, polar rotating hub coordinates
         CASE ( BldNd_Uin )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     Vind_s = (/ -m%BEMT_u(Indx)%Vx(IdxNode,IdxBlade)*m%BEMT_y%axInduction(IdxNode,IdxBlade), m%BEMT_u(Indx)%Vy(IdxNode,IdxBlade)*m%BEMT_y%tanInduction(IdxNode,IdxBlade), 0.0_ReKi /)
                     Vind_g = matmul(Vind_s, m%WithoutSweepPitchTwist(:,:,IdxNode,IdxBlade))
                     y%WriteOutput( OutIdx ) = dot_product(M_pg(1,1:3,IdxBlade), Vind_g(1:3) ) ! Uihn, hub normal
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = dot_product(M_pg(1,1:3,IdxBlade), m_AD%FVW_y%W(iW)%Vind(1:3,IdxNode) ) ! Uihn, hub normal
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            endif

            ! Tangential induction, polar rotating hub coordinates
         CASE ( BldNd_Uit )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     Vind_s = (/ -m%BEMT_u(Indx)%Vx(IdxNode,IdxBlade)*m%BEMT_y%axInduction(IdxNode,IdxBlade), m%BEMT_u(Indx)%Vy(IdxNode,IdxBlade)*m%BEMT_y%tanInduction(IdxNode,IdxBlade), 0.0_ReKi /)
                     Vind_g = matmul(Vind_s, m%WithoutSweepPitchTwist(:,:,IdxNode,IdxBlade))
                     y%WriteOutput( OutIdx ) = dot_product(M_pg(2,1:3,IdxBlade), Vind_g(1:3) ) ! Uiht, hub tangential
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = dot_product(M_pg(2,1:3,IdxBlade), m_AD%FVW_y%W(iW)%Vind(1:3,IdxNode) ) ! Uiht, hub tangential
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            endif
 
            ! Radial induction, polar rotating hub coordinates
         CASE ( BldNd_Uir )
            if (p_AD%WakeMod /= WakeMod_FVW) then
               DO IdxBlade=1,p%BldNd_BladesOut
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     Vind_s = (/ -m%BEMT_u(Indx)%Vx(IdxNode,IdxBlade)*m%BEMT_y%axInduction(IdxNode,IdxBlade), m%BEMT_u(Indx)%Vy(IdxNode,IdxBlade)*m%BEMT_y%tanInduction(IdxNode,IdxBlade), 0.0_ReKi /)
                     Vind_g = matmul(Vind_s, m%WithoutSweepPitchTwist(:,:,IdxNode,IdxBlade))
                     y%WriteOutput( OutIdx ) = dot_product(M_pg(3,1:3,IdxBlade), Vind_g(1:3) ) ! Uihr, hub radial
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            else
               DO IdxBlade=1,p%BldNd_BladesOut
                  iW = p_AD%FVW%Bld2Wings(iRot, IdxBlade)
                  DO IdxNode=1,u%BladeMotion(IdxBlade)%NNodes
                     y%WriteOutput( OutIdx ) = dot_product(M_pg(3,1:3,IdxBlade), m_AD%FVW_y%W(iW)%Vind(1:3,IdxNode) ) ! Uihr, hub radial
                     OutIdx = OutIdx + 1
                  ENDDO
               ENDDO
            endif
 

         END SELECT
         
      END DO ! each channel
      
                    
END SUBROUTINE Calc_WriteAllBldNdOutput


!----------------------------------------------------------------------------------------------------------------------------------
!> This routine validates and sets the parameters for the nodal outputs.
SUBROUTINE AllBldNdOuts_SetParameters( InputFileData, p, p_AD, ErrStat, ErrMsg )
!..................................................................................................................................


      ! Passed variables:

   TYPE(AD_InputFile),           INTENT(IN   )  :: InputFileData    !< Data stored in the module's input file
   TYPE(RotParameterType),       INTENT(INOUT)  :: p                !< Parameters
   TYPE(AD_ParameterType),       INTENT(INOUT)  :: p_AD             !< Parameters
   INTEGER(IntKi),               INTENT(  OUT)  :: ErrStat          !< Error status of the operation
   CHARACTER(*),                 INTENT(  OUT)  :: ErrMsg           !< Error message if ErrStat /= ErrID_None

   
      ! local variables
   character(*), parameter                  :: RoutineName = 'AllBldNdOuts_SetParameters'
   
   ErrStat = ErrID_None
   ErrMsg  = ""


      ! Check if the requested blades exist
   IF ( (InputFileData%BldNd_BladesOut < 0_IntKi) ) then
      p%BldNd_BladesOut = 0_IntKi
   ELSE IF ((InputFileData%BldNd_BladesOut > p%NumBlades) ) THEN
      CALL SetErrStat( ErrID_Warn, " Number of blades to output data at all blade nodes (BldNd_BladesOut) must be less than "//TRIM(Num2LStr(p%NumBlades))//".", ErrStat, ErrMsg, RoutineName)
      p%BldNd_BladesOut = p%NumBlades ! NOTE: we are forgiving and plateau to numBlades
   ELSE
      p%BldNd_BladesOut = InputFileData%BldNd_BladesOut
   ENDIF


      ! Check if the requested blade nodes are valid
   ! InputFileData%BldNd_BlOutNd


      ! Set the parameter to store number of requested Blade Node output sets
   p%BldNd_NumOuts = InputFileData%BldNd_NumOuts

      ! Set the total number of outputs ( requested channel groups * number requested nodes * number requested blades )
   p%BldNd_TotNumOuts = p%BldNd_NumOuts*p%NumBlNds*p%BldNd_BladesOut    ! p%BldNd_NumOuts * size(p%BldNd_BlOutNd) * size(p%BldNd_BladesOut)
   
!      ! Check if the blade node array to output is valid: p%BldNd_BlOutNd 
!      ! TODO: this value is not read in by the input file reading yet, so setting to all blade nodes
!      !        -- check if list handed in is of nodes that exist (not sure this is ever checked)
!      !        -- copy values over
!
!      ! Temporary workaround here:
!   ALLOCATE ( p%BldNd_BlOutNd(1:p%NumBlNds) , STAT=ErrStat2 )
!   IF ( ErrStat2 /= 0_IntKi )  THEN
!      CALL SetErrStat( ErrID_Fatal,"Error allocating memory for the AeroDyn OutParam array.", ErrStat, ErrMsg, RoutineName )
!      RETURN
!   ENDIF
!   DO I=1,p%NumBlNds          ! put all nodes in the list
!      p%BldNd_BlOutNd(i) = i
!   ENDDO
   

!      ! Check if the requested blades are actually in use: 
!      ! TODO: this value is not read in by the input file reading yet, so setting to all blades
!      !        -- check if list handed in is of blades that exist (not sure this is ever checked)
!      !        -- copy values over
!   ALLOCATE ( p%BldNd_BladesOut(1:p%NumBlades), STAT=ErrStat2 )
!   IF ( ErrStat2 /= 0_IntKi )  THEN
!      CALL SetErrStat( ErrID_Fatal,"Error allocating memory for the AeroDyn OutParam array.", ErrStat, ErrMsg, RoutineName )
!      RETURN
!   ENDIF
!   DO I=1,p%NumBlades        ! put all blades in the list
!      p%BldNd_BladesOut(i) = i
!   ENDDO
   
   if (p%BldNd_TotNumOuts > 0) then
      call BldNdOuts_SetOutParam(InputFileData%BldNd_OutList, p, p_AD, ErrStat, ErrMsg ) ! requires: p%NumOuts, p%numBlades, p%NumBlNds, p%NumTwrNds; sets: p%BldNdOutParam.
         if (ErrStat >= AbortErrLev) return
   end if



END SUBROUTINE AllBldNdOuts_SetParameters  
 

!**********************************************************************************************************************************
! NOTE: The following lines of code were generated by a Matlab script called "Write_ChckOutLst.m"
!      using the parameters listed in the "OutListParameters.xlsx" Excel file. Any changes to these 
!      lines should be modified in the Matlab script and/or Excel worksheet as necessary. 
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine checks to see if any requested output channel names (stored in the OutList(:)) are invalid. It returns a 
!! warning if any of the channels are not available outputs from the module.
!!  It assigns the settings for OutParam(:) (i.e, the index, name, and units of the output channels, WriteOutput(:)).
!!  the sign is set to 0 if the channel is invalid.
!! It sets assumes the value p%NumOuts has been set before this routine has been called, and it sets the values of p%OutParam here.
!! 
!! This routine was generated by Write_ChckOutLst.m using the parameters listed in OutListParameters.xlsx at 02-Mar-2022 11:12:19.
SUBROUTINE BldNdOuts_SetOutParam(BldNd_OutList, p, p_AD, ErrStat, ErrMsg )
!..................................................................................................................................

   IMPLICIT                        NONE

      ! Passed variables

   CHARACTER(ChanLen),        INTENT(IN)     :: BldNd_OutList(:)                        !< The list out user-requested outputs
   TYPE(RotParameterType),    INTENT(INOUT)  :: p                                 !< The rotor parameters
   TYPE(AD_ParameterType),    INTENT(INOUT)  :: p_AD                              !< The module parameters
   INTEGER(IntKi),            INTENT(OUT)    :: ErrStat                           !< The error status code
   CHARACTER(*),              INTENT(OUT)    :: ErrMsg                            !< The error message, if an error occurred

      ! Local variables

   INTEGER                      :: ErrStat2                                        ! temporary (local) error status
   INTEGER                      :: I                                               ! Generic loop-counting index
   INTEGER                      :: J                                               ! Generic loop-counting index
   INTEGER                      :: INDX                                            ! Index for valid arrays

   LOGICAL                      :: InvalidOutput(1:BldNd_MaxOutPts)                      ! This array determines if the output channel is valid for this configuration
   CHARACTER(ChanLen)           :: OutListTmp                                      ! A string to temporarily hold OutList(I)
   CHARACTER(*), PARAMETER      :: RoutineName = "BldNdOuts_SetOutParam"

   CHARACTER(OutStrLenM1), PARAMETER  :: ValidParamAry(62) =  (/  &   ! This lists the names of the allowed parameters, which must be sorted alphabetically
                               "ALPHA    ","AXIND    ","CD       ","CD_STATIC","CHI      ","CL       ","CLRNC    ","CL_STATIC", &
                               "CM       ","CM_STATIC","CN       ","CPMIN    ","CT       ","CURVE    ","CX       ","CY       ", &
                               "DEBUG1   ","DEBUG2   ","DEBUG3   ","DYNP     ","FD       ","FL       ","FN       ","FT       ", &
                               "FX       ","FY       ","GAM      ","GEOMPHI  ","M        ","MM       ","PHI      ","RE       ", &
                               "SGCAV    ","SIGCR    ","STVX     ","STVY     ","STVZ     ","THETA    ","TNIND    ","UA_FLAG  ", &
                               "UA_X1    ","UA_X2    ","UA_X3    ","UA_X4    ","UA_X5    ","UIN      ","UIR      ","UIT      ", &
                               "VDISX    ","VDISY    ","VDISZ    ","VINDX    ","VINDY    ","VREL     ","VUNDX    ","VUNDXI   ", &
                               "VUNDY    ","VUNDYI   ","VUNDZ    ","VUNDZI   ","VX       ","VY       "/)
   INTEGER(IntKi), PARAMETER :: ParamIndxAry(62) =  (/ &                            ! This lists the index into AllOuts(:) of the allowed parameters ValidParamAry(:)
                                    BldNd_Alpha ,     BldNd_AxInd ,        BldNd_Cd , BldNd_Cd_Static ,       BldNd_Chi ,        BldNd_Cl ,     BldNd_Clrnc , BldNd_Cl_Static , &
                                       BldNd_Cm , BldNd_Cm_Static ,        BldNd_Cn ,     BldNd_CpMin ,        BldNd_Ct ,     BldNd_Curve ,        BldNd_Cx ,        BldNd_Cy , &
                                   BldNd_Debug1 ,    BldNd_Debug2 ,    BldNd_Debug3 ,      BldNd_DynP ,        BldNd_Fd ,        BldNd_Fl ,        BldNd_Fn ,        BldNd_Ft , &
                                       BldNd_Fx ,        BldNd_Fy ,       BldNd_Gam ,   BldNd_GeomPhi ,         BldNd_M ,        BldNd_Mm ,       BldNd_Phi ,        BldNd_Re , &
                                    BldNd_SgCav ,     BldNd_SigCr ,      BldNd_STVx ,      BldNd_STVy ,      BldNd_STVz ,     BldNd_Theta ,     BldNd_TnInd ,   BldNd_UA_Flag , &
                                    BldNd_UA_x1 ,     BldNd_UA_x2 ,     BldNd_UA_x3 ,     BldNd_UA_x4 ,     BldNd_UA_x5 ,       BldNd_Uin ,       BldNd_Uir ,       BldNd_Uit , &
                                    BldNd_VDisx ,     BldNd_VDisy ,     BldNd_VDisz ,     BldNd_Vindx ,     BldNd_Vindy ,      BldNd_VRel ,     BldNd_VUndx ,    BldNd_Vundxi , &
                                    BldNd_VUndy ,    BldNd_Vundyi ,     BldNd_VUndz ,    BldNd_Vundzi ,        BldNd_Vx ,        BldNd_Vy /)
   CHARACTER(ChanLen), PARAMETER :: ParamUnitsAry(62) =  (/  &  ! This lists the units corresponding to the allowed parameters
                               "(deg)  ","(-)    ","(-)    ","(-)    ","(deg)  ","(-)    ","(m)    ","(-)    ", &
                               "(-)    ","(-)    ","(-)    ","(-)    ","(-)    ","(deg)  ","(-)    ","(-)    ", &
                               "(-)    ","(-)    ","(-)    ","(Pa)   ","(N/m)  ","(N/m)  ","(N/m)  ","(N/m)  ", &
                               "(N/m)  ","(N/m)  ","(m^2/s)","(1/0)  ","(-)    ","(N-m/m)","(deg)  ","(-)    ", &
                               "(-)    ","(-)    ","(m/s)  ","(m/s)  ","(m/s)  ","(deg)  ","(-)    ","(-)    ", &
                               "(rad)  ","(rad)  ","(-)    ","(-)    ","(-)    ","(m/s)  ","(m/s)  ","(m/s)  ", &
                               "(m/s)  ","(m/s)  ","(m/s)  ","(m/s)  ","(m/s)  ","(m/s)  ","(m/s)  ","(m/s)  ", &
                               "(m/s)  ","(m/s)  ","(m/s)  ","(m/s)  ","(m/s)  ","(m/s)  "/)


      ! Initialize values
   ErrStat = ErrID_None
   ErrMsg = ""
   InvalidOutput = .FALSE.


!   ..... Developer must add checking for invalid inputs here: .....

      ! The following are valid only for BEMT/DBEMT
   if (p_AD%WakeMod /= WakeMod_FVW) then
      InvalidOutput( BldNd_Cl_Static ) = .true.
      InvalidOutput( BldNd_Cd_Static ) = .true.
      InvalidOutput( BldNd_Cm_Static ) = .true.
   else
         ! The following are invalid for free vortex wake
      InvalidOutput( BldNd_Chi       ) = .true.
      InvalidOutput( BldNd_Curve     ) = .true.
      InvalidOutput( BldNd_CpMin     ) = .true.
      InvalidOutput( BldNd_GeomPhi   ) = .true.  ! applies only to BEM
   endif

   ! it's going to be very difficult to get the FVW states without rewriting a bunch of code
   if (.not. p_AD%UA_Flag .or. p_AD%WakeMod == WakeMod_FVW) then ! also invalid if AFAeroMod is not 4,5,6
      InvalidOutput( BldNd_UA_x1 ) = .true.
      InvalidOutput( BldNd_UA_x2 ) = .true.
      InvalidOutput( BldNd_UA_x3 ) = .true.
      InvalidOutput( BldNd_UA_x4 ) = .true.
      InvalidOutput( BldNd_UA_x5 ) = .true.
   end if
   
   
!   ................. End of validity checking .................


   !-------------------------------------------------------------------------------------------------
   ! Allocate and set index, name, and units for the output channels
   ! If a selected output channel is not available in this module, set error flag.
   !-------------------------------------------------------------------------------------------------

   ALLOCATE ( p%BldNd_OutParam(1:p%BldNd_NumOuts) , STAT=ErrStat2 )
   IF ( ErrStat2 /= 0_IntKi )  THEN
      CALL SetErrStat( ErrID_Fatal,"Error allocating memory for the AeroDyn BldNd_OutParam array.", ErrStat, ErrMsg, RoutineName )
      RETURN
   ENDIF


      ! Set index, name, and units for all of the output channels.
      ! If a selected output channel is not available by this module set ErrStat = ErrID_Warn.

   DO I = 1,p%BldNd_NumOuts

      p%BldNd_OutParam(I)%Name  = BldNd_OutList(I)
      OutListTmp          = BldNd_OutList(I)
      p%BldNd_OutParam(I)%SignM = 1   ! this won't be used

      CALL Conv2UC( OutListTmp )    ! Convert OutListTmp to upper case


      Indx = IndexCharAry( OutListTmp(1:OutStrLenM1), ValidParamAry )

      IF ( Indx > 0 ) THEN ! we found the channel name
         IF ( InvalidOutput( ParamIndxAry(Indx) ) ) THEN  ! but, it isn't valid for these settings
            p%BldNd_OutParam(I)%Indx  = 0                 ! pick any valid channel (I just picked "Time=0" here because it's universal)
            p%BldNd_OutParam(I)%Units = "INVALID"
            p%BldNd_OutParam(I)%SignM = 0
         ELSE
            p%BldNd_OutParam(I)%Indx  = ParamIndxAry(Indx)
            p%BldNd_OutParam(I)%Units = ParamUnitsAry(Indx) ! it's a valid output
         END IF
      ELSE ! this channel isn't valid
         p%BldNd_OutParam(I)%Indx  = 0                    ! pick any valid channel (I just picked "Time=0" here because it's universal)
         p%BldNd_OutParam(I)%Units = "INVALID"
         p%BldNd_OutParam(I)%SignM = 0                    ! multiply all results by zero

         CALL SetErrStat(ErrID_Fatal, TRIM(p%BldNd_OutParam(I)%Name)//" is not an available output channel.",ErrStat,ErrMsg,RoutineName)
      END IF

   END DO

   RETURN
END SUBROUTINE BldNdOuts_SetOutParam
!----------------------------------------------------------------------------------------------------------------------------------
!End of code generated by Matlab script
!**********************************************************************************************************************************
END MODULE AeroDyn_AllBldNdOuts_IO
