!**********************************************************************************************************************************
!> ## FAST_Farm
!! The FAST_Farm, FAST_Farm_Subs, and FAST_Farm_Types modules make up a driver for the multi-turbine FAST.Farm code. 
!! FAST_Farms_Types will be auto-generated by the FAST registry program, based on the variables specified in the
!! FAST_Farm_Registry.txt file.
!!
! ..................................................................................................................................
!! ## LICENSING
!! Copyright (C) 2017  Bonnie Jonkman, independent contributor
!! Copyright (C) 2017  National Renewable Energy Laboratory
!!
!!    This file is part of FAST_Farm.
!!
!! Licensed under the Apache License, Version 2.0 (the "License");
!! you may not use this file except in compliance with the License.
!! You may obtain a copy of the License at
!!
!!     http://www.apache.org/licenses/LICENSE-2.0
!!
!! Unless required by applicable law or agreed to in writing, software
!! distributed under the License is distributed on an "AS IS" BASIS,
!! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
!! See the License for the specific language governing permissions and
!! limitations under the License.
!**********************************************************************************************************************************
MODULE FAST_Farm_Subs

   USE FAST_Farm_Types
   USE NWTC_Library
   USE WakeDynamics
   USE AWAE
   USE FAST_Farm_IO
   USE FAST_Subs
   USE FASTWrapper
   USE SuperController
   
#ifdef _OPENMP
   USE OMP_LIB 
#endif


   IMPLICIT NONE


    
   integer, parameter :: maxOutputPoints = 9
   
   CONTAINS
 

   subroutine TrilinearInterpRegGrid(V, pt, dims, val)
   
      real(SiKi),     intent(in   ) :: V(:,0:,0:,0:)      !< The volume data being sampled
      real(ReKi),     intent(in   ) :: pt(3)           !< The point, in grid coordinates where we want to sample the data
      integer(IntKi), intent(in   ) :: dims(3)         !< The grid dimensions
      real(ReKi),     intent(  out) :: val(3)          !< The interpolated value of V at location, pt
   
      integer(IntKi) :: x0,x1,y0,y1,z0,z1, i
      real(ReKi) :: xd,yd,zd,c00(3),c01(3),c10(3),c11(3),c0(3),c1(3)
      REAL(ReKi)      :: N(8)           ! array for holding scaling factors for the interpolation algorithm
      REAL(ReKi)      :: u(8)           ! array for holding the corner values for the interpolation algorithm across a cubic volume
      real(ReKi)      :: val2(3)
      
      x0 = min(max(floor(pt(1)),0),dims(1)-1)
      x1 = x0 + 1
      if (x0 == (dims(1)-1)) x1 = x0  ! Handle case where x0 is the last index in the grid, in this case xd = 0.0, so the 2nd term in the interpolation will not contribute
      xd = 2.0_ReKi * (pt(1) - REAL(x0, ReKi)) - 1.0_ReKi

      y0 = min(max(floor(pt(2)),0),dims(2)-1)
      y1 = y0 + 1
      if (y0 == (dims(2)-1)) y1 = y0  ! Handle case where y0 is the last index in the grid, in this case yd = 0.0, so the 2nd term in the interpolation will not contribute
      yd = 2.0_ReKi * (pt(2) - REAL(y0, ReKi)) - 1.0_ReKi

      z0 = min(max(floor(pt(3)),0),dims(3)-1)
      z1 = z0 + 1
      if (z0 == (dims(3)-1)) z1 = z0  ! Handle case where z0 is the last index in the grid, in this case zd = 0.0, so the 2nd term in the interpolation will not contribute
      zd = 2.0_ReKi * (pt(3) - REAL(z0, ReKi)) - 1.0_ReKi
      
      !-------------------------------------------------------------------------------------------------
      ! Interpolate on the grid
      !-------------------------------------------------------------------------------------------------

      N(1)  = ( 1.0_ReKi + zd )*( 1.0_ReKi - yd )*( 1.0_ReKi - xd )
      N(2)  = ( 1.0_ReKi + zd )*( 1.0_ReKi + yd )*( 1.0_ReKi - xd )
      N(3)  = ( 1.0_ReKi - zd )*( 1.0_ReKi + yd )*( 1.0_ReKi - xd )
      N(4)  = ( 1.0_ReKi - zd )*( 1.0_ReKi - yd )*( 1.0_ReKi - xd )
      N(5)  = ( 1.0_ReKi + zd )*( 1.0_ReKi - yd )*( 1.0_ReKi + xd )
      N(6)  = ( 1.0_ReKi + zd )*( 1.0_ReKi + yd )*( 1.0_ReKi + xd )
      N(7)  = ( 1.0_ReKi - zd )*( 1.0_ReKi + yd )*( 1.0_ReKi + xd )
      N(8)  = ( 1.0_ReKi - zd )*( 1.0_ReKi - yd )*( 1.0_ReKi + xd )
      N     = N / real( size(N), ReKi )  ! normalize

      do i=1,3
         u(1)  = real(V( i, x0, y0, z1 ), ReKi)
         u(2)  = real(V( i, x0, y1, z1 ), ReKi)
         u(3)  = real(V( i, x0, y1, z0 ), ReKi)
         u(4)  = real(V( i, x0, y0, z0 ), ReKi)
         u(5)  = real(V( i, x1, y0, z1 ), ReKi)
         u(6)  = real(V( i, x1, y1, z1 ), ReKi)
         u(7)  = real(V( i, x1, y1, z0 ), ReKi)
         u(8)  = real(V( i, x1, y0, z0 ), ReKi)
            
         val(i)  =  SUM ( N * u ) 
      end do
      
      !
      !
      !xd = pt(1) - x0
      !yd = pt(2) - y0
      !zd = pt(3) - z0
      !c00 = V(:,x0,y0,z0)*(1.0_ReKi-xd) + V(:,x1,y0,z0)*xd
      !c01 = V(:,x0,y0,z1)*(1.0_ReKi-xd) + V(:,x1,y0,z1)*xd
      !c10 = V(:,x0,y1,z0)*(1.0_ReKi-xd) + V(:,x1,y1,z0)*xd
      !c11 = V(:,x0,y1,z1)*(1.0_ReKi-xd) + V(:,x1,y1,z1)*xd
      !
      !c0  = c00*(1.0_ReKi-yd) + c10*yd
      !c1  = c01*(1.0_ReKi-yd) + c11*yd
      !
      !val2 = c0 *(1.0_ReKi-zd) + c1 *zd
      !do i = 1,3
      !   if ( .not. EqualRealNos(val(i),val2(i)) ) then
      !      write(*,*) "Different inpolated wind values: "//trim(Num2LStr(val(1)))//", "//trim(Num2LStr(val(2)))//", "//trim(Num2LStr(val(3)))//", "//trim(Num2LStr(val2(1)))//", "//trim(Num2LStr(val2(2)))//", "//trim(Num2LStr(val2(3)))
      !      return
      !   end if
      !end do
   end subroutine TrilinearInterpRegGrid

   
!----------------------------------------------------------------------------------------------------------------------------------
!> Routine to call Init routine for each module. This routine sets all of the init input data for each module. The initialization algorithm is: \n
!!   -  Read-In Input File
!!   -  Check Inputs and Set Parameters
!!   -  In parallel:
!!      1.  CALL AWAE_Init
!!      2.  CALL_SC_Init
!!      3.  CALL WD_Init
!!   -  Transfer y_AWAE_Init to u_F_Init and CALL F_Init
!!   -  Open Output File
!!   -  n=0
!!   -  t=0
SUBROUTINE Farm_Initialize( farm, InputFile, ErrStat, ErrMsg )

   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                !< FAST.Farm data
      
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat             !< Error status of the operation
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg              !< Error message if ErrStat /= ErrID_None
   CHARACTER(*),             INTENT(IN   ) :: InputFile           !< A CHARACTER string containing the name of the primary FAST.Farm input file
   
   
   ! local variables 
   type(AWAE_InitInputType)                :: AWAE_InitInput
   type(AWAE_InitOutputType)               :: AWAE_InitOutput
   
   INTEGER(IntKi)                          :: ErrStat2   
   CHARACTER(ErrMsgLen)                    :: ErrMsg2
   TYPE(WD_InitInputType)                  :: WD_InitInput            ! init-input data for WakeDynamics module
   TYPE(SC_InitInputType)                  :: SC_InitInp              ! input-file data for SC module
   TYPE(SC_InitOutputType)                 :: SC_InitOut              ! Init output for SC module
   CHARACTER(*), PARAMETER                 :: RoutineName = 'Farm_Initialize'       
   CHARACTER(ChanLen)                      :: OutList(Farm_MaxOutPts) ! list of user-requested output channels
   INTEGER(IntKi)                          :: i
   !..........
   ErrStat = ErrID_None
   ErrMsg  = ""         
   AbortErrLev  = ErrID_Fatal                                 ! Until we read otherwise from the FAST input file, we abort only on FATAL errors
      
   
      ! ... Open and read input files, initialize global parameters. ...
      
   IF (LEN_TRIM(InputFile) == 0) THEN ! no input file was specified
      CALL SetErrStat( ErrID_Fatal, 'The required input file was not specified on the command line.', ErrStat, ErrMsg, RoutineName )

      CALL NWTC_DisplaySyntax( InputFile, 'FAST.Farm.exe' )
         
      RETURN
   END IF            
                        
      ! Determine the root name of the primary file (will be used for output files)
   CALL GetRoot( InputFile, farm%p%OutFileRoot )      
    
   DO i=1,NumFFModules
      farm%p%Module_Ver(i)%Date = 'unknown date'
      farm%p%Module_Ver(i)%Ver  = 'unknown version'
   END DO       
   farm%p%Module_Ver( ModuleFF_SC    )%Name = 'Super Controller'
   farm%p%Module_Ver( ModuleFF_FWrap )%Name = 'FAST Wrapper'
   farm%p%Module_Ver( ModuleFF_WD    )%Name = 'Wake Dynamics'
   farm%p%Module_Ver( ModuleFF_AWAE  )%Name = 'Ambient Wind and Array Effects'
   
   !...............................................................................................................................  
   ! step 1: read input file
   !...............................................................................................................................  
      
   call Farm_ReadPrimaryFile( InputFile, farm%p, WD_InitInput%InputFileData, AWAE_InitInput%InputFileData, SC_InitInp, OutList, ErrStat2, ErrMsg2 )
      CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF (ErrStat >= AbortErrLev) THEN
         CALL Cleanup()
         RETURN
      END IF

   !...............................................................................................................................  
   ! step 2: validate input & set parameters
   !...............................................................................................................................  
   call Farm_ValidateInput( farm%p, WD_InitInput%InputFileData, AWAE_InitInput%InputFileData, SC_InitInp, ErrStat2, ErrMsg2 )
      CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF (ErrStat >= AbortErrLev) THEN
         CALL Cleanup()
         RETURN
      END IF   
      
   
   farm%p%NOutTurb = min(farm%p%NumTurbines,9)  ! We only support output for the first 9 turbines, even if the farm has more than 9 
   
   farm%p%n_high_low = NINT( farm%p%dt_low / farm%p%dt_high )
            
         ! let's make sure the FAST.Farm DT_low is an exact multiple of dt_high 
         ! (i'm doing this outside of Farm_ValidateInput so we know that dt_low/=0 before computing n_high_low):
      IF ( .NOT. EqualRealNos( real(farm%p%DT_low,SiKi), real(farm%p%DT_high,SiKi) * farm%p%n_high_low )  ) THEN
         CALL SetErrStat(ErrID_Fatal, "DT_high ("//TRIM(Num2LStr(farm%p%dt_high))//" s) must be an integer divisor of DT_low (" &
                        //TRIM(Num2LStr(farm%p%dt_low))//" s).", ErrStat, ErrMsg, RoutineName ) 
      END IF
      
   farm%p%TChanLen = max( 10, int(log10(farm%p%TMax))+7 )
   farm%p%OutFmt_t = 'F'//trim(num2lstr( farm%p%TChanLen ))//'.4' ! 'F10.4'    
   farm%p%n_TMax  = FLOOR( ( farm%p%TMax / farm%p%DT_low ) ) + 1  ! We're going to go from step 0 to (n_TMax - 1)
                   ! [note that FAST uses the ceiling function, so it might think we're doing one more step than FAST.Farm; 
                   ! This difference will be a problem only if FAST thinks it's doing FEWER timesteps than FAST.Farm does.]
   
   IF ( WD_InitInput%InputFileData%NumPlanes > farm%p%n_TMax ) THEN
      WD_InitInput%InputFileData%NumPlanes = max( 2, min( WD_InitInput%InputFileData%NumPlanes, farm%p%n_TMax ) )
      call SetErrStat(ErrID_Warn, "For efficiency, NumPlanes has been reduced to the number of time steps ("//TRIM(Num2LStr(WD_InitInput%InputFileData%NumPlanes))//").", ErrStat, ErrMsg, RoutineName )
   ENDIF
   
   !...............................................................................................................................  
   ! step 3: initialize SC, AWAE, and WD (a, b, and c can be done in parallel)
   !...............................................................................................................................  
   
      !-------------------
      ! a. CALL AWAE_Init
   
   AWAE_InitInput%InputFileData%dr           = WD_InitInput%InputFileData%dr
   AWAE_InitInput%InputFileData%dt_low       = farm%p%dt_low 
   AWAE_InitInput%InputFileData%NumTurbines  = farm%p%NumTurbines
   AWAE_InitInput%InputFileData%NumRadii     = WD_InitInput%InputFileData%NumRadii
   AWAE_InitInput%InputFileData%NumPlanes    = WD_InitInput%InputFileData%NumPlanes
   AWAE_InitInput%InputFileData%WindFilePath = farm%p%WindFilePath
   AWAE_InitInput%n_high_low                 = farm%p%n_high_low
   AWAE_InitInput%NumDT                      = farm%p%n_TMax
   AWAE_InitInput%OutFileRoot                = farm%p%OutFileRoot
   call AWAE_Init( AWAE_InitInput, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, farm%AWAE%OtherSt, farm%AWAE%y, &
                   farm%AWAE%m, farm%p%DT_low, AWAE_InitOutput, ErrStat2, ErrMsg2 )
      CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF (ErrStat >= AbortErrLev) THEN
         CALL Cleanup()
         RETURN
      END IF
      
   farm%AWAE%IsInitialized = .true.

   farm%p%X0_Low = AWAE_InitOutput%X0_Low
   farm%p%Y0_low = AWAE_InitOutput%Y0_low
   farm%p%Z0_low = AWAE_InitOutput%Z0_low
   farm%p%nX_Low = AWAE_InitOutput%nX_Low
   farm%p%nY_low = AWAE_InitOutput%nY_low
   farm%p%nZ_low = AWAE_InitOutput%nZ_low
   farm%p%dX_low = AWAE_InitOutput%dX_low
   farm%p%dY_low = AWAE_InitOutput%dY_low
   farm%p%dZ_low = AWAE_InitOutput%dZ_low
   farm%p%Module_Ver( ModuleFF_AWAE  ) = AWAE_InitOutput%Ver
   
      !-------------------
      ! b. CALL SC_Init
   if ( farm%p%useSC ) then
      SC_InitInp%nTurbines = farm%p%NumTurbines
      call SC_Init(SC_InitInp, farm%SC%uInputs, farm%SC%p, farm%SC%x, farm%SC%xd, farm%SC%z, farm%SC%OtherState, &
                     farm%SC%y, farm%SC%m, farm%p%DT_low, SC_InitOut, ErrStat2, ErrMsg2)
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
            if (ErrStat >= AbortErrLev) then
               call Cleanup()
               return
            end if 
      farm%p%Module_Ver( ModuleFF_SC  ) = SC_InitOut%Ver
      farm%SC%IsInitialized = .true.
   else
      farm%SC%p%nInpGlobal = 0
      farm%SC%p%NumParamGlobal = 0
      farm%SC%p%NumParamTurbine = 0
      farm%SC%p%NumSC2CtrlGlob = 0
      farm%SC%p%NumSC2Ctrl = 0
      farm%SC%p%NumCtrl2SC = 0
      farm%SC%p%NumStatesGlobal = 0
      farm%SC%p%NumStatesTurbine = 0
      SC_InitOut%nInpGlobal = 0 
      SC_InitOut%NumSC2CtrlGlob = 0
      SC_InitOut%NumSC2Ctrl = 0
      SC_InitOut%NumCtrl2SC = 0
      allocate(farm%SC%y%fromscglob(0))
      allocate(farm%SC%y%fromsc(0))
   end if
   
      !-------------------
      ! c. initialize WD (one instance per turbine, each can be done in parallel, too)
      
   call Farm_InitWD( farm, WD_InitInput, ErrStat2, ErrMsg2 )
      CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF (ErrStat >= AbortErrLev) THEN
         CALL Cleanup()
         RETURN
      END IF   
      
      
   !...............................................................................................................................  
   ! step 4: initialize FAST (each instance of FAST can also be done in parallel)
   !...............................................................................................................................  

   CALL Farm_InitFAST( farm, WD_InitInput%InputFileData, AWAE_InitOutput, SC_InitOut, farm%SC%y, ErrStat2, ErrMsg2)
     CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF (ErrStat >= AbortErrLev) THEN
         CALL Cleanup()
         RETURN
      END IF   

   !...............................................................................................................................  
   ! step 5: Open output file (or set up output file handling)      
   !...............................................................................................................................  
   
      ! Set parameters for output channels:
   CALL Farm_SetOutParam(OutList, farm, ErrStat2, ErrMsg2 ) ! requires: p%NumOuts, sets: p%OutParam.
      CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF (ErrStat >= AbortErrLev) THEN
         CALL Cleanup()
         RETURN
      END IF  
      
   call Farm_InitOutput( farm, ErrStat2, ErrMsg2 )
      CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF (ErrStat >= AbortErrLev) THEN
         CALL Cleanup()
         RETURN
      END IF  

      ! Print the summary file if requested:
   IF (farm%p%SumPrint) THEN
      CALL Farm_PrintSum( farm, WD_InitInput%InputFileData, ErrStat2, ErrMsg2 )
         CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
   END IF
   
   !...............................................................................................................................
   ! Destroy initializion data
   !...............................................................................................................................      
   CALL Cleanup()
   
CONTAINS
   SUBROUTINE Cleanup()
   
      call WD_DestroyInitInput(WD_InitInput, ErrStat2, ErrMsg2)
      call AWAE_DestroyInitInput(AWAE_InitInput, ErrStat2, ErrMsg2)
      call AWAE_DestroyInitOutput(AWAE_InitOutput, ErrStat2, ErrMsg2)
         
   END SUBROUTINE Cleanup

END SUBROUTINE Farm_Initialize
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine reads in the primary FAST.Farm input file, does some validation, and places the values it reads in the
!!   parameter structure (p). It prints to an echo file if requested.
SUBROUTINE Farm_ReadPrimaryFile( InputFile, p, WD_InitInp, AWAE_InitInp, SC_InitInp, OutList, ErrStat, ErrMsg )


      ! Passed variables
   TYPE(Farm_ParameterType),       INTENT(INOUT) :: p                               !< The parameter data for the FAST (glue-code) simulation
   CHARACTER(*),                   INTENT(IN   ) :: InputFile                       !< Name of the file containing the primary input data
   TYPE(WD_InputFileType),         INTENT(  OUT) :: WD_InitInp                      !< input-file data for WakeDynamics module
   TYPE(AWAE_InputFileType),       INTENT(  OUT) :: AWAE_InitInp                    !< input-file data for AWAE module
   TYPE(SC_InitInputType),         INTENT(  OUT) :: SC_InitInp                      !< input-file data for SC module
   CHARACTER(ChanLen),             INTENT(  OUT) :: OutList(:)                      !< list of user-requested output channels
   INTEGER(IntKi),                 INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),                   INTENT(  OUT) :: ErrMsg                          !< Error message

      ! Local variables:
   REAL(DbKi)                    :: TmpTime                                   ! temporary variable to read SttsTime and ChkptTime before converting to #steps based on DT_low
   INTEGER(IntKi)                :: I                                         ! loop counter
   INTEGER(IntKi)                :: UnIn                                      ! Unit number for reading file
   INTEGER(IntKi)                :: UnEc                                      ! I/O unit for echo file. If > 0, file is open for writing.

   INTEGER(IntKi)                :: IOS                                       ! Temporary Error status
   INTEGER(IntKi)                :: OutFileFmt                                ! An integer that indicates what kind of tabular output should be generated (1=text, 2=binary, 3=both)
   INTEGER(IntKi)                :: NLinTimes                                 ! An integer that indicates how many times to linearize
   LOGICAL                       :: Echo                                      ! Determines if an echo file should be written
   LOGICAL                       :: TabDelim                                  ! Determines if text output should be delimited by tabs (true) or space (false)
   CHARACTER(1024)               :: PriPath                                   ! Path name of the primary file

   CHARACTER(10)                 :: AbortLevel                                ! String that indicates which error level should be used to abort the program: WARNING, SEVERE, or FATAL
   CHARACTER(30)                 :: Line                                      ! string for default entry in input file

   INTEGER(IntKi)                :: ErrStat2                                  ! Temporary Error status
   CHARACTER(ErrMsgLen)          :: ErrMsg2                                   ! Temporary Error message
   CHARACTER(*),   PARAMETER     :: RoutineName = 'Farm_ReadPrimaryFile'
   
   
      ! Initialize some variables:
   UnEc = -1
   Echo = .FALSE.                        ! Don't echo until we've read the "Echo" flag
   CALL GetPath( InputFile, PriPath )    ! Input files will be relative to the path where the primary input file is located.


      ! Get an available unit number for the file.

   CALL GetNewUnit( UnIn, ErrStat, ErrMsg )
   IF ( ErrStat >= AbortErrLev ) RETURN


      ! Open the Primary input file.

   CALL OpenFInpFile ( UnIn, InputFile, ErrStat2, ErrMsg2 )
      CALL SetErrStat( ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

   ! Read the lines up/including to the "Echo" simulation control variable
   ! If echo is FALSE, don't write these lines to the echo file.
   ! If Echo is TRUE, rewind and write on the second try.

   I = 1 !set the number of times we've read the file
   DO
   !-------------------------- HEADER ---------------------------------------------

      CALL ReadCom( UnIn, InputFile, 'File header: FAST.Farm Version (line 1)', ErrStat2, ErrMsg2, UnEc )
         CALL SetErrStat( ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if

      CALL ReadStr( UnIn, InputFile, p%FTitle, 'FTitle', 'File Header: File Description (line 2)', ErrStat2, ErrMsg2, UnEc )
         CALL SetErrStat( ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if


   !---------------------- SIMULATION CONTROL --------------------------------------
      CALL ReadCom( UnIn, InputFile, 'Section Header: Simulation Control', ErrStat2, ErrMsg2, UnEc )
         CALL SetErrStat( ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if


         ! Echo - Echo input data to <RootName>.ech (flag):
      CALL ReadVar( UnIn, InputFile, Echo, "Echo", "Echo input data to <RootName>.ech (flag)", ErrStat2, ErrMsg2, UnEc)
         CALL SetErrStat( ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if


      IF (.NOT. Echo .OR. I > 1) EXIT !exit this loop

         ! Otherwise, open the echo file, then rewind the input file and echo everything we've read

      I = I + 1         ! make sure we do this only once (increment counter that says how many times we've read this file)

      CALL OpenEcho ( UnEc, TRIM(p%OutFileRoot)//'.ech', ErrStat2, ErrMsg2, Farm_Ver )
         CALL SetErrStat( ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if

      IF ( UnEc > 0 )  WRITE (UnEc,'(/,A,/)')  'Data from '//TRIM(Farm_Ver%Name)//' primary input file "'//TRIM( InputFile )//'":'

      REWIND( UnIn, IOSTAT=ErrStat2 )
         IF (ErrStat2 /= 0_IntKi ) THEN
            CALL SetErrStat( ErrID_Fatal, 'Error rewinding file "'//TRIM(InputFile)//'".',ErrStat,ErrMsg,RoutineName)
            call cleanup()
            RETURN        
         END IF

   END DO

   CALL WrScr( ' Heading of the '//TRIM(Farm_Ver%Name)//' input file: ' )
   CALL WrScr( '   '//TRIM( p%FTitle ) )


      ! AbortLevel - Error level when simulation should abort:
   CALL ReadVar( UnIn, InputFile, AbortLevel, "AbortLevel", "Error level when simulation should abort (string)", &
                        ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! Let's set the abort level here.... knowing that everything before this aborted only on FATAL errors!
      CALL Conv2UC( AbortLevel ) !convert to upper case
      SELECT CASE( TRIM(AbortLevel) )
         CASE ( "WARNING" )
            AbortErrLev = ErrID_Warn
         CASE ( "SEVERE" )
            AbortErrLev = ErrID_Severe
         CASE ( "FATAL" )
            AbortErrLev = ErrID_Fatal
         CASE DEFAULT
            CALL SetErrStat( ErrID_Fatal, 'Invalid AbortLevel specified in FAST.Farm input file. '// &
                             'Valid entries are "WARNING", "SEVERE", or "FATAL".',ErrStat,ErrMsg,RoutineName)
            call cleanup()
            RETURN
      END SELECT


      ! TMax - Total run time (s):
   CALL ReadVar( UnIn, InputFile, p%TMax, "TMax", "Total run time (s)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
       ! UseSC - Use a super controller? (flag):
   CALL ReadVar( UnIn, InputFile, p%UseSC, "UseSC", "Use a super controller? (flag)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
       ! Mod_AmbWind - Ambient wind model (-) (switch) {1: high-fidelity precursor in VTK format, 2: one InflowWind module, 3: multiple InflowWind modules}:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%Mod_AmbWind, "Mod_AmbWind", "Ambient wind model (-) (switch) {1: high-fidelity precursor in VTK format, 2: one InflowWind module, 3: multiple InflowWind modules}", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
   !---------------------- SUPER CONTROLLER ------------------------------------------------------------------
   CALL ReadCom( UnIn, InputFile, 'Section Header: Super Controller', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! SC_FileName - Name/location of the dynamic library {.dll [Windows] or .so [Linux]} containing the Super Controller algorithms (quoated string):
   CALL ReadVar( UnIn, InputFile, p%SC_FileName, "SC_FileName", "Name/location of the dynamic library {.dll [Windows] or .so [Linux]} containing the Super Controller algorithms (quoated string)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   IF ( PathIsRelative( p%SC_FileName ) ) p%SC_FileName = TRIM(PriPath)//TRIM(p%SC_FileName)
   SC_InitInp%DLL_FileName =  p%SC_FileName
   
   !---------------------- AMBIENT WIND: PRECURSOR IN VTK FORMAT ---------------------------------------------
   CALL ReadCom( UnIn, InputFile, 'Section Header: Ambient Wind: Precursor in VTK Format', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! DT_low - Time step for low-resolution wind data input files; will be used as the global FAST.Farm time step (s) [>0.0]:
   CALL ReadVar( UnIn, InputFile, p%DT_low, "DT_Low-VTK", "Time step for low-resolution wind data input files; will be used as the global FAST.Farm time step (s) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! DT_high - Time step for high-resolution wind data input files (s) [>0.0]:
   CALL ReadVar( UnIn, InputFile, p%DT_high, "DT_High-VTK", "Time step for high-resolution wind data input files (s) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! WindFilePath - Path name of wind data files from ABLSolver precursor (string):
   CALL ReadVar( UnIn, InputFile, p%WindFilePath, "WindFilePath", "Path name of wind data files from ABLSolver precursor (string)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   IF ( PathIsRelative( p%WindFilePath ) ) p%WindFilePath = TRIM(PriPath)//TRIM(p%WindFilePath)
            
      ! ChkWndFiles - Check all the ambient wind files for data consistency? (flag):
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%ChkWndFiles, "ChkWndFiles", "Check all the ambient wind files for data consistency? (flag)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName) 
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
   !---------------------- AMBIENT WIND: INFLOWWIND MODULE ---------------------------------------------
   CALL ReadCom( UnIn, InputFile, 'Section Header: Ambient Wind: InflowWind Module', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! DT_low - Time step for low-resolution wind data input files; will be used as the global FAST.Farm time step (s) [>0.0]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%DT_low, "DT_Low", "Time step for low-resolution wind data input files; will be used as the global FAST.Farm time step (s) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   if ( AWAE_InitInp%Mod_AmbWind > 1 ) p%DT_low = AWAE_InitInp%DT_low
   
      ! DT_high - Time step for high-resolution wind data input files (s) [>0.0]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%DT_high, "DT_High", "Time step for high-resolution wind data input files (s) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   if ( AWAE_InitInp%Mod_AmbWind > 1 ) p%DT_high = AWAE_InitInp%DT_high
   
      ! NX_Low - Number of low-resolution spatial nodes in X direction for wind data interpolation (-) [>=2]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%nX_Low, "nX_Low", "Number of low-resolution spatial nodes in X direction for wind data interpolation (-) [>=2]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! NY_Low - Number of low-resolution spatial nodes in Y direction for wind data interpolation (-) [>=2]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%nY_Low, "nY_Low", "Number of low-resolution spatial nodes in Y direction for wind data interpolation (-) [>=2]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! NZ_Low - Number of low-resolution spatial nodes in Z direction for wind data interpolation (-) [>=2]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%nZ_Low, "nZ_Low", "Number of low-resolution spatial nodes in Z direction for wind data interpolation (-) [>=2]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! X0_Low - Origin of low-resolution spatial nodes in X direction for wind data interpolation (m):
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%X0_Low, "X0_Low", "Origin of low-resolution spatial nodes in X direction for wind data interpolation (m)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! Y0_Low - Origin of low-resolution spatial nodes in Y direction for wind data interpolation (m):
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%Y0_Low, "Y0_Low", "Origin of low-resolution spatial nodes in Y direction for wind data interpolation (m)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! Z0_Low - Origin of low-resolution spatial nodes in Z direction for wind data interpolation (m):
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%Z0_Low, "Z0_Low", "Origin of low-resolution spatial nodes in Z direction for wind data interpolation (m)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! dX_Low - Spacing of low-resolution spatial nodes in X direction for wind data interpolation (m) [>0.0]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%dX_Low, "dX_Low", "Spacing of low-resolution spatial nodes in X direction for wind data interpolation (m) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! dY_Low - Spacing of low-resolution spatial nodes in Y direction for wind data interpolation (m) [>0.0]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%dY_Low, "dY_Low", "Spacing of low-resolution spatial nodes in Y direction for wind data interpolation (m) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! dZ_Low - Spacing of low-resolution spatial nodes in Z direction for wind data interpolation (m) [>0.0]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%dZ_Low, "dZ_Low", "Spacing of low-resolution spatial nodes in Z direction for wind data interpolation (m) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! NX_High - Number of high-resolution spatial nodes in X direction for wind data interpolation (-) [>=2]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%nX_High, "nX_High", "Number of high-resolution spatial nodes in X direction for wind data interpolation (-) [>=2]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! NY_High - Number of high-resolution spatial nodes in Y direction for wind data interpolation (-) [>=2]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%nY_High, "nY_High", "Number of high-resolution spatial nodes in Y direction for wind data interpolation (-) [>=2]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! NZ_High - Number of high-resolution spatial nodes in Z direction for wind data interpolation (-) [>=2]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%nZ_High, "nZ_High", "Number of high-resolution spatial nodes in Z direction for wind data interpolation (-) [>=2]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! InflowFile - Name of file containing InflowWind module input parameters (quoted string):
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%InflowFile, "InflowFile", "Name of file containing InflowWind module input parameters (quoted string)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   IF ( PathIsRelative( AWAE_InitInp%InflowFile ) ) AWAE_InitInp%InflowFile = TRIM(PriPath)//TRIM(AWAE_InitInp%InflowFile)
   if ( AWAE_InitInp%Mod_AmbWind > 1 ) p%WindFilePath = AWAE_InitInp%InflowFile  ! For the summary file
   
   !---------------------- WIND TURBINES ---------------------------------------------
   CALL ReadCom( UnIn, InputFile, 'Section Header: Wind Turbines', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      
      
      ! NumTurbines - Number of wind turbines (-) [>=1]:
   CALL ReadVar( UnIn, InputFile, p%NumTurbines, "NumTurbines", "Number of wind turbines (-) [>=1]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
   CALL ReadCom( UnIn, InputFile, 'Section Header: WT column names', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   CALL ReadCom( UnIn, InputFile, 'Section Header: WT column units', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      
   call AllocAry( p%WT_Position, 3, p%NumTurbines, 'WT_Position',   ErrStat2, ErrMsg2);  CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   call AllocAry( p%WT_FASTInFile,  p%NumTurbines, 'WT_FASTInFile', ErrStat2, ErrMsg2);  CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   call AllocAry( AWAE_InitInp%WT_Position, 3, p%NumTurbines, 'AWAE_InitInp%WT_Position', ErrStat2, ErrMsg2);  CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      

   if ( AWAE_InitInp%Mod_AmbWind > 1 ) then   
         ! Using InflowWind
      call AllocAry(AWAE_InitInp%X0_high, p%NumTurbines, 'AWAE_InitInp%X0_high', ErrStat2, ErrMsg2)
         call SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      call AllocAry(AWAE_InitInp%Y0_high, p%NumTurbines, 'AWAE_InitInp%Y0_high', ErrStat2, ErrMsg2)
         call SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      call AllocAry(AWAE_InitInp%Z0_high, p%NumTurbines, 'AWAE_InitInp%Z0_high', ErrStat2, ErrMsg2)
         call SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      call AllocAry(AWAE_InitInp%dX_high, p%NumTurbines, 'AWAE_InitInp%dX_high', ErrStat2, ErrMsg2)
         call SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      call AllocAry(AWAE_InitInp%dY_high, p%NumTurbines, 'AWAE_InitInp%dY_high', ErrStat2, ErrMsg2)
         call SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      call AllocAry(AWAE_InitInp%dZ_high, p%NumTurbines, 'AWAE_InitInp%dZ_high', ErrStat2, ErrMsg2)
         call SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
   end if

      ! WT_Position (WT_X, WT_Y, WT_Z) and WT_FASTInFile
   do i=1,p%NumTurbines
      
      if ( AWAE_InitInp%Mod_AmbWind == 1 ) then 
         READ (UnIn, *, IOSTAT=IOS) p%WT_Position(:,i), p%WT_FASTInFile(i)
      else
         READ (UnIn, *, IOSTAT=IOS) p%WT_Position(:,i), p%WT_FASTInFile(i), AWAE_InitInp%X0_high(i), AWAE_InitInp%Y0_high(i), AWAE_InitInp%Z0_high(i), AWAE_InitInp%dX_high(i), AWAE_InitInp%dY_high(i), AWAE_InitInp%dZ_high(i)
      end if
      AWAE_InitInp%WT_Position(:,i) = p%WT_Position(:,i)
      
      CALL CheckIOS ( IOS, InputFile, 'Wind Turbine Columns', NumType, ErrStat2, ErrMsg2 )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      IF ( UnEc > 0 ) THEN 
         if ( AWAE_InitInp%Mod_AmbWind == 1 ) then 
            WRITE( UnEc, "(3(ES11.4e2,2X),'""',A,'""',T50,' - WT(',I5,')')" ) p%WT_Position(:,i), TRIM( p%WT_FASTInFile(i) ), I
         else
            WRITE( UnEc, "(3(ES11.4e2,2X),'""',A,'""',T50,6(ES11.4e2,2X),' - WT(',I5,')')" ) p%WT_Position(:,i), TRIM( p%WT_FASTInFile(i) ), AWAE_InitInp%X0_high(i), AWAE_InitInp%Y0_high(i), AWAE_InitInp%Z0_high(i), AWAE_InitInp%dX_high(i), AWAE_InitInp%dY_high(i), AWAE_InitInp%dZ_high(i), I
         end if
         
      END IF
      IF ( PathIsRelative( p%WT_FASTInFile(i) ) ) p%WT_FASTInFile(i) = TRIM(PriPath)//TRIM(p%WT_FASTInFile(i))
      
   end do
      
      
   !---------------------- WAKE DYNAMICS ---------------------------------------------
   CALL ReadCom( UnIn, InputFile, 'Section Header: Wake Dynamics', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      
      
      ! dr - Radial increment of radial finite-difference grid (m) [>0.0]:
   CALL ReadVar( UnIn, InputFile, WD_InitInp%dr, "dr", "Radial increment of radial finite-difference grid (m) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   
      ! NumRadii - Number of radii in the radial finite-difference grid (-) [>=2]:
   CALL ReadVar( UnIn, InputFile, WD_InitInp%NumRadii, "NumRadii", "Number of radii in the radial finite-difference grid (-) [>=2]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   
      ! NumPlanes - Number of wake planes (-) [>=2]:
   CALL ReadVar( UnIn, InputFile, WD_InitInp%NumPlanes, "NumPlanes", "Number of wake planes (-) [>=2]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! f_c - Cut-off (corner) frequency of the low-pass time-filter for the wake advection, deflection, and meandering model (Hz) [>0.0] or DEFAULT [DEFAULT=0.0007]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%f_c, "f_c", &
      "Cut-off (corner) frequency of the low-pass time-filter for the wake advection, deflection, and meandering model (Hz) [>0.0] or DEFAULT [DEFAULT=0.0007]", &
      0.0007_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! C_HWkDfl_O - Calibrated parameter in the correction for wake deflection defining the horizontal offset at the rotor (m) or DEFAULT [DEFAULT=0.0]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_HWkDfl_O, "C_HWkDfl_O", &
      "Calibrated parameter in the correction for wake deflection defining the horizontal offset at the rotor (m) or DEFAULT [DEFAULT=0.0]", &
      0.0_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! C_HWkDfl_OY - Calibrated parameter in the correction for wake deflection defining the horizontal offset at the rotor scaled with yaw error (m/deg) or DEFAULT [DEFAULT=0.3]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_HWkDfl_OY, "C_HWkDfl_OY", &
      "Calibrated parameter in the correction for wake deflection defining the horizontal offset at the rotor scaled with yaw error (m/deg) or DEFAULT [DEFAULT=0.3]", &
      0.3_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   WD_InitInp%C_HWkDfl_OY = WD_InitInp%C_HWkDfl_OY/D2R !immediately convert to m/radians instead of m/degrees      
      
      ! C_HWkDfl_x - Calibrated parameter in the correction for wake deflection defining the horizontal offset scaled with downstream distance (-) or DEFAULT [DEFAULT=0.0]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_HWkDfl_x, "C_HWkDfl_x", &
      "Calibrated parameter in the correction for wake deflection defining the horizontal offset scaled with downstream distance (-) or DEFAULT [DEFAULT=0.0]", &
      0.0_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
         
      ! C_HWkDfl_xY - Calibrated parameter in the correction for wake deflection defining the horizontal offset scaled with downstream distance and yaw error (1/deg) or DEFAULT [DEFAULT=-0.004]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_HWkDfl_xY, "C_HWkDfl_xY", &
      "Calibrated parameter in the correction for wake deflection defining the horizontal offset scaled with downstream distance and yaw error (1/deg) or DEFAULT [DEFAULT=-0.004]", &
      -0.004_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
   WD_InitInp%C_HWkDfl_xY = WD_InitInp%C_HWkDfl_xY/D2R !immediately convert to 1/radians instead of 1/degrees      

   
      ! C_NearWake - Calibrated parameter for the near-wake correction (-) [>1.0] or DEFAULT [DEFAULT=1.8]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_NearWake, "C_NearWake", &
      "Calibrated parameter for the near-wake correction (-) [>1.0] or DEFAULT [DEFAULT=1.8]", &
      1.8_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
              
      ! k_vAmb - Calibrated parameter for the influence of ambient turbulence in the eddy viscosity (-) [>=0.0] or DEFAULT [DEFAULT=0.05 ]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%k_vAmb, "k_vAmb", &
      "Calibrated parameter for the influence of ambient turbulence in the eddy viscosity (-) [>=0.0] or DEFAULT [DEFAULT=0.05]", &
      0.05_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
         
      ! k_vShr - Calibrated parameter for the influence of the shear layer in the eddy viscosity (-) [>=0.0] or DEFAULT [DEFAULT=0.016]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%k_vShr, "k_vShr", &
      "Calibrated parameter for the influence of the shear layer in the eddy viscosity (-) [>=0.0] or DEFAULT [DEFAULT=0.016]", &
      0.016_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      

      ! C_vAmb_DMin - Calibrated parameter in the eddy viscosity filter function for ambient turbulence defining the transitional diameter fraction between the minimum and exponential regions (-) [>=0.0] or DEFAULT [DEFAULT=0.0]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_vAmb_DMin, "C_vAmb_DMin", &
      "Calibrated parameter in the eddy viscosity filter function for ambient turbulence defining the transitional diameter fraction between the minimum and exponential regions (-) [>=0.0] or DEFAULT [DEFAULT=0.0]", &
      0.0_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      
      ! C_vAmb_DMax - Calibrated parameter in the eddy viscosity filter function for ambient turbulence defining the transitional diameter fraction between the exponential and maximum regions (-) [> C_vAmb_DMin  ] or DEFAULT [DEFAULT=1.0]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_vAmb_DMax, "C_vAmb_DMax", &
      "Calibrated parameter in the eddy viscosity filter function for ambient turbulence defining the transitional diameter fraction between the exponential and maximum regions (-) [> C_vAmb_DMin  ] or DEFAULT [DEFAULT=1.0]", &
      1.0_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
        
      ! C_vAmb_FMin - Calibrated parameter in the eddy viscosity filter function for ambient turbulence defining the value in the minimum region (-) [>=0.0 and <=1.0] or DEFAULT [DEFAULT=1.0]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_vAmb_FMin, "C_vAmb_FMin", &
      "Calibrated parameter in the eddy viscosity filter function for ambient turbulence defining the value in the minimum region (-) [>=0.0 and <=1.0] or DEFAULT [DEFAULT=1.0]", &
      1.0_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      
      ! C_vAmb_Exp - Calibrated parameter in the eddy viscosity filter function for ambient turbulence defining the exponent in the exponential region (-) [> 0.0] or DEFAULT [DEFAULT=0.01]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_vAmb_Exp, "C_vAmb_Exp", &
      "Calibrated parameter in the eddy viscosity filter function for ambient turbulence defining the exponent in the exponential region (-) [> 0.0] or DEFAULT [DEFAULT=0.01]", &
      0.01_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
        
      ! C_vShr_DMin - Calibrated parameter in the eddy viscosity filter function for the shear layer defining the transitional diameter fraction between the minimum and exponential regions (-) [>=0.0] or DEFAULT [DEFAULT=3.0]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_vShr_DMin, "C_vShr_DMin", &
      "Calibrated parameter in the eddy viscosity filter function for the shear layer defining the transitional diameter fraction between the minimum and exponential regions (-) [>=0.0] or DEFAULT [DEFAULT=3.0]", &
      3.0_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if            
         
      ! C_vShr_DMax - Calibrated parameter in the eddy viscosity filter function for the shear layer defining the transitional diameter fraction between the exponential and maximum regions (-) [> C_vShr_DMin] or DEFAULT [DEFAULT=25.0]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_vShr_DMax, "C_vShr_DMax", &
      "Calibrated parameter in the eddy viscosity filter function for the shear layer defining the transitional diameter fraction between the exponential and maximum regions (-) [> C_vShr_DMin] or DEFAULT [DEFAULT=25.0]", &
      25.0_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if            
         
      ! C_vShr_FMin - Calibrated parameter in the eddy viscosity filter function for the shear layer defining the value in the minimum region (-) [>=0.0 and <=1.0] or DEFAULT [DEFAULT=0.2]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_vShr_FMin, "C_vShr_FMin", &
      "Calibrated parameter in the eddy viscosity filter function for the shear layer defining the value in the minimum region (-) [>=0.0 and <=1.0] or DEFAULT [DEFAULT=0.2]", &
      0.2_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if            
        
      ! C_vShr_Exp - Calibrated parameter in the eddy viscosity filter function for the shear layer defining the exponent in the exponential region (-) [> 0.0] or DEFAULT [DEFAULT=0.1]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_vShr_Exp, "C_vShr_Exp", &
      "Calibrated parameter in the eddy viscosity filter function for the shear layer defining the exponent in the exponential region (-) [> 0.0] or DEFAULT [DEFAULT=0.1]", &
      0.1_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if            
        
      ! Mod_WakeDiam - Wake diameter calculation model (-) (switch) {1: rotor diameter, 2: velocity-based, 3: mass-flux based, 4: momentum-flux based} or DEFAULT [DEFAULT=1]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%Mod_WakeDiam, "Mod_WakeDiam", &
      "Wake diameter calculation model (-) (switch) {1: rotor diameter, 2: velocity-based, 3: mass-flux based, 4: momentum-flux based} or DEFAULT [DEFAULT=1]", &
      WakeDiamMod_RotDiam, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if            
        
      ! C_WakeDiam - Calibrated parameter for wake diameter calculation (-) [>0.0 and <1.0] or DEFAULT [DEFAULT=0.95] [unused for Mod_WakeDiam=1]:
   CALL ReadVarWDefault( UnIn, InputFile, WD_InitInp%C_WakeDiam, "C_WakeDiam", &
      "Calibrated parameter for wake diameter calculation (-) [>0.0 and <1.0] or DEFAULT [DEFAULT=0.95] [unused for Mod_WakeDiam=1]", &
      0.95_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if            

      ! Mod_Meander - Spatial filter model for wake meandering (-) (switch) {1: uniform, 2: truncated jinc, 3: windowed jinc} or DEFAULT [DEFAULT=3]:
   CALL ReadVarWDefault( UnIn, InputFile, AWAE_InitInp%Mod_Meander, "Mod_Meander", &
      "Spatial filter model for wake meandering (-) (switch) {1: uniform, 2: truncated jinc, 3: windowed jinc} or DEFAULT [DEFAULT=3]", &
      MeanderMod_WndwdJinc, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if            
        
      ! C_Meander - Calibrated parameter for wake meandering (-) [>=1.0] or DEFAULT [DEFAULT=1.9]:
   CALL ReadVarWDefault( UnIn, InputFile, AWAE_InitInp%C_Meander, "C_Meander", &
      "Calibrated parameter for wake meandering (-) [>=1.0] or DEFAULT [DEFAULT=1.9]", &
      1.9_ReKi, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if            

   !---------------------- VISUALIZATION --------------------------------------------------
   CALL ReadCom( UnIn, InputFile, 'Section Header: Visualization', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      
      ! WrDisWind - Write disturbed wind data to <OutFileRoot>.Low.Dis.t<n/n_low-out>.vtk etc.? (flag):
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%WrDisWind, "WrDisWind", "Write disturbed wind data to <OutFileRoot>.Low.Dis.t<n/n_low-out>.vtk etc.? (flag)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName) 
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! NOutDisWindXY - Number of XY planes for output of disturbed wind data across the low-resolution domain to <OutFileRoot>.Low.DisXY.<n_out>.t<n/n_low-out>.vtk (-) [0 to 9]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%NOutDisWindXY, "NOutDisWindXY", "Number of XY planes for output of disturbed wind data across the low-resolution domain to <OutFileRoot>.Low.DisXY.<n_out>.t<n/n_low-out>.vtk (-) [0 to 9]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
     
      call allocAry( AWAE_InitInp%OutDisWindZ, AWAE_InitInp%NOutDisWindXY, "OutDisWindZ", ErrStat2, ErrMsg2 )
         CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if
      
      ! OutDisWindZ - Z coordinates of XY planes for output of disturbed wind data across the low-resolution domain (m) [1 to NOutDisWindXY] [unused for NOutDisWindXY=0]:
   CALL ReadAry( UnIn, InputFile, AWAE_InitInp%OutDisWindZ, AWAE_InitInp%NOutDisWindXY, "OutDisWindZ", "Z coordinates of XY planes for output of disturbed wind data across the low-resolution domain (m) [1 to NOutDisWindXY] [unused for NOutDisWindXY=0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! NOutDisWindYZ - Number of YZ planes for output of disturbed wind data across the low-resolution domain to <OutFileRoot>.Low.DisYZ.<n_out>.t<n/n_low-out>.vtk (-) [0 to 9]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%NOutDisWindYZ, "NOutDisWindYZ", "Number of YZ planes for output of disturbed wind data across the low-resolution domain to <OutFileRoot>.Low.DisYZ.<n_out>.t<n/n_low-out>.vtk (-) [0 to 9]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
     
      call allocAry( AWAE_InitInp%OutDisWindX, AWAE_InitInp%NOutDisWindYZ, "OutDisWindX", ErrStat2, ErrMsg2 )
         CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if
      
      ! OutDisWindX - X coordinates of YZ planes for output of disturbed wind data across the low-resolution domain (m) [1 to NOutDisWindYZ] [unused for NOutDisWindYZ=0]:
   CALL ReadAry( UnIn, InputFile, AWAE_InitInp%OutDisWindX, AWAE_InitInp%NOutDisWindYZ, "OutDisWindX", "X coordinates of YZ planes for output of disturbed wind data across the low-resolution domain (m) [1 to NOutDisWindYZ] [unused for NOutDisWindYZ=0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      
      ! NOutDisWindXZ - Number of XZ planes for output of disturbed wind data across the low-resolution domain to <OutFileRoot>.Low/DisXZ.<n_out>.t<n/n_low-out>.vtk (-) [0 to 9]:
   CALL ReadVar( UnIn, InputFile, AWAE_InitInp%NOutDisWindXZ, "NOutDisWindXZ", "Number of XZ planes for output of disturbed wind data across the low-resolution domain to <OutFileRoot>.Low/DisXZ.<n_out>.t<n/n_low-out>.vtk (-) [0 to 9]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      call allocAry( AWAE_InitInp%OutDisWindY, AWAE_InitInp%NOutDisWindXZ, "OutDisWindY", ErrStat2, ErrMsg2 )
         CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if
      
      ! OutDisWindY - Y coordinates of XZ planes for output of disturbed wind data across the low-resolution domain (m) [1 to NOutDisWindXZ] [unused for NOutDisWindXZ=0]:
   CALL ReadAry( UnIn, InputFile, AWAE_InitInp%OutDisWindY, AWAE_InitInp%NOutDisWindXZ, "OutDisWindY", "Y coordinates of XZ planes for output of disturbed wind data across the low-resolution domain (m) [1 to NOutDisWindXZ] [unused for NOutDisWindXZ=0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
                  
          ! WrDisDT -The time between vtk outputs [must be a multiple of the low resolution time step]:
   CALL ReadVarWDefault( UnIn, InputFile, AWAE_InitInp%WrDisDT, "WrDisDT", &
      "The time between vtk outputs [must be a multiple of the low resolution time step]", &
      p%DT_low, ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      
   !---------------------- OUTPUT --------------------------------------------------
   CALL ReadCom( UnIn, InputFile, 'Section Header: Output', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      ! SumPrint - Print summary data to <RootName>.sum? (flag):
   CALL ReadVar( UnIn, InputFile, p%SumPrint, "SumPrint", "Print summary data to <RootName>.sum? (flag)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! ChkptTime - Amount of time between creating checkpoint files for potential restart (s) [>0.0]:
   CALL ReadVar( UnIn, InputFile, TmpTime, "ChkptTime", "Amount of time between creating checkpoint files for potential restart (s) [>0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      IF (TmpTime > p%TMax) THEN
         p%n_ChkptTime = HUGE(p%n_ChkptTime)
      ELSE         
         p%n_ChkptTime = NINT( TmpTime / p%DT_low )
      END IF
      

      ! TStart - Time to begin tabular output (s) [>=0.0]:
   CALL ReadVar( UnIn, InputFile, p%TStart, "TStart", "Time to begin tabular output (s) [>=0.0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   
      ! OutFileFmt - Format for tabular (time-marching) output file (switch) {1: text file [<RootName>.out], 2: binary file [<RootName>.outb], 3: both}:
   CALL ReadVar( UnIn, InputFile, OutFileFmt, "OutFileFmt", "Format for tabular (time-marching) output file (switch) {1: text file [<RootName>.out], 2: binary file [<RootName>.outb], 3: both}", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if

      SELECT CASE (OutFileFmt)
         CASE (1_IntKi)
            p%WrBinOutFile = .FALSE.
            p%WrTxtOutFile = .TRUE.
         CASE (2_IntKi)
            p%WrBinOutFile = .TRUE.
            p%WrTxtOutFile = .FALSE.
         CASE (3_IntKi)
            p%WrBinOutFile = .TRUE.
            p%WrTxtOutFile = .TRUE.
         CASE DEFAULT
            ! we'll check this later....
            !CALL SetErrStat( ErrID_Fatal, "FAST.Farm's OutFileFmt must be 1, 2, or 3.",ErrStat,ErrMsg,RoutineName)
            !if ( ErrStat >= AbortErrLev ) then
            !   call cleanup()
            !   RETURN        
            !end if
      END SELECT
   
      if ( OutFileFmt /= 1_IntKi ) then ! TODO: Only allow text format for now; add binary format later.
         CALL SetErrStat( ErrID_Fatal, "FAST.Farm's OutFileFmt must be 1.",ErrStat,ErrMsg,RoutineName) 
         call cleanup()
         RETURN        
      end if
      
   
      ! TabDelim - Use tab delimiters in text tabular output file? (flag) {uses spaces if False}:
   CALL ReadVar( UnIn, InputFile, TabDelim, "TabDelim", "Use tab delimiters in text tabular output file? (flag) {uses spaces if False}", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
   
      IF ( TabDelim ) THEN
         p%Delim = TAB
      ELSE
         p%Delim = ' '
      END IF
   
      ! OutFmt - Format used for text tabular output, excluding the time channel. Resulting field should be 10 characters. (quoted string):
   CALL ReadVar( UnIn, InputFile, p%OutFmt, "OutFmt", "Format used for text tabular output, excluding the time channel. Resulting field should be 10 characters. (quoted string)", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      ! NOutRadii - Number of radial nodes for wake output for an individual rotor (-) [0 to 20]:
   CALL ReadVar( UnIn, InputFile, p%NOutRadii, "NOutRadii", "Number of radial nodes for wake output for an individual rotor (-) [0 to 20]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      call allocary( p%OutRadii, p%NOutRadii, "OutRadii", ErrStat2, ErrMsg2 )
         CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if
      
      ! OutRadii - List of radial nodes for wake output for an individual rotor (-) [1 to NOutRadii]:
   CALL ReadAry( UnIn, InputFile, p%OutRadii, p%NOutRadii, "OutRadii", "List of radial nodes for wake output for an individual rotor (-) [1 to NOutRadii]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      
      ! NOutDist - Number of downstream distances for wake output for an individual rotor (-) [0 to 9]:
   CALL ReadVar( UnIn, InputFile, p%NOutDist, "NOutDist", "Number of downstream distances for wake output for an individual rotor (-) [0 to 9]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      call allocary( p%OutDist, p%NOutDist, "OutDist", ErrStat2, ErrMsg2 )
         CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if
      
      ! OutDist - List of downstream distances for wake output for an individual rotor (m) [1 to NOutDist] [unused for NOutDist=0]:
   CALL ReadAry( UnIn, InputFile, p%OutDist, p%NOutDist, "OutDist", "List of downstream distances for wake output for an individual rotor (m) [1 to NOutDist] [unused for NOutDist=0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
            
      ! NWindVel - Number of points for wind output (-) [0 to 9]:
   CALL ReadVar( UnIn, InputFile, p%NWindVel, "NWindVel", "Number of points for wind output (-) [0 to 9]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if
      
      call allocAry( p%WindVelX, p%NWindVel, "WindVelX", ErrStat2, ErrMsg2 );  CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      call allocAry( p%WindVelY, p%NWindVel, "WindVelY", ErrStat2, ErrMsg2 );  CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      call allocAry( p%WindVelZ, p%NWindVel, "WindVelZ", ErrStat2, ErrMsg2 );  CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         if ( ErrStat >= AbortErrLev ) then
            call cleanup()
            RETURN        
         end if
      
      ! WindVelX - List of coordinates in the X direction for wind output (m) [1 to NWindVel] [unused for NWindVel=0]:
   CALL ReadAry( UnIn, InputFile, p%WindVelX, p%NWindVel, "WindVelX", "List of coordinates in the X direction for wind output (m) [1 to NWindVel] [unused for NWindVel=0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
         
      ! WindVelY - List of coordinates in the Y direction for wind output (m) [1 to NWindVel] [unused for NWindVel=0]:
   CALL ReadAry( UnIn, InputFile, p%WindVelY, p%NWindVel, "WindVelY", "List of coordinates in the Y direction for wind output (m) [1 to NWindVel] [unused for NWindVel=0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      
      ! WindVelZ - List of coordinates in the Z direction for wind output (m) [1 to NWindVel] [unused for NWindVel=0]:
   CALL ReadAry( UnIn, InputFile, p%WindVelZ, p%NWindVel, "WindVelZ", "List of coordinates in the Z direction for wind output (m) [1 to NWindVel] [unused for NWindVel=0]", ErrStat2, ErrMsg2, UnEc)
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)      
      if ( ErrStat >= AbortErrLev ) then
         call cleanup()
         RETURN        
      end if      
      
      
 !!!!!!!                  OutList            The next line(s) contains a list of output parameters.  See OutListParameters.xlsx for a listing of available output channels (quoted string)      
      !---------------------- OUTLIST  --------------------------------------------
   CALL ReadCom( UnIn, InputFile, 'Section Header: OutList', ErrStat2, ErrMsg2, UnEc )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF ( ErrStat >= AbortErrLev ) THEN
         CALL Cleanup()
         RETURN
      END IF

      ! OutList - List of user-requested output channels (-):
   CALL ReadOutputList ( UnIn, InputFile, OutList, p%NumOuts, 'OutList', "List of user-requested output channels", ErrStat2, ErrMsg2, UnEc  )     ! Routine in NWTC Subroutine Library
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      IF ( ErrStat >= AbortErrLev ) THEN
         CALL Cleanup()
         RETURN
      END IF      
   !---------------------- END OF FILE -----------------------------------------

   call cleanup()
   RETURN

CONTAINS
   !...............................................................................................................................
   subroutine cleanup()
      CLOSE( UnIn )
      IF ( UnEc > 0 ) CLOSE ( UnEc )   
   end subroutine cleanup
   !...............................................................................................................................
END SUBROUTINE Farm_ReadPrimaryFile
!----------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE Farm_ValidateInput( p, WD_InitInp, AWAE_InitInp, SC_InitInp, ErrStat, ErrMsg )
      ! Passed variables
   TYPE(Farm_ParameterType), INTENT(INOUT) :: p                               !< The parameter data for the FAST (glue-code) simulation
   TYPE(WD_InputFileType),   INTENT(IN   ) :: WD_InitInp                      !< input-file data for WakeDynamics module
   TYPE(AWAE_InputFileType), INTENT(INOUT) :: AWAE_InitInp                    !< input-file data for AWAE module
   TYPE(SC_InitInputType),   INTENT(INOUT) :: SC_InitInp              ! input-file data for SC module
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

      ! Local variables:
   INTEGER(IntKi)                :: i  
   INTEGER(IntKi)                :: ErrStat2                                  ! Temporary Error status
   CHARACTER(ErrMsgLen)          :: ErrMsg2                                   ! Temporary Error message
   CHARACTER(*),   PARAMETER     :: RoutineName = 'Farm_ValidateInput'
   INTEGER(IntKi)                :: n_disDT_dt
   
   ErrStat = ErrID_None
   ErrMsg  = ""
   
   IF (p%DT_low <= 0.0_ReKi) CALL SetErrStat(ErrID_Fatal,'DT_low must be positive.',ErrStat,ErrMsg,RoutineName)
   IF (p%DT_high <= 0.0_ReKi) CALL SetErrStat(ErrID_Fatal,'DT_high must be positive.',ErrStat,ErrMsg,RoutineName)
   IF (p%TMax < 0.0_ReKi) CALL SetErrStat(ErrID_Fatal,'TMax must not be negative.',ErrStat,ErrMsg,RoutineName)
   IF (p%NumTurbines < 1) CALL SetErrStat(ErrID_Fatal,'FAST.Farm requires at least 1 turbine. Set NumTurbines > 0.',ErrStat,ErrMsg,RoutineName)
   
   ! --- SUPER CONTROLLER ---
   ! TODO : Verify that the DLL file exists
   
   
   ! --- WAKE DYNAMICS ---
   IF (WD_InitInp%dr <= 0.0_ReKi) CALL SetErrStat(ErrID_Fatal,'dr (radial increment) must be larger than 0.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%NumRadii < 2) CALL SetErrStat(ErrID_Fatal,'NumRadii (number of radii) must be at least 2.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%NumPlanes < 2) CALL SetErrStat(ErrID_Fatal,'NumPlanes (number of wake planes) must be at least 2.',ErrStat,ErrMsg,RoutineName)

   IF (WD_InitInp%f_c <= 0.0_ReKi) CALL SetErrStat(ErrID_Fatal,'f_c (cut-off [corner] frequency) must be more than 0 Hz.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%C_NearWake <= 1.0_Reki) CALL SetErrStat(ErrID_Fatal,'C_NearWake parameter must be greater than 1.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%k_vAmb < 0.0_Reki) CALL SetErrStat(ErrID_Fatal,'k_vAmb parameter must not be negative.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%k_vShr < 0.0_Reki) CALL SetErrStat(ErrID_Fatal,'k_vShr parameter must not be negative.',ErrStat,ErrMsg,RoutineName)
   
   IF (WD_InitInp%C_vAmb_DMin < 0.0_Reki) CALL SetErrStat(ErrID_Fatal,'C_vAmb_DMin parameter must not be negative.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%C_vAmb_DMax <= WD_InitInp%C_vAmb_DMin) CALL SetErrStat(ErrID_Fatal,'C_vAmb_DMax parameter must be larger than C_vAmb_DMin.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%C_vAmb_FMin < 0.0_Reki .or. WD_InitInp%C_vAmb_FMin > 1.0_Reki) CALL SetErrStat(ErrID_Fatal,'C_vAmb_FMin parameter must be between 0 and 1 (inclusive).',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%C_vAmb_Exp  <= 0.0_Reki) CALL SetErrStat(ErrID_Fatal,'C_vAmb_Exp parameter must be positive.',ErrStat,ErrMsg,RoutineName)

   IF (WD_InitInp%C_vShr_DMin < 0.0_Reki) CALL SetErrStat(ErrID_Fatal,'C_vShr_DMin parameter must not be negative.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%C_vShr_DMax <= WD_InitInp%C_vShr_DMin) CALL SetErrStat(ErrID_Fatal,'C_vShr_DMax parameter must be larger than C_vShr_DMin.',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%C_vShr_FMin < 0.0_Reki .or. WD_InitInp%C_vShr_FMin > 1.0_ReKi) CALL SetErrStat(ErrID_Fatal,'C_vShr_FMin parameter must be between 0 and 1 (inclusive).',ErrStat,ErrMsg,RoutineName)
   IF (WD_InitInp%C_vShr_Exp  <= 0.0_Reki) CALL SetErrStat(ErrID_Fatal,'C_vShr_Exp parameter must be positive.',ErrStat,ErrMsg,RoutineName)

   IF (WD_InitInp%Mod_WakeDiam < WakeDiamMod_RotDiam .or. WD_InitInp%Mod_WakeDiam > WakeDiamMod_MtmFlux) THEN
      call SetErrStat(ErrID_Fatal,'Wake diameter calculation model, Mod_WakeDiam, must be 1 (rotor diameter), 2 (velocity-based), 3 (mass-flux based), 4 (momentum-flux based) or DEFAULT.',ErrStat,ErrMsg,RoutineName)
   END IF
   
   IF (WD_InitInp%Mod_WakeDiam /= WakeDiamMod_RotDiam) THEN
      IF (WD_InitInp%C_WakeDiam <= 0.0_Reki .or. WD_InitInp%C_WakeDiam >= 1.0_ReKi) THEN
         CALL SetErrStat(ErrID_Fatal,'C_WakeDiam parameter must be between 0 and 1 (exclusive).',ErrStat,ErrMsg,RoutineName)
      END IF
   END IF
   
   IF (AWAE_InitInp%Mod_Meander < MeanderMod_Uniform .or. AWAE_InitInp%Mod_Meander > MeanderMod_WndwdJinc) THEN
      call SetErrStat(ErrID_Fatal,'Spatial filter model for wake meandering, Mod_Meander, must be 1 (uniform), 2 (truncated jinc), 3 (windowed jinc) or DEFAULT.',ErrStat,ErrMsg,RoutineName)
   END IF
   
   IF (AWAE_InitInp%C_Meander < 1.0_Reki) THEN
      CALL SetErrStat(ErrID_Fatal,'C_Meander parameter must not be less than 1.',ErrStat,ErrMsg,RoutineName)
   END IF
         
   !--- OUTPUT ---
   IF ( p%n_ChkptTime < 1_IntKi   ) CALL SetErrStat( ErrID_Fatal, 'ChkptTime must be greater than 0 seconds.', ErrStat, ErrMsg, RoutineName )
   IF (p%TStart < 0.0_ReKi) CALL SetErrStat(ErrID_Fatal,'TStart must not be negative.',ErrStat,ErrMsg,RoutineName)
   IF (.not. p%WrBinOutFile .and. .not. p%WrTxtOutFile) CALL SetErrStat( ErrID_Fatal, "FAST.Farm's OutFileFmt must be 1, 2, or 3.",ErrStat,ErrMsg,RoutineName)
   
   if (AWAE_InitInp%WrDisDT < p%DT_low) CALL SetErrStat(ErrID_Fatal,'WrDisDT must greater than or equal to dt_low.',ErrStat,ErrMsg,RoutineName)
   
      ! let's make sure the FAST.Farm DT_low is an exact integer divisor of AWAE_InitInp%WrDisDT 
   n_disDT_dt = nint( AWAE_InitInp%WrDisDT / p%DT_low )
      ! (i'm doing this outside of Farm_ValidateInput so we know that dt_low/=0 before computing n_high_low):
   IF ( .NOT. EqualRealNos( real(p%DT_low,SiKi)* n_disDT_dt, real(AWAE_InitInp%WrDisDT,SiKi)  )  ) THEN
      CALL SetErrStat(ErrID_Fatal, "WrDisDT ("//TRIM(Num2LStr(AWAE_InitInp%WrDisDT))//" s) must be an integer multiple of dt_low ("//TRIM(Num2LStr(p%DT_low))//" s).", ErrStat, ErrMsg, RoutineName ) 
   END IF
   AWAE_InitInp%WrDisDT =  p%DT_low * n_disDT_dt
   
   
   if (AWAE_InitInp%NOutDisWindXY < 0 .or. AWAE_InitInp%NOutDisWindXY > maxOutputPoints ) CALL SetErrStat( ErrID_Fatal, 'NOutDisWindXY must be in the range [0, 9].', ErrStat, ErrMsg, RoutineName )
   if (AWAE_InitInp%NOutDisWindYZ < 0 .or. AWAE_InitInp%NOutDisWindYZ > maxOutputPoints ) CALL SetErrStat( ErrID_Fatal, 'NOutDisWindYZ must be in the range [0, 9].', ErrStat, ErrMsg, RoutineName )
   if (AWAE_InitInp%NOutDisWindXZ < 0 .or. AWAE_InitInp%NOutDisWindXZ > maxOutputPoints ) CALL SetErrStat( ErrID_Fatal, 'NOutDisWindXZ must be in the range [0, 9].', ErrStat, ErrMsg, RoutineName )
   if (p%NOutDist < 0 .or. p%NOutDist > maxOutputPoints ) then
      CALL SetErrStat( ErrID_Fatal, 'NOutDist must be in the range [0, 9].', ErrStat, ErrMsg, RoutineName )
   else
      do i=1,p%NOutDist
         if (p%OutDist(i) <  0.0_ReKi) then
            CALL SetErrStat( ErrID_Fatal, 'OutDist values must be greater than or equal to zero.', ErrStat, ErrMsg, RoutineName )
            exit
         end if
      end do      
   end if

   if (p%NWindVel < 0 .or. p%NWindVel > maxOutputPoints ) CALL SetErrStat( ErrID_Fatal, 'NWindVel must be in the range [0, 9].', ErrStat, ErrMsg, RoutineName )
   if (p%NOutRadii < 0 .or. p%NOutRadii > 20 ) then
      CALL SetErrStat( ErrID_Fatal, 'NOutRadii must be in the range [0, 20].', ErrStat, ErrMsg, RoutineName )
   else
      do i=1,p%NOutRadii
         if (p%OutRadii(i) > WD_InitInp%NumRadii - 1 .or. p%OutRadii(i) < 0) then
            CALL SetErrStat( ErrID_Fatal, 'OutRadii must be in the range [0, NumRadii - 1].', ErrStat, ErrMsg, RoutineName )
            exit
         end if
      end do      
   end if
   
   
   
      ! Check that OutFmt is a valid format specifier and will fit over the column headings
  CALL ChkRealFmtStr( p%OutFmt, 'OutFmt', p%FmtWidth, ErrStat2, ErrMsg2 ) !this sets p%FmtWidth!
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      
   IF ( p%FmtWidth /= ChanLen ) CALL SetErrStat( ErrID_Warn, 'OutFmt produces a column width of '// &
         TRIM(Num2LStr(p%FmtWidth))//' instead of '//TRIM(Num2LStr(ChanLen))//' characters.', ErrStat, ErrMsg, RoutineName )
      
   
END SUBROUTINE Farm_ValidateInput
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine initializes all instances of WakeDynamics
SUBROUTINE Farm_InitWD( farm, WD_InitInp, ErrStat, ErrMsg )


      ! Passed variables
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data
   TYPE(WD_InitInputType),   INTENT(INOUT) :: WD_InitInp                      !< init input for WakeDynamics module; input file data already filled in
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   ! local variables
   type(WD_InitOutputType)                 :: WD_InitOut

   INTEGER(IntKi)                          :: nt                          ! loop counter for rotor number
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'Farm_InitWD'
         
   ErrStat = ErrID_None
   ErrMsg = ""
   
   ALLOCATE(farm%WD(farm%p%NumTurbines),STAT=ErrStat2)
   if (ErrStat2 /= 0) then
      CALL SetErrStat( ErrID_Fatal, 'Could not allocate memory for Wake Dynamics data', ErrStat, ErrMsg, RoutineName )
      return
   end if
            
      !.................
      ! Initialize each instance of WD
      !................                  
      
      DO nt = 1,farm%p%NumTurbines
         !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         ! initialization can be done in parallel (careful for FWrap_InitInp, though)
         !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++         
         
         WD_InitInp%TurbNum     = nt
         
            ! note that WD_Init has Interval as INTENT(IN) so, we don't need to worry about overwriting farm%p%dt_low here:
         call WD_Init( WD_InitInp, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                          farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, farm%p%dt_low, WD_InitOut, ErrStat2, ErrMsg2 )
         
         farm%WD(nt)%IsInitialized = .true.
            CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
            if (ErrStat >= AbortErrLev) then
               call cleanup()
               return
            end if
            
      END DO   
      
      farm%p%Module_Ver( ModuleFF_WD ) = WD_InitOut%Ver
      
      call cleanup()
      
contains
   subroutine cleanup()
      call WD_DestroyInitOutput( WD_InitOut, ErrStat2, ErrMsg2 )
   end subroutine cleanup
END SUBROUTINE Farm_InitWD
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine initializes all instances of FAST using the FASTWrapper module
SUBROUTINE Farm_InitFAST( farm, WD_InitInp, AWAE_InitOutput, SC_InitOutput, SC_y, ErrStat, ErrMsg )


      ! Passed variables
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data
   TYPE(WD_InputFileType),   INTENT(IN   ) :: WD_InitInp                      !< input-file data for WakeDynamics module
   TYPE(AWAE_InitOutputType),INTENT(IN   ) :: AWAE_InitOutput                 !< initialization output from AWAE
   type(SC_InitOutputType),  INTENT(INOUT) :: SC_InitOutput                   !< Initialization output from SC
   type(SC_OutputType),      INTENT(INOUT) :: SC_y                            !< SuperController inital outputs
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   ! local variables
   type(FWrap_InitInputType)               :: FWrap_InitInp
   type(FWrap_InitOutputType)              :: FWrap_InitOut

   INTEGER(IntKi)                          :: nt                          ! loop counter for rotor number
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'Farm_InitFAST'
   
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
   ALLOCATE(farm%FWrap(farm%p%NumTurbines),STAT=ErrStat2)
   if (ErrStat2 /= 0) then
      CALL SetErrStat( ErrID_Fatal, 'Could not allocate memory for FAST Wrapper data', ErrStat, ErrMsg, RoutineName )
      return
   end if
            
      !.................
      ! Initialize each instance of FAST
      !................            
      FWrap_InitInp%nr            = WD_InitInp%NumRadii
      FWrap_InitInp%dr            = WD_InitInp%dr
      FWrap_InitInp%tmax          = farm%p%TMax
      FWrap_InitInp%n_high_low    = farm%p%n_high_low + 1   ! Add 1 because the FAST wrapper uses an index that starts at 1
      FWrap_InitInp%dt_high       = farm%p%dt_high
     
      FWrap_InitInp%nX_high       = AWAE_InitOutput%nX_high
      FWrap_InitInp%nY_high       = AWAE_InitOutput%nY_high
      FWrap_InitInp%nZ_high       = AWAE_InitOutput%nZ_high
      FWrap_InitInp%UseSC         = farm%p%UseSC
      FWrap_InitInp%NumSC2Ctrl    = SC_InitOutput%NumSC2Ctrl
      FWrap_InitInp%NumSC2CtrlGlob= SC_InitOutput%NumSC2CtrlGlob
      FWrap_InitInp%NumCtrl2SC    = SC_InitOutput%NumCtrl2SC
      allocate(FWrap_InitInp%fromSCglob(SC_InitOutput%NumSC2CtrlGlob))
      FWrap_InitInp%fromSCglob = SC_y%fromSCglob
      
      allocate(FWrap_InitInp%fromSC(SC_InitOutput%NumSC2Ctrl))
      
      
      DO nt = 1,farm%p%NumTurbines
         !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         ! initialization can be done in parallel (careful for FWrap_InitInp, though)
         !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++         
         
         FWrap_InitInp%FASTInFile    = farm%p%WT_FASTInFile(nt)
         FWrap_InitInp%p_ref_Turbine = farm%p%WT_Position(:,nt)
         FWrap_InitInp%TurbNum       = nt
         FWrap_InitInp%RootName      = trim(farm%p%OutFileRoot)//'.T'//num2lstr(nt)
         
         
         FWrap_InitInp%p_ref_high(1) = AWAE_InitOutput%X0_high(nt)
         FWrap_InitInp%p_ref_high(2) = AWAE_InitOutput%Y0_high(nt)
         FWrap_InitInp%p_ref_high(3) = AWAE_InitOutput%Z0_high(nt)

         FWrap_InitInp%dX_high       = AWAE_InitOutput%dX_high(nt)
         FWrap_InitInp%dY_high       = AWAE_InitOutput%dY_high(nt)
         FWrap_InitInp%dZ_high       = AWAE_InitOutput%dZ_high(nt)
         if (SC_InitOutput%NumSC2Ctrl>0) then
            FWrap_InitInp%fromSC = SC_y%fromSC((nt-1)*SC_InitOutput%NumSC2Ctrl+1:nt*SC_InitOutput%NumSC2Ctrl)
         end if
            ! note that FWrap_Init has Interval as INTENT(IN) so, we don't need to worry about overwriting farm%p%dt_low here:
         call FWrap_Init( FWrap_InitInp, farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                          farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, farm%p%dt_low, FWrap_InitOut, ErrStat2, ErrMsg2 )
         
         farm%FWrap(nt)%IsInitialized = .true.
         
            CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
            if (ErrStat >= AbortErrLev) then
               call cleanup()
               return
            end if
            
      END DO   
   
      farm%p%Module_Ver( ModuleFF_FWrap ) = FWrap_InitOut%Ver
      
      call cleanup()
      
contains
   subroutine cleanup()
      call FWrap_DestroyInitInput( FWrap_InitInp, ErrStat2, ErrMsg2 )
      call FWrap_DestroyInitOutput( FWrap_InitOut, ErrStat2, ErrMsg2 )
   end subroutine cleanup
END SUBROUTINE Farm_InitFAST
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine performs the initial call to calculate outputs (at t=0).
!! The Initial Calculate Output algorithm: \n
!!    -  In parallel: 
!!       1. Set u_AWAE=0, CALL AWAE_CO, and transfer y_AWAE to u_F and u_WD
!!       2. Set u_SC=0, CALL SC_CO, and transfer y_SC to u_F
!!    -  CALL F_t0
!!    -  Transfer y_F to u_SC and u_WD
!!    -  CALL WD_CO
!!    -  Transfer y_WD to u_AWAE
!!    -  CALL AWAE_CO
!!    -  Transfer y_AWAE to u_F and u_WD
!!    -  Write Output to File
subroutine FARM_InitialCO(farm, ErrStat, ErrMsg)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                          :: nt                    
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_InitialCO'
   
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
   

   
   !.......................................................................................
   ! Initial calls to AWAE and SC modules (steps 1. and 2. can be done in parallel)
   !.......................................................................................
   
      !--------------------
      ! 1a. u_AWAE=0         
   farm%AWAE%u%xhat_plane = 0.0_ReKi     ! Orientations of wake planes, normal to wake planes, for each turbine
   farm%AWAE%u%p_plane    = 0.0_ReKi     ! Center positions of wake planes for each turbine
   farm%AWAE%u%Vx_wake    = 0.0_ReKi     ! Axial wake velocity deficit at wake planes, distributed radially, for each turbine
   farm%AWAE%u%Vr_wake    = 0.0_ReKi     ! Radial wake velocity deficit at wake planes, distributed radially, for each turbine
   farm%AWAE%u%D_wake     = 0.0_ReKi     ! Wake diameters at wake planes for each turbine      
   
      !--------------------
      ! 1b. CALL AWAE_CO      
   call AWAE_CalcOutput( 0.0_DbKi, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%y, farm%AWAE%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         if (ErrStat >= AbortErrLev) return
      !--------------------
      ! 1c. transfer y_AWAE to u_F and u_WD         
   
   call Transfer_AWAE_to_FAST(farm)      
   call Transfer_AWAE_to_WD(farm)   

   if (farm%p%UseSC) then
      !--------------------
      ! 2a. u_SC=0         
      if ( farm%SC%p%NInpGlobal > 0 ) farm%SC%uInputs%toSCglob = 0.0_SiKi
      if ( farm%SC%p%NumCtrl2SC > 0 ) farm%SC%uInputs%toSC     = 0.0_SiKi
      
      !--------------------
      ! 2b. CALL SC_CO 
   
      call SC_CalcOutput(0.0_DbKi, farm%SC%uInputs, farm%SC%p, farm%SC%x, farm%SC%xd, farm%SC%z, &
                           farm%SC%OtherState, farm%SC%y, farm%SC%m, ErrStat, ErrMsg )         
            call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
            if (ErrStat >= AbortErrLev) return
         
      !--------------------
      ! 2c. transfer y_SC to u_F         
   
      do nt = 1,farm%p%NumTurbines
         farm%FWrap(nt)%u%fromSCglob  = farm%SC%y%fromSCglob
            ! SC stores all turbine-controller data in a 1D array, need to separate these out for each turbine
         farm%FWrap(nt)%u%fromSC(:) = farm%SC%y%fromSC( (nt-1)*farm%SC%p%NumSC2Ctrl+1:nt*farm%SC%p%NumSC2Ctrl ) 
      end do
      
   end if ! (farm%p%UseSC)
   
   !.......................................................................................
   ! CALL F_t0 (can be done in parallel)
   !.......................................................................................
         
   DO nt = 1,farm%p%NumTurbines
      
      call FWrap_t0( farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                     farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
               
   END DO
   if (ErrStat >= AbortErrLev) return
   
   !.......................................................................................
   ! Transfer y_F to u_SC and u_WD (can be done in parallel)
   !.......................................................................................
      
      !--------------------
      ! 1.  Transfer y_F to u_SC     
   if (farm%p%UseSC) then
      
      farm%SC%uInputs%toSCglob = 0.0_SiKi  ! We currently do not have a way to set global SC inputs from FAST.Farm
   
      do nt = 1,farm%p%NumTurbines 
        farm%SC%uInputs%toSC( (nt-1)*farm%SC%p%NumCtrl2SC+1 : nt*farm%SC%p%NumCtrl2SC )   = farm%FWrap(nt)%y%toSC(:)
      end do
      
   end if
      !--------------------
      ! 2.  Transfer y_F to u_WD     
   
   call Transfer_FAST_to_WD(farm)
      
   !.......................................................................................
   ! CALL WD_CO (can be done in parallel)
   !.......................................................................................
   
   DO nt = 1,farm%p%NumTurbines
      
      call WD_CalcOutput( 0.0_DbKi, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                     farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
               
   END DO
   if (ErrStat >= AbortErrLev) return
   
   !.......................................................................................
   ! Transfer y_WD to u_AWAE
   !.......................................................................................
   
   call Transfer_WD_to_AWAE(farm)
   
   !.......................................................................................
   ! CALL AWAE_CO
   !.......................................................................................
   
   call AWAE_CalcOutput( 0.0_DbKi, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%y, farm%AWAE%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   if (ErrStat >= AbortErrLev) return
   
   !.......................................................................................
   ! Transfer y_AWAE to u_F and u_WD
   !.......................................................................................
   
   call Transfer_AWAE_to_FAST(farm)              
   call Transfer_AWAE_to_WD(farm)   
   
   !.......................................................................................
   ! Write Output to File
   !.......................................................................................
   
   call Farm_WriteOutput(0, 0.0_DbKi, farm, ErrStat2, ErrMsg2)
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   
end subroutine FARM_InitialCO
!---------------------------------------------------------------------------------------------------------------------------------- 
!> This routine updates states each time increment. 
!! The update states algorithm: \n 
!!    -  In parallel:  
!!       1. call WD_US 
!!       2. call SC_US 
!!       3. call F_Increment 
!!       4. call AWAE_UpdateStates 
!!    -  \f$ n = n + 1 \f$ 
!!    -  \f$ t = t + \Delta t \f$ 
subroutine FARM_UpdateStates(t, n, farm, ErrStat, ErrMsg)
   REAL(DbKi),               INTENT(IN   ) :: t                               !< Current simulation time in seconds
   INTEGER(IntKi),           INTENT(IN   ) :: n                               !< Current step of the simulation: t = n*Interval
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                          :: nt                    
   INTEGER(IntKi)                          :: ErrStatWD, ErrStat2 
   INTEGER(IntKi), ALLOCATABLE             :: ErrStatF(:)                     ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsgWD
   CHARACTER(ErrMsgLen), ALLOCATABLE       :: ErrMsgF (:)                     ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_UpdateStates'
!   REAL(DbKi)                              :: tm1,tm2,tm3
   
   ErrStat = ErrID_None
   ErrMsg = ""

   allocate ( ErrStatF ( farm%p%NumTurbines + 1 ), STAT=errStat2 )
       if (errStat2 /= 0) call SetErrStat ( ErrID_Fatal, 'Could not allocate memory for ErrStatF.', errStat, errMsg, RoutineName )
   allocate ( ErrMsgF ( farm%p%NumTurbines + 1 ), STAT=errStat2 )
       if (errStat2 /= 0) call SetErrStat ( ErrID_Fatal, 'Could not allocate memory for ErrMsgF.', errStat, errMsg, RoutineName )
   if (ErrStat >= AbortErrLev) return
   
   !.......................................................................................
   ! update module states (steps 1. and 2. and 3. and 4. can be done in parallel)
   !.......................................................................................
   
      !--------------------
      ! 1. CALL WD_US         
  
   DO nt = 1,farm%p%NumTurbines
      
      call WD_UpdateStates( t, n, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                     farm%WD(nt)%OtherSt, farm%WD(nt)%m, ErrStatWD, ErrMsgWD )         
         call SetErrStat(ErrStatWD, ErrMsgWD, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':FARM_UpdateStates')
         
   END DO
   
   if (ErrStat >= AbortErrLev) return
   
   
      !--------------------
      ! 2. CALL SC_US  
   if (farm%p%useSC) then
      farm%SC%utimes(1) = t
      call SC_UpdateStates(t, n, farm%SC%uInputs,farm%SC%utimes, farm%SC%p, farm%SC%x, farm%SC%xd, farm%SC%z, farm%SC%OtherState, farm%SC%m, errStat, errMsg ) ! implement framework interface arguments
      if (errStat >= AbortErrLev) return
   end if
   
      !--------------------
      ! 3. CALL F_Increment and 4. CALL AWAE_UpdateStates  
!#ifdef _OPENMP
!   tm1 = omp_get_wtime()  
!#endif     
   !$OMP PARALLEL DO DEFAULT(Shared) Private(nt) !Private(nt,tm2,tm3)
   DO nt = 1,farm%p%NumTurbines+1
      if(nt.ne.farm%p%NumTurbines+1) then  
!#ifdef _OPENMP
!         tm3 = omp_get_wtime()  
!#endif     
         call FWrap_Increment( t, n, farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                     farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, ErrStatF(nt), ErrMsgF(nt) )         
         
!#ifdef _OPENMP
!         tm2 = omp_get_wtime() 
!         write(*,*)  '    FWrap_Increment for turbine #'//trim(num2lstr(nt))//' using thread #'//trim(num2lstr(omp_get_thread_num()))//' taking '//trim(num2lstr(tm2-tm3))//' seconds'
!#endif

      else
!#ifdef _OPENMP
!         tm3 = omp_get_wtime()  
!#endif    
         call AWAE_UpdateStates( t, n, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%m, errStatF(nt), errMsgF(nt) )       

!#ifdef _OPENMP
!         tm2 = omp_get_wtime() 
!         write(*,*)  '    AWAE_UpdateStates using thread #'//trim(num2lstr(omp_get_thread_num()))//' taking '//trim(num2lstr(tm2-tm3))//' seconds'
!#endif
      endif
      
   END DO
   !$OMP END PARALLEL DO  

   DO nt = 1,farm%p%NumTurbines 
      call SetErrStat(ErrStatF(nt), ErrMsgF(nt), ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':FARM_UpdateStates')
   END DO

   call SetErrStat(ErrStatF(farm%p%NumTurbines+1), ErrMsgF(farm%p%NumTurbines+1), ErrStat, ErrMsg, 'FARM_UpdateStates')

   if (ErrStat >= AbortErrLev) return

   
!#ifdef _OPENMP   
!  tm2 = omp_get_wtime()
!  write(*,*) 'Total Farm_US took '//trim(num2lstr(tm2-tm1))//' seconds.'
!#endif 
   
end subroutine FARM_UpdateStates

subroutine Farm_WriteOutput(n, t, farm, ErrStat, ErrMsg)
   INTEGER(IntKi),           INTENT(IN   ) :: n                               !< Time step increment number
   REAL(DbKi),               INTENT(IN   ) :: t                               !< Current simulation time in seconds
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message
                
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_WriteOutput'
   INTEGER(IntKi)                          :: nt, iSC, ir, iOutDist, np, iVelPt  ! Loop counters
   REAL(ReKi)                              :: vel(3), pt(3)
   REAL(ReKi)                              :: vec_interp(3)
   REAL(ReKi)                              :: norm2_vec, delta, deltad
   
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
      ! If requested write output channel data
   if ( farm%p%NumOuts > 0 ) then
    
      
         ! Define the output channel specifying the current simulation time:
      farm%m%AllOuts(  Farm_Time_Indx) = REAL( t, ReKi )

         !.......................................................................................
         ! Super controller Outputs - Global
         !.......................................................................................
             
      do iSC = 1, farm%SC%p%nInpGlobal
         farm%m%AllOuts(SCGblIn(iSC)) = farm%SC%uInputs%toSCglob(iSC)
      end do

      do iSC = 1, farm%SC%p%NumSC2CtrlGlob
         farm%m%AllOuts(SCGblOt(iSC)) = farm%SC%y%fromSCglob(iSC)
      end do

      do nt = 1, farm%p%NOutTurb
         
         !.......................................................................................
         ! Super controller Outputs - Turbine Dependent
         !.......................................................................................
             
         do iSC = 1, farm%SC%p%NumCtrl2SC
            farm%m%AllOuts(SCTIn(iSC,nt)) = farm%FWrap(nt)%y%toSC(iSC)
         end do

         do iSC = 1, farm%SC%p%NumSC2Ctrl
            farm%m%AllOuts(SCTOt(iSC,nt)) = farm%FWrap(nt)%u%fromSC(iSC)
         end do
         
         !.......................................................................................
         ! Wind Turbine and its Inflow
         !.......................................................................................

            ! Orientation of rotor centerline, normal to disk
         farm%m%AllOuts(RtAxsXT(nt)) = farm%FWrap(nt)%y%xHat_Disk(1)
         farm%m%AllOuts(RtAxsYT(nt)) = farm%FWrap(nt)%y%xHat_Disk(2)
         farm%m%AllOuts(RtAxsZT(nt)) = farm%FWrap(nt)%y%xHat_Disk(3)
         
            ! Center position of hub, m
         farm%m%AllOuts(RtPosXT(nt)) = farm%FWrap(nt)%y%p_hub(1)
         farm%m%AllOuts(RtPosYT(nt)) = farm%FWrap(nt)%y%p_hub(2)
         farm%m%AllOuts(RtPosZT(nt)) = farm%FWrap(nt)%y%p_hub(3)
         
            ! Rotor diameter, m
         farm%m%AllOuts(RtDiamT(nt)) = farm%FWrap(nt)%y%D_rotor
         
            ! Nacelle-yaw error at the wake planes, deg 
         farm%m%AllOuts(YawErrT(nt)) = farm%FWrap(nt)%y%YawErr*R2D
         
            ! Ambient turbulence intensity of the wind at the rotor disk, percent
         farm%m%AllOuts(TIAmbT(nt))  = farm%AWAE%y%TI_amb(nt)*100.0_ReKi 
         
            ! Rotor-disk-averaged ambient wind speed (normal to disk, not including structural motion, local induction or wakes from upstream turbines), m/s
         farm%m%AllOuts(RtVAmbT(nt)) = farm%AWAE%y%Vx_wind_disk(nt)
         
            ! Rotor-disk-averaged relative wind speed (normal to disk, including structural motion and wakes from upstream turbines, but not including local induction), m/s
         farm%m%AllOuts(RtVRelT(nt)) = farm%FWrap(nt)%y%DiskAvg_Vx_Rel
         
            ! Azimuthally averaged thrust force coefficient (normal to disk), distributed radially, -
         do ir = 1, farm%p%NOutRadii
            farm%m%AllOuts(CtTN(ir, nt)) = farm%FWrap(nt)%y%AzimAvg_Ct(farm%p%OutRadii(ir)+1)  ! y%AzimAvg_Ct is a 1-based array but the user specifies 0-based node indices, so we need to add 1
         end do
         
         !.......................................................................................
         ! Wake (for an Individual Rotor)
         !.......................................................................................
         
            ! Loop over user-requested, downstream distances (OutDist), m   
         do iOutDist = 1, farm%p%NOutDist
            
            if (  farm%p%OutDist(iOutDist) >= maxval( farm%WD(nt)%y%x_plane(0:min(farm%WD(nt)%p%NumPlanes-1,n+1)) ) ) then
               
               farm%m%AllOuts(WkAxsXTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkAxsYTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkAxsZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                  ! Center position of the wake centerline 
               farm%m%AllOuts(WkPosXTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkPosYTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkPosZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                  ! Advection, deflection, and meandering  
                  !  of the wake for downstream wake volum 
               farm%m%AllOuts(WkVelXTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkVelYTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkVelZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                  ! Wake diameter for downstream wake volu 
               farm%m%AllOuts(WkDiamTD(iOutDist,nt)) = 0.0_ReKi
               
               do ir = 1, farm%p%NOutRadii
                  
                     ! Axial and radial wake velocity deficits for radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                  farm%m%AllOuts(WkDfVxTND(ir,iOutDist,nt)) = 0.0_ReKi
                  farm%m%AllOuts(WkDfVrTND(ir,iOutDist,nt)) = 0.0_ReKi
               
                     ! Total eddy viscosity, and individual contributions to the eddy viscosity from ambient turbulence and the shear layer, 
                     !  or radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                  farm%m%AllOuts(EddVisTND(ir,iOutDist,nt)) = 0.0_ReKi
                  farm%m%AllOuts(EddAmbTND(ir,iOutDist,nt)) = 0.0_ReKi
                  farm%m%AllOuts(EddShrTND(ir,iOutDist,nt)) = 0.0_ReKi
                  
               end do  

            else
               
                  ! Find wake volume which contains the user-requested downstream location.
               do np = 0, min(farm%WD(nt)%p%NumPlanes-2 , n)

                  if ( ( farm%p%OutDist(iOutDist) >= farm%WD(nt)%y%x_plane(np) ) .and. ( farm%p%OutDist(iOutDist) < farm%WD(nt)%y%x_plane(np+1) ) ) then   ! A wake volume has been found

                     delta = ( farm%p%OutDist(iOutDist) - farm%WD(nt)%y%x_plane(np) ) / ( farm%WD(nt)%y%x_plane(np+1) - farm%WD(nt)%y%x_plane(np) )
                     deltad = (1.0_ReKi-delta)

                     vec_interp       =  delta*farm%WD(nt)%y%xhat_plane(:, np+1) + deltad*farm%WD(nt)%y%xhat_plane(:, np)
                     norm2_vec        =  TwoNorm( vec_interp ) 
                        ! Orientation of the wake centerline for downstream wake volume, np, of turbine, nt, in the global coordinate system, -
                     farm%m%AllOuts(WkAxsXTD(iOutDist,nt)) = vec_interp(1)/norm2_vec
                     farm%m%AllOuts(WkAxsYTD(iOutDist,nt)) = vec_interp(2)/norm2_vec
                     farm%m%AllOuts(WkAxsZTD(iOutDist,nt)) = vec_interp(3)/norm2_vec 

                     if ( farm%AWAE%m%parallelFlag(np,nt) ) then
                        vec_interp       =  delta*farm%WD(nt)%y%p_plane(:, np+1) + deltad*farm%WD(nt)%y%p_plane(:, np)
                     else
                        vec_interp = delta*farm%AWAE%m%rhat_e(:,np,nt) + deltad*farm%AWAE%m%rhat_s(:,np,nt)
                        vec_interp = delta*farm%AWAE%m%pvec_ce(:,np,nt) + deltad*farm%AWAE%m%pvec_cs(:,np,nt) + ( delta*farm%AWAE%m%r_e(np,nt) + deltad*farm%AWAE%m%r_s(np,nt) )* vec_interp / TwoNorm(vec_interp)
                     end if
               
                        ! Center position of the wake centerline for downstream wake volume, np, of turbine, nt, in the global coordinate system, m
                     farm%m%AllOuts(WkPosXTD(iOutDist,nt)) = vec_interp(1)
                     farm%m%AllOuts(WkPosYTD(iOutDist,nt)) = vec_interp(2)
                     farm%m%AllOuts(WkPosZTD(iOutDist,nt)) = vec_interp(3)

                        ! Advection, deflection, and meandering velocity (not including the horizontal wake-deflection correction) 
                        !  of the wake for downstream wake volume, np, of turbine, nt, in the global coordinate system, m/s
                     farm%m%AllOuts(WkVelXTD(iOutDist,nt)) = delta*farm%AWAE%y%V_plane(1,np+1,nt) + deltad*farm%AWAE%y%V_plane(1,np,nt)
                     farm%m%AllOuts(WkVelYTD(iOutDist,nt)) = delta*farm%AWAE%y%V_plane(2,np+1,nt) + deltad*farm%AWAE%y%V_plane(2,np,nt)
                     farm%m%AllOuts(WkVelZTD(iOutDist,nt)) = delta*farm%AWAE%y%V_plane(3,np+1,nt) + deltad*farm%AWAE%y%V_plane(3,np,nt)

                        ! Wake diameter for downstream wake volume, np, of turbine, nt, m
                     farm%m%AllOuts(WkDiamTD(iOutDist,nt)) = delta*farm%WD(nt)%y%D_wake(np+1) + deltad*farm%WD(nt)%y%D_wake(np)  !farm%AWAE%u%D_wake(np,nt)
            
                  
                     do ir = 1, farm%p%NOutRadii
                  
                           ! Axial and radial wake velocity deficits for radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                        farm%m%AllOuts(WkDfVxTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%y%Vx_wake(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%y%Vx_wake(farm%p%OutRadii(ir),np)
                        farm%m%AllOuts(WkDfVrTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%y%Vr_wake(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%y%Vr_wake(farm%p%OutRadii(ir),np)
               
                           ! Total eddy viscosity, and individual contributions to the eddy viscosity from ambient turbulence and the shear layer, 
                           !  or radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                        farm%m%AllOuts(EddVisTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%m%vt_tot(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%m%vt_tot(farm%p%OutRadii(ir),np)
                        farm%m%AllOuts(EddAmbTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%m%vt_amb(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%m%vt_amb(farm%p%OutRadii(ir),np)
                        farm%m%AllOuts(EddShrTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%m%vt_shr(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%m%vt_shr(farm%p%OutRadii(ir),np)
                  
                     end do  

                  else if ( ( farm%p%OutDist(iOutDist) >= farm%WD(nt)%y%x_plane(np+1) ) .and. ( farm%p%OutDist(iOutDist) < farm%WD(nt)%y%x_plane(np) ) ) then   ! Overlapping wake volumes result in invalid output
               
                     farm%m%AllOuts(WkAxsXTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkAxsYTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkAxsZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                        ! Center position of the wake centerline 
                     farm%m%AllOuts(WkPosXTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkPosYTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkPosZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                        ! Advection, deflection, and meandering  
                        !  of the wake for downstream wake volum 
                     farm%m%AllOuts(WkVelXTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkVelYTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkVelZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                        ! Wake diameter for downstream wake volu 
                     farm%m%AllOuts(WkDiamTD(iOutDist,nt)) = 0.0_ReKi
               
                     do ir = 1, farm%p%NOutRadii
                  
                           ! Axial and radial wake velocity deficits for radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                        farm%m%AllOuts(WkDfVxTND(ir,iOutDist,nt)) = 0.0_ReKi
                        farm%m%AllOuts(WkDfVrTND(ir,iOutDist,nt)) = 0.0_ReKi
               
                           ! Total eddy viscosity, and individual contributions to the eddy viscosity from ambient turbulence and the shear layer, 
                           !  or radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                        farm%m%AllOuts(EddVisTND(ir,iOutDist,nt)) = 0.0_ReKi
                        farm%m%AllOuts(EddAmbTND(ir,iOutDist,nt)) = 0.0_ReKi
                        farm%m%AllOuts(EddShrTND(ir,iOutDist,nt)) = 0.0_ReKi
                  
                     end do  
              
                     exit

                  end if
                  
               end do  

            end if

         end do

      end do
      
      !.......................................................................................
      ! Ambient Wind and Array Effects
      !.......................................................................................
      
         ! Loop over user-requested, velocity locations  
      do iVelPt = 1, farm%p%NWindVel        

            ! Determine the requested pt in grid coordinates
         pt = (/farm%p%WindVelX(iVelPt), farm%p%WindVelY(iVelPt),farm%p%WindVelZ(iVelPt)/)
         pt(1) = (pt(1) - farm%p%X0_low)/ farm%p%dX_low
         pt(2) = (pt(2) - farm%p%Y0_low)/ farm%p%dY_low
         pt(3) = (pt(3) - farm%p%Z0_low)/ farm%p%dZ_low
         
            ! Ambient wind velocity (not including wakes) for point, pt,  in global coordinates (from the low-resolution domain), m/s
         call TrilinearInterpRegGrid(farm%AWAE%m%Vamb_low, pt, (/farm%p%nX_low,farm%p%nY_low,farm%p%nZ_low/), vel)
         farm%m%AllOuts(WVAmbX(iVelPt)) = vel(1)
         farm%m%AllOuts(WVAmbY(iVelPt)) = vel(2)
         farm%m%AllOuts(WVAmbZ(iVelPt)) = vel(3)
         
            ! Disturbed wind velocity (including wakes) for point, pt,  in global coordinates (from the low-resolution domain), m/s
         call TrilinearInterpRegGrid(farm%AWAE%m%Vdist_low, pt, (/farm%p%nX_low,farm%p%nY_low,farm%p%nZ_low/), vel)
         farm%m%AllOuts(WVDisX(iVelPt)) = vel(1)
         farm%m%AllOuts(WVDisY(iVelPt)) = vel(2)
         farm%m%AllOuts(WVDisZ(iVelPt)) = vel(3)
            
              
      end do
      
      

      
      call WriteFarmOutputToFile(t, farm, ErrStat2, ErrMsg2)
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         
   end if
end subroutine Farm_WriteOutput
!---------------------------------------------------------------------------------------------------------------------------------- 
!> This routine calculates outputs at each time increment and solves for the inputs at the next step. 
!! The calculate output algorithm: \n 
!!    -  In parallel:  
!!       1. call WD_CO and transfer y_WD to u_AWAE 
!!       2. call SC_CO and transfer y_SC to u_F 
!!       3. Transfer y_F to u_SC and u_WD 
!!    -  CALL AWAE_CO 
!!    -  Transfer y_AWAE to u_F and u_WD 
!!    -  Write Output to File
subroutine FARM_CalcOutput(t, farm, ErrStat, ErrMsg)
   REAL(DbKi),               INTENT(IN   ) :: t                               !< Current simulation time in seconds
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                          :: nt                    
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_CalcOutput'
   INTEGER(IntKi)                          :: n                               ! time step increment number
!   REAL(DbKi)                              :: tm1
   ErrStat = ErrID_None
   ErrMsg = ""
   
  ! tm1 = omp_get_wtime()
   
   !.......................................................................................
   ! calculate module outputs and perform some input-output solves (steps 1. and 2. and 3. can be done in parallel,
   !  but be careful that step 3 doesn't modify the inputs to steps 1 or 2)
   !.......................................................................................
   
      !--------------------
      ! 1. call WD_CO and transfer y_WD to u_AWAE        
   
   DO nt = 1,farm%p%NumTurbines
      
      call WD_CalcOutput( t, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                     farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)       
         
   END DO
   if (ErrStat >= AbortErrLev) return

   call Transfer_WD_to_AWAE(farm)
   
   if ( farm%p%UseSC ) then

         !--------------------
         ! 3a. Transfer y_F to u_SC, at n+1
      do nt = 1,farm%p%NumTurbines

         farm%SC%uInputs%toSC( (nt-1)*farm%SC%p%NumCtrl2SC + 1 : nt*farm%SC%p%NumCtrl2SC ) = farm%FWrap(nt)%y%toSC    

      end do

      !--------------------
      ! 2. call SC_CO and transfer y_SC to u_F, at n+1 
      call SC_CalcOutput(t, farm%SC%uInputs, farm%SC%p, farm%SC%x, farm%SC%xd, farm%SC%z, &
                           farm%SC%OtherState, farm%SC%y, farm%SC%m, ErrStat2, ErrMsg2 ) 
      
      do nt = 1,farm%p%NumTurbines
            
         farm%FWrap(nt)%u%fromSCglob  = farm%SC%y%fromSCglob
         farm%FWrap(nt)%u%fromSC      = farm%SC%y%fromSC( (nt-1)*farm%SC%p%NumSC2Ctrl + 1 : nt*farm%SC%p%NumSC2Ctrl )
         
      end do
      
   end if
   
      !--------------------
      ! 3b. Transfer y_F to u_WD         
         
   call Transfer_FAST_to_WD(farm)
         
   !.......................................................................................
   ! calculate AWAE outputs and perform rest of input-output solves
   !.......................................................................................
   
      !--------------------
      ! 1. call AWAE_CO 
   call AWAE_CalcOutput( t, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%y, farm%AWAE%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)

      !--------------------
      ! 2. Transfer y_AWAE to u_F  and u_WD   
   call Transfer_AWAE_to_FAST(farm)      
   call Transfer_AWAE_to_WD(farm)   
   
   
   !.......................................................................................
   ! Write Output to File
   !.......................................................................................
      ! NOTE: Visualization data is output via the AWAE module
   n = nint(t/farm%p%DT_low)
   call Farm_WriteOutput(n, t, farm, ErrStat2, ErrMsg2)
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   
 !  write(*,*) 'Total Farm_CO-serial took '//trim(num2lstr(omp_get_wtime()-tm1))//' seconds.' 
   
end subroutine FARM_CalcOutput
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine ends the modules used in this simulation. It does not exit the program.
!!    -  In parallel: 
!!       1. CALL AWAE_End
!!       2. CALL WD_End
!!       3. CALL SC_End
!!       4. CALL F_End
!!    -  Close Output File   
subroutine FARM_End(farm, ErrStat, ErrMsg)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm  
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                          :: nt                    
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_End'
   
   
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
   !.......................................................................................
   ! end all modules (1-4 can be done in parallel) 
   !.......................................................................................
   
      !--------------
      ! 1. end AWAE   
   if (farm%AWAE%IsInitialized) then      
      call AWAE_End( farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%y, farm%AWAE%m, ErrStat2, ErrMsg2 )
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         farm%AWAE%IsInitialized = .false.
   end if      
      
   
      !--------------
      ! 2. end WakeDynamics
   if (allocated(farm%WD)) then
      
      DO nt = 1,farm%p%NumTurbines
         if (farm%WD(nt)%IsInitialized) then      
            call WD_End( farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                         farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, ErrStat2, ErrMsg2 )
               call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
            farm%WD(nt)%IsInitialized = .false.
         end if      
      END DO
      
   end if
   
      !--------------
      ! 3. End supercontroller
                     
   if ( farm%p%useSC ) then
      CALL SC_End(farm%SC%uInputs, farm%SC%p, farm%SC%x, farm%SC%xd, farm%SC%z, farm%SC%OtherState, &
                     farm%SC%y, farm%SC%m, ErrStat2, ErrMsg2)
      farm%SC%IsInitialized = .false.
   end if
   
      !--------------
      ! 4. End each instance of FAST (each instance of FAST can be done in parallel, too)   
   if (allocated(farm%FWrap)) then
      
      DO nt = 1,farm%p%NumTurbines
         if (farm%FWrap(nt)%IsInitialized) then
            CALL FWrap_End( farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                            farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, ErrStat2, ErrMsg2 )
            call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
            farm%FWrap(nt)%IsInitialized = .false.
         end if
      END DO
      
   end if   
   
   !.......................................................................................
   ! close output file
   !.......................................................................................
   call Farm_EndOutput( farm, ErrStat2, ErrMsg2 )
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg,RoutineName)
      
      
   !.......................................................................................
   ! clear all data from 'farm' structure
   !.......................................................................................
   call Farm_DestroyAll_FastFarm_Data( farm, ErrStat2, ErrMsg2 )
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   
end subroutine FARM_End
!----------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE Transfer_FAST_to_WD(farm)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  

   integer(intKi)  :: nt
   
   DO nt = 1,farm%p%NumTurbines   
      farm%WD(nt)%u%xhat_disk      = farm%FWrap(nt)%y%xHat_Disk       ! Orientation of rotor centerline, normal to disk
      farm%WD(nt)%u%p_hub          = farm%FWrap(nt)%y%p_hub           ! Center position of hub, m
      farm%WD(nt)%u%D_rotor        = farm%FWrap(nt)%y%D_rotor         ! Rotor diameter, m
      farm%WD(nt)%u%Vx_rel_disk    = farm%FWrap(nt)%y%DiskAvg_Vx_Rel  ! Rotor-disk-averaged relative wind speed (ambient + deficits + motion), normal to disk, m/s
      farm%WD(nt)%u%Ct_azavg       = farm%FWrap(nt)%y%AzimAvg_Ct      ! Azimuthally averaged thrust force coefficient (normal to disk), distributed radially, -
      farm%WD(nt)%u%YawErr         = farm%FWrap(nt)%y%YawErr          ! Nacelle-yaw error at the wake planes, rad   
   END DO
   
END SUBROUTINE Transfer_FAST_to_WD
!----------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE Transfer_AWAE_to_WD(farm)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  

   integer(intKi)  :: nt
   
   DO nt = 1,farm%p%NumTurbines
      farm%WD(nt)%u%V_plane      = farm%AWAE%y%V_plane(:,:,nt)   ! Advection, deflection, and meandering velocity of wake planes, m/s
      farm%WD(nt)%u%Vx_wind_disk = farm%AWAE%y%Vx_wind_disk(nt)  ! Rotor-disk-averaged ambient wind speed, normal to planes, m/s
      farm%WD(nt)%u%TI_amb       = farm%AWAE%y%TI_amb(nt)        ! Ambient turbulence intensity of wind at rotor disk
   END DO
   
END SUBROUTINE Transfer_AWAE_to_WD
!----------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE Transfer_AWAE_to_FAST(farm)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  

   integer(intKi)  :: nt
   
   DO nt = 1,farm%p%NumTurbines
         ! allocated in FAST's IfW initialization as 3,x,y,z,t
      farm%FWrap(nt)%u%Vdist_High = farm%AWAE%y%Vdist_High(nt)%data
   END DO
   
END SUBROUTINE Transfer_AWAE_to_FAST
!----------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE Transfer_WD_to_AWAE(farm)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  

   integer(intKi)  :: nt
   
   DO nt = 1,farm%p%NumTurbines   
      farm%AWAE%u%xhat_plane(:,:,nt) = farm%WD(nt)%y%xhat_plane     ! Orientations of wake planes, normal to wake planes, for each turbine
      farm%AWAE%u%p_plane(:,:,nt)    = farm%WD(nt)%y%p_plane        ! Center positions of wake planes for each turbine
      farm%AWAE%u%Vx_wake(:,:,nt)    = farm%WD(nt)%y%Vx_wake        ! Axial wake velocity deficit at wake planes, distributed radially, for each turbine
      farm%AWAE%u%Vr_wake(:,:,nt)    = farm%WD(nt)%y%Vr_wake        ! Radial wake velocity deficit at wake planes, distributed radially, for each turbine
      farm%AWAE%u%D_wake(:,nt)       = farm%WD(nt)%y%D_wake         ! Wake diameters at wake planes for each turbine      
   END DO
   
END SUBROUTINE Transfer_WD_to_AWAE
!----------------------------------------------------------------------------------------------------------------------------------
END MODULE FAST_Farm_Subs
!**********************************************************************************************************************************
