
library(tidyverse)
library(cowplot)
library(emmeans)
library(patchwork)
library(scales)

#Read in MOSS earthworm totals
MOSS_EarthwormTotals <- read.csv("RawData/MOSS_EarthwormTotals.csv")

#Convert relevant columns to factors for earthworm totals
cols1 <- c("block", "fence", "quad_type", "quadnum", "tpgapnm", "tpgap_quad") 

MOSS_EarthwormTotals[cols1] <- lapply(MOSS_EarthwormTotals[cols1], factor)

MOSS_EarthwormTotals$tot.afdb <- round(MOSS_EarthwormTotals$tot.afdb, digits = 3)

MOSS_EarthwormTotals$n <- round(MOSS_EarthwormTotals$n, digits = 0)

  
levels(MOSS_EarthwormTotals$fence) <- list(In = "in",
                                          Out = "out")

levels(MOSS_EarthwormTotals$quad_type) <- list(Closed = "ctrl",
                                           Open = "gap")

##Figure 2 ----
#Showing general earthworm biomass and density trends with treatment in MOSS experiment 
#Top Left

MOSS_Fence_Biomass <- MOSS_EarthwormTotals %>% #Top Left
  ggplot(aes(fence, (tot.afdb/0.07), fill = fence)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(1)) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  scale_fill_manual(values=c("honeydew", "red2"), name = "Fencing", labels = c("In", "Out")) +
  stat_summary(fun=mean, geom="point", shape=23, size=2, stroke = 0.85) +
  ylab(expression(atop("Earthworm Biomass", "(grams per m"^"2"~")"))) +
  xlab(label = ~Fence) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        plot.title = element_text(size = 10, family = "Times New Roman", face = "bold"),
        text=element_text(family = "Times New Roman"),
        axis.title.y = element_text(size = 10, colour = "black", face = "bold"),
        axis.text.y = element_text(colour ="black", size = 9),
        axis.title.x=element_blank(),
        axis.text.x=element_blank(),
        axis.ticks.x=element_blank(),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        legend.position = "none")

MOSS_Fence_Biomass

MOSS_Fence_Density <- MOSS_EarthwormTotals %>% #Bottom Left
  ggplot(aes(fence, (n/0.07), fill = fence)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(1)) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun=mean, geom="point", shape=23, size=2, stroke = 0.85) + 
  scale_fill_manual(values=c("honeydew", "red2"), name = "Fence", labels = c("In", "Out")) +
  xlab(label = ~Fence) +
  ylab(expression(atop("Earthworm Density", "(# per m"^"2"~")"))) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        plot.title = element_text(size = 10, face = "bold"),
        axis.title.x = element_text(size = 10, colour = "black", face = "bold", vjust = -0.15),
        axis.title.y = element_text(size = 10, colour = "black", face = "bold"),
        axis.text.x = element_text(colour = "black", size = 9),
        axis.text.y = element_text(colour="black", size = 9),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        legend.position = "none")

MOSS_Fence_Density 

#Top Right

MOSS_Gap_Biomass <- MOSS_EarthwormTotals %>% 
  ggplot(aes(quad_type, (tot.afdb/0.07), fill = quad_type)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(1)) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  scale_fill_manual(values = c("darkolivegreen4", "skyblue1"), labels = c("No Gap", "Gap")) +
  stat_summary(fun=mean, geom="point", shape=23, size=2, stroke = 0.85) +
  ylab(label = Earthworm~Biomass~(g~per~m^{"2"})) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        plot.title = element_text(size = 10, family = "Times New Roman", face = "bold"),
        axis.title = element_text(face="bold", colour = "black"),
        axis.title.x=element_blank(),
        axis.text.x=element_blank(),
        axis.ticks.x=element_blank(),
        axis.title.y=element_blank(),
        axis.text.y=element_blank(),
        axis.ticks.y=element_blank(),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        legend.position = "none")

MOSS_Gap_Biomass

#Bottom Right

GapPresence <- c("Closed", "Open")

MOSS_Density_Gap <- MOSS_EarthwormTotals %>% 
  ggplot(aes(quad_type, (n/0.07), fill = quad_type)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(1)) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) +   
  scale_fill_manual(values = c("darkolivegreen4", "skyblue1"), name = "Canopy") +
  stat_summary(fun=mean, geom="point", shape=23, size=2, stroke = 0.85) +
  xlab(label = ~Canopy) +
  ylab(label = Earthworm~Density~(N~per~m^{"2"})) + 
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        plot.title = element_text(size = 10, family = "Times New Roman", face = "bold"),
        axis.title.x = element_text(face="bold", size = 10, vjust = -0.25),
        axis.text.x = element_text(colour="black", size = 9),
        axis.title.y=element_blank(),
        axis.text.y=element_blank(),
        axis.ticks.y=element_blank(),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        legend.position = "none")

MOSS_Density_Gap 

Fig_2 <- (MOSS_Fence_Biomass + MOSS_Gap_Biomass) / (MOSS_Fence_Density + MOSS_Density_Gap)

jpeg("Fig_2.jpeg", 
     width = 8.5,
     height = 8.5,
     units = "cm",
     res = 600)

plot(Fig_2)

dev.off()

##Figure 3 ----
##Showing dynamics between earthworm species, gaps, and deer in MOSS experiment

#Read in genera-specific earthworm data
MOSS_Aporrectodea <- read.csv("RawData/MOSS_aporrectodea.csv")
MOSS_Dendrobaena <- read.csv("RawData/MOSS_dendrobaena.csv")
MOSS_Lumbricus <- read.csv("RawData/MOSS_lumbricus.csv")

MOSS_Aporrectodea[cols1] <- lapply(MOSS_Aporrectodea[cols1], factor)
MOSS_Dendrobaena[cols1] <- lapply(MOSS_Dendrobaena[cols1], factor)
MOSS_Lumbricus[cols1] <- lapply(MOSS_Lumbricus[cols1], factor)

levels(MOSS_Lumbricus$fence) <- list(In = "in",
                                        Out = "out")

GapPresence <- c("Closed", "Open")

#Top Left

MOSS_Dend_Fence <- ggplot(MOSS_Dendrobaena, aes(fence, (tot.afdb/0.07), fill = fence)) + 
  geom_boxplot(outlier.shape = NA, width = 0.9) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun=mean, geom="point", shape=23, size=2.5, stroke = 0.85) +
  ggtitle("Dendrobaena") +
  xlab(label = ~Fence) +
  ylab(label = ~Dendrobaena~Biomass~(g~per~m^{"2"})) +
  scale_fill_manual(values=c("honeydew", "red2"), name = "Fencing", labels = c("In", "Out")) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        plot.title = element_text(face = "italic", size = 10),
        axis.title.y = element_blank(),
        axis.title.x = element_blank(),
        axis.text.x = element_blank(),
        axis.text.y = element_text(colour = "black", size = 9),
        axis.ticks.y=element_blank(),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        strip.text.x = element_text(size = 10),
        legend.position = "none",
        plot.margin=margin(c(0,0,0,0),unit="cm"))

MOSS_Dend_Fence 

#Top Right

MOSS_Dend_Gap <- MOSS_Dendrobaena %>% 
  ggplot(aes(quad_type, tot.afdb/0.07, fill = quad_type)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(), width = 0.9) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun=mean, geom="point", shape=23, size=2.5, stroke = 0.85) +
  xlab(label = Earthworm~Genera) +
  ylab(label = ~Density~(n~per~m^{"2"})) +
  labs(fill = "Fencing") + 
  scale_fill_manual(values = c("darkolivegreen4", "skyblue1")) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        axis.title = element_text(face="bold", size = 10),
        axis.title.x = element_blank(),
        axis.title.y = element_blank(),
        axis.text.x = element_blank(),
        axis.text.y = element_blank(),
        axis.ticks.y=element_blank(),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        strip.text.x = element_text(size = 10),
        legend.position = "none",
        plot.margin=margin(c(0,0,0,0),unit="cm"))


MOSS_Dend_Gap 

#Middle Left

MOSS_Apo_Fence <- ggplot(MOSS_Aporrectodea, aes(fence, tot.afdb/0.07, fill = fence)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(1), width = 0.9) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun=mean, geom="point", shape=23, size=2.5, stroke = 0.85) +
  ggtitle("Aporrectodea") +
  xlab(label = ~Fence) +
  ylab(expression(atop("Earthworm Biomass", "(grams per m"^"2"~")"))) +
  scale_fill_manual(values=c("honeydew", "red2"), name = "Fencing", labels = c("In", "Out")) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        plot.title = element_text(face = "italic", size = 10),
        axis.title.y = element_text(colour = "black", face="bold", size = 10),
        axis.title.x = element_blank(),
        axis.text.x = element_blank(),
        panel.grid.major.x=element_blank(),
        axis.text.y = element_text(colour = "black", size = 9),
        axis.line = element_line(size=0.5, colour = "black"),
        strip.text.x = element_blank(),
        legend.position = "none")


MOSS_Apo_Fence

#Middle Right

MOSS_Apo_Gap <- MOSS_Aporrectodea %>% 
  ggplot(aes(quad_type, (tot.afdb/0.07), fill=quad_type)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(1), width = 0.9) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun=mean, geom="point", shape=23, size=2.5, stroke = 0.85) +
  xlab(label = ~Fence) +
  ylab(label = ~Biomass~(n~per~m^{"2"})) +
  scale_fill_manual(values = c("darkolivegreen4", "skyblue1")) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        axis.title = element_text(face="bold", size = 10),
        axis.title.x = element_blank(),
        axis.title.y = element_blank(),
        axis.text.x = element_blank(),
        axis.text.y = element_blank(),
        axis.ticks.y=element_blank(),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        strip.text.x = element_text(size = 10),
        legend.position = "none",
        plot.margin=margin(c(0,0,0,0),unit="cm"))

MOSS_Apo_Gap 

#Bottom Left

MOSS_Lumb_Fence <- ggplot(MOSS_Lumbricus, aes(fence, (tot.afdb/0.07), fill = fence)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(1), width = 0.9) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun=mean, geom="point", shape=23, size=2.5, stroke = 0.85) +
  ggtitle("Lumbricus") +
  xlab(label = ~Fence) +
  ylab(label = ~Lumbricus~Biomass~(g~per~m^{"2"})) +
  scale_fill_manual(values=c("honeydew", "red2")) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        plot.title = element_text(face = "italic", size = 10),
        axis.title.y = element_blank(),
        axis.title.x = element_text(size = 10),
        axis.text.x = element_text(colour = "black", size = 9),
        axis.text.y = element_text(size=10, colour = "black"),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        strip.text.x = element_blank(),
        legend.position = "none",
        plot.margin=margin(c(0,0,0,0),unit="cm"))

MOSS_Lumb_Fence 

#Bottom Right

MOSS_Lumb_Gap <- MOSS_Lumbricus %>% 
  ggplot(aes(factor(quad_type), tot.afdb/0.07, fill = quad_type)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge(1), width = 0.9) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun=mean, geom="point", shape=23, size=2.5, stroke = 0.85) +  
  ylab(label = Lumbricus~Biomass~(n~per~m^{"2"})) + 
  xlab(label = ~Canopy) + 
  scale_x_discrete(labels = GapPresence) +
  scale_fill_manual(values = c("darkolivegreen4", "skyblue1")) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        text=element_text(family = "Times New Roman"),
        axis.title.x = element_text(face="bold", size = 10),
        axis.title.y = element_blank(),
        axis.text.x = element_text(colour = "black", size = 10),
        axis.text.y = element_blank(),
        panel.grid.major.x=element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        strip.text.x = element_blank(),
        legend.position = "none",
        plot.margin=margin(c(0,0,0,0),unit="cm"))

MOSS_Lumb_Gap #Bottom Right

Fig_3 <- (MOSS_Dend_Fence + MOSS_Dend_Gap) / (MOSS_Apo_Fence + MOSS_Apo_Gap) / (MOSS_Lumb_Fence + (MOSS_Lumb_Gap))

jpeg("Fig_3.jpeg", 
    width = 12,
    height = 14.5,
    units = "cm",
    res = 600)

plot(Fig_3)

dev.off()

## Figure 4 -----

#Compare earthworm densities between years in various fencing treatments. We use all data here because we are only looking at fencing treatment between years

y0y13_EarthwormDens <- read.csv("RawData/FE_totdense_y0y13.csv")

y0y13_EarthwormDens$count <- round(y0y13_EarthwormDens$count, digits = 0) #Round all count data

cols4 <- c("tpgapnm", "location", "fence", "Year") #Convert vars to factor
y0y13_EarthwormDens[cols4] <- lapply(y0y13_EarthwormDens[cols4], factor)

levels(y0y13_EarthwormDens$fence) <- list(In = "in",
                                           Out = "out")

y0y13_EarthwormDens <- y0y13_EarthwormDens %>% rename(Fence = "fence")

Fig_4a <- y0y13_EarthwormDens %>%
  ggplot(aes(Fence, count, fill = Year)) + 
  geom_boxplot(outlier.shape = NA, position = position_dodge()) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun = mean, position=position_dodge(width = 0.75), geom="point", shape=23, size=2.5, stroke = 0.85) + 
  ylab(expression(atop("Earthworm Density", "(# per m"^"2"~")"))) +  
  scale_fill_manual(values=c("lightgoldenrod1", "lightgoldenrod4")) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        text=element_text(family = "Times New Roman"),
        panel.grid.major.x = element_blank(),
        axis.title = element_text(size = 10),
        axis.title.x = element_text(size = 10, vjust = 12),
        axis.text.x = element_text(colour="black", size = 9),
        axis.text.y = element_text(colour="black", size = 10),
        legend.title = element_text(face = "bold", size=10), 
        legend.text = element_text(size=10))

y0y13_EarthwormDens %>%  
  group_by(Exclosure,Year) %>% 
  summarise(mean.afdb = (mean(count)))


#Compare earthworm densities between years in various gap locations

y0y13_yearlocation <- y0y13_EarthwormDens %>% #Remove south buffer and south gap location as these were not sampled in 2019
  filter(location != "SB" & location != "SG" & location != "C")

y0y13_yearlocation$count <- round(y0y13_yearlocation$count, digits = 0)

levels(y0y13_yearlocation$location) <- list(South.Edge = "ST",
                                           North.Gap = "NG" , 
                                           North.Edge = "NT", 
                                           North.Buffer = "NB", ordered = TRUE)

y0y13_yearlocation$Year <- ordered(y0y13_yearlocation$Year, levels = c(2019, 2006)) #Relevel to get proper order

y0y13_yearlocation <- y0y13_yearlocation %>% 
  rename(Gap.Location = "location")

Fig_4b <- y0y13_yearlocation %>%
  mutate(Gap.Location = fct_relevel(Gap.Location, "North.Buffer", "North.Edge","North.Gap", "South.Edge")) %>%
  ggplot(aes(Gap.Location, count, fill = Year)) +
  geom_boxplot(outlier.shape = NA, position = position_dodge()) +
  geom_point(colour = "black", alpha = 1/2, size = 0.75, shape = 21, position = position_jitterdodge()) + 
  stat_summary(fun = mean, position=position_dodge(width = 0.75), geom="point", shape=23, size=2, stroke = 0.85) +
  scale_x_discrete(labels = function(Gap.Location) str_wrap(Gap.Location, width = 3)) +
  scale_fill_manual(values=c("lightgoldenrod1", "lightgoldenrod4"), breaks=c('2006', '2019')) +
  ylab(expression(atop("Earthworm Density", "(grams per m"^"2"~")"))) +
  xlab(expression("Gap Location")) +
  ylim(19,400) + #Edit scale so that it starts on 0
  guides(x =  guide_axis(angle = 50)) +
  theme_bw() +
  theme(panel.grid.major = element_line(colour = "#d3d3d3"),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank(),
        panel.grid.major.x = element_blank(),
        axis.line = element_line(size=0.5, colour = "black"),
        text=element_text(family = "Times New Roman"),
        axis.title.x = element_text(size = 10),
        axis.text.x = element_text(colour="black", size = 8),
        axis.title.y = element_blank(),
        axis.text.y = element_blank(),
        axis.ticks.x = element_blank(),
        legend.position="none")

Fig_4 <- Fig_4a + Fig_4b

jpeg("Fig_4.jpeg", 
     width = 16,
     height = 10,
     units = "cm",
     res = 600)

plot(Fig_4)

dev.off()



y0y13_fencelocation %>%  
  group_by(location,Year) %>% 
  summarise(mean.afdb = (mean(count)))

#NB: +62.2%
#NT: +36.9%
#CNG: -10.9%
#ST: +28.6

#Table 1 MOSS -----

MOSS_4Genera <- read.csv("RawData/MOSS_4Genera.csv")

Table1_MOSSTotal <- MOSS_EarthwormTotals %>% 
  group_by(block) %>%
  summarise(mean.afdb = ((mean(tot.afdb))/0.07),
            mean.density = ((mean(n))/0.07))

Table1_MOSSTotal

Table1_MOSSGenera <- MOSS_4Genera %>% 
  group_by(block, spp.group) %>% 
  summarise(mean.afdb = (mean(tot.afdb)/0.07),
            mean.density = (mean(count)/0.07))

Table1_MOSSGenera

#Table 1 FE -----

Table1_FETotal <- FE_EarthwormTotals %>%  #Note, these totals include Octolasion
  summarise(mean.afdb = ((mean(tot.afdb))/0.07),
            mean.density = ((mean(n))/0.07))

Table1_FETotal

FE_4Genera <- read.csv("RawData/FE_4Genera.csv")

Table1_FETotal <- FE_4Genera %>%
  group_by(spp.group) %>% 
  summarise(mean.afdb = ((mean(tot.afdb))/0.07),
            mean.density = ((mean(n))/0.07))

Table1_FETotal

