#Reed et al. 2023 Linked Disturbance Data Analysis ----

library(tidyverse)
library(lme4)
library(glmmTMB)
library(emmeans)
library(lmerTest)
library(DHARMa)
library(car)
library(openxlsx)
library(sjPlot)
library(performance)

##MOSS ----

#Read in MOSS earthworm totals
MOSS_EarthwormTotals <- read.csv("RawData/MOSS_EarthwormTotals.csv")

#Convert relevant columns to factors for earthworm totals
cols1 <- c("block", "fence", "quad_type", "quadnum", "tpgapnm", "tpgap_quad") 
MOSS_EarthwormTotals[cols1] <- lapply(MOSS_EarthwormTotals[cols1], factor)

###Earthworm Totals ----

####MOSS Biomass ----

#Total earthworm biomass response to deer exclosure and gaps

MOSS_EarthwormTotals$tot.afdb <- round(MOSS_EarthwormTotals$tot.afdb, digits = 3)

MOSS_Biomass <- lmer((tot.afdb)^(1/3) ~ fence*quad_type + (1|block/tpgapnm/quadnum),
                     data = MOSS_EarthwormTotals)

anova(MOSS_Biomass, type = 3, ddf="Kenward-Roger")
sim = simulateResiduals(MOSS_Biomass)
plot(sim)

emmeans(MOSS_Biomass, specs = pairwise ~ fence:quad_type, p.adjust.method = "bonferroni")

MOSS_EarthwormTotals %>%  
  group_by(fence) %>% 
  summarise(mean.afdb = (mean(tot.afdb)))

MOSS_EarthwormTotals %>%  
  group_by(quad_type) %>% 
  summarise(mean.afdb = (mean(tot.afdb)))

####MOSS Density ----

#Total earthworm density response to deer exclosure and gaps. Modeled using a poisson GLMM as this is non-negative count data. Minor in the homogeneity of variance

MOSS_Density <- glmmTMB(n ~ fence*quad_type + (1|block/tpgapnm/quadnum),
                        family = poisson(link = "sqrt"), 
                        data = MOSS_EarthwormTotals)

Anova(MOSS_Density, type = 3)
sim = simulateResiduals(MOSS_Density)
plot(sim)

MOSS_EarthwormTotals %>%  
  group_by(fence) %>% 
  summarise(mean.n = (mean(n)))

MOSS_EarthwormTotals %>%  
  group_by(quad_type) %>% 
  summarise(mean.n = (mean(n)))

###Genera Specific Analysis ----

###Read in genera-specific earthworm biomass response to treatments and convert relevant columns to factors###

MOSS_Aporrectodea <- read.csv("RawData/MOSS_aporrectodea.csv")
MOSS_Dendrobaena <- read.csv("RawData/MOSS_dendrobaena.csv")
MOSS_Lumbricus <- read.csv("RawData/MOSS_lumbricus.csv")

MOSS_Aporrectodea[cols1] <- lapply(MOSS_Aporrectodea[cols1], factor)
MOSS_Dendrobaena[cols1] <- lapply(MOSS_Dendrobaena[cols1], factor)
MOSS_Lumbricus[cols1] <- lapply(MOSS_Lumbricus[cols1], factor)


####MOSS APORRECTODEA ----

#Round all biomass to 3 decimal points

MOSS_Aporrectodea$tot.afdb <- round(MOSS_Aporrectodea$tot.afdb, digits = 3)

#Test Aporrectodea biomass response to gaps and fencing using a zero-inflated glmm with a tweedie distribution

MOSS_AporBiomass <- glmmTMB(tot.afdb ~ quad_type*fence + (1|block/tpgapnm/quadnum), 
                            family = tweedie(),
                            ziformula = ~1,
                            data = MOSS_Aporrectodea)

Anova(MOSS_AporBiomass, type = 3)
sim = simulateResiduals(MOSS_AporBiomass)
plot(sim)

MOSS_Aporrectodea %>%  
  group_by(fence) %>% 
  summarise(mean.afdb = (mean(tot.afdb)))


#Test Aporrectodea density response to gap and fencing using a zero-inflated glmm with a poisson distribution

MOSS_Aporrectodea$count <- round(MOSS_Aporrectodea$count, digits = 0)

MOSS_AporDens <- glmmTMB(count ~ quad_type*fence + (1|block/tpgapnm/quadnum), 
               family = poisson(link = "log"),
               ziformula = ~1,
               data =  MOSS_Aporrectodea)

Anova(MOSS_AporDens, type = 3)
sim = simulateResiduals(MOSS_AporDens)
plot(sim)

emmeans(MOSS_AporDens, specs = pairwise ~ fence|quad_type, p.adjust.method = "bonferroni")

MOSS_Aporrectodea %>%  
  group_by(fence) %>% 
  summarise(mean.n = (mean(count)))

MOSS_Aporrectodea %>%  
  group_by(quad_type, fence) %>% 
  summarise(mean.n = (mean(count)))

####MOSS DENDROBAENA ----

#Round all biomass to 3 decimal points

MOSS_Dendrobaena$tot.afdb <- round(MOSS_Dendrobaena$tot.afdb, digits = 3)

#Test Dendrobaena biomass response to gap and presence

MOSS_DendBiomass <- lmer(tot.afdb^(1/3) ~ quad_type*fence + (1|block/tpgapnm/quadnum), 
              data = MOSS_Dendrobaena)

anova(MOSS_DendBiomass)
sim = simulateResiduals(MOSS_DendBiomass)
plot(sim)

MOSS_Dendrobaena %>%  
  group_by(quad_type) %>% 
  summarise(mean.afdb = (mean(tot.afdb)))


#Test Dendrobaena density response to gap and presence

MOSS_DendDens <- glmmTMB(count ~ quad_type*fence + (1|block/tpgapnm/quadnum),
               family = poisson(link = "log"),
               data = MOSS_Dendrobaena)

Anova(MOSS_DendDens, type = 3)
sim = simulateResiduals(MOSS_DendDens)
plot(sim)

MOSS_Dendrobaena %>%  
  group_by(quad_type) %>% 
  summarise(mean.n = (mean(count)))


####MOSS LUMBRICUS ----

#Test Lumbricus biomass response to gap and presence

MOSS_Lumbricus$tot.afdb <- round(MOSS_Lumbricus$tot.afdb, digits = 3)

MOSS_LumbBiomass <- lmer(tot.afdb^(1/3) ~ quad_type*fence + (1|block/tpgapnm/quadnum), 
                            data = MOSS_Lumbricus)

anova(MOSS_LumbBiomass, ddf="Kenward-Roger")
sim = simulateResiduals(MOSS_LumbBiomass)
plot(sim)

MOSS_Lumbricus %>%  
  group_by(fence) %>% 
  summarise(mean.afdb = (mean(tot.afdb)))

MOSS_Lumbricus %>%  
  group_by(quad_type) %>% 
  summarise(mean.afdb = (mean(tot.afdb)))

#Test Lumbricus density response to gap and deer presence

MOSS_LumbDens <- glmmTMB(count ~ quad_type*fence + (1|block/tpgapnm/quadnum),
                         family = poisson(link = "sqrt"),
                         data = MOSS_Lumbricus)

Anova(MOSS_LumbDens, type = 3)
sim = simulateResiduals(MOSS_LumbDens)
plot(sim)

MOSS_Lumbricus %>%  
  group_by(fence) %>% 
  summarise(mean.n = (mean(count)))

MOSS_Lumbricus %>%  
  group_by(quad_type) %>% 
  summarise(mean.n = (mean(count)))


##The Flambeau Experiment ----

###Earthworm Totals ----

#Read in FE earthworm totals
FE_EarthwormTotals <- read.csv("RawData/FE_EarthwormTotals.csv")

#Convert relevant columns to factors for earthworm totals
cols2 <- c("block", "tpgapnm", "tpgap_quad", "quadnum", "fence", "location") 
FE_EarthwormTotals[cols2] <- lapply(FE_EarthwormTotals[cols2], factor)

#Test how earthworm biomass and density respond to disturbance treatments

####FE Biomass -----

FE_EarthwormTotals$tot.afdb <- round(FE_EarthwormTotals$tot.afdb, digits = 3)

FE_TotBiomass <- lmer(tot.afdb ~ fence*location + (1|tpgapnm), #Biomass
            data = FE_EarthwormTotals)

anova(FE_TotBiomass, type = 3, ddf = "Kenward-Roger")
sim = simulateResiduals(FE_TotBiomass)
plot(sim)

FE_EarthwormTotals %>%  
  group_by(fence) %>% 
  summarise(mean.afdb = (mean(tot.afdb)))

FE_EarthwormTotals %>%  
  group_by(fence, quadnum) %>% 
  summarise(mean.afdb = (mean(tot.afdb)))

####FE Density -----

FE_TotDensity <- glmmTMB(n ~ location*fence + (1|tpgapnm), #Density
               family = poisson(link = "log"),
               ziformula = ~0,
               data = FE_EarthwormTotals)

Anova(FE_TotDensity, type = 3)
sim = simulateResiduals(FE_TotDensity)
plot(sim)

emmeans(FE_TotDensity, specs = pairwise ~ fence|location, p.adjust.method = "bonferroni")

FE_EarthwormTotals %>%  
  group_by(fence) %>% 
  summarise(mean.count = (mean(n)))

FE_EarthwormTotals %>%  
  group_by(location, fence) %>% 
  summarise(mean.n = (mean(n)))

###Genera specific analysis -----

#Read in FE genera-specific earthworm biomass response to treatments and convert relevant columns to factors

FE_Aporrectodea <- read.csv("RawData/FE_aporrectodea.csv")
FE_Dendrobaena <- read.csv("RawData/FE_dendrobaena.csv")
FE_Lumbricus <- read.csv("RawData/FE_lumbricus.csv")

cols3 <- c("tpgapnm", "tpgap_quad", "fence", "quadnum", "location") 

FE_Aporrectodea[cols3] <- lapply(FE_Aporrectodea[cols3], factor)
FE_Dendrobaena[cols3] <- lapply(FE_Dendrobaena[cols3], factor)
FE_Lumbricus[cols3] <- lapply(FE_Lumbricus[cols3], factor)


####FE APORRECTODEA ----

#Round all biomass to 3 decimal points

FE_Aporrectodea$tot.afdb <- round(FE_Aporrectodea$tot.afdb, digits = 3)

FE_AporBiomass <- lmer(tot.afdb ~ location*fence + (1|tpgapnm), 
                       FE_Aporrectodea)

anova(FE_AporBiomass, ddf = "Kenward-Roger")
sim = simulateResiduals(FE_AporBiomass)
plot(sim)

FE_Aporrectodea$n <- round(FE_Aporrectodea$n, digits = 0)

FE_AporDensity <- glmmTMB(n ~ location*fence + (1|tpgapnm),
                          family = poisson("log"),
                          ziformula = ~0,
                          FE_Aporrectodea)

Anova(FE_AporDensity, type = 3)
sim = simulateResiduals(FE_AporDensity) 
plot(sim) #No discernable difference in residuals. Error in qgam package as per Florian Hartig leads to funky lines

emmeans(FE_AporDensity, specs = pairwise ~ fence|location, p.adjust.method = "bonferroni")

FE_Aporrectodea %>%  
  group_by(fence) %>% 
  summarise(mean.count = (mean(n)))

FE_Aporrectodea %>%  
  group_by(location, fence) %>% 
  summarise(mean.count = (mean(n)))

testZeroInflation(FE_AporDensity)


####FE DENDROBAENA ----

#Round all biomass to 3 decimal points

FE_DendBiomass <- lmer(tot.afdb^(1/2) ~ location*fence + (1|tpgapnm), #Biomass
              FE_Dendrobaena)

anova(FE_DendBiomass, ddf = "Kenward-Roger")
sim = simulateResiduals(FE_DendBiomass)
plot(sim)

FE_Dendrobaena$n <- round(FE_Dendrobaena$n, digits = 0)

FE_DendDensity <- glmmTMB(n ~ location*fence + (1|tpgapnm), #Density
               family = poisson(link = "log"),
               FE_Dendrobaena)

Anova(FE_DendDensity, type = 3)
sim = simulateResiduals(FE_DendDensity)
plot(sim)


####FE LUMBRICUS -----

#Round all biomass to 3 decimal points
FE_Lumbricus$tot.afdb <- round(FE_Lumbricus$tot.afdb, digits = 3)

FE_LumbBiomass <- lmer(tot.afdb ~ location*fence + (1|tpgapnm),
              FE_Lumbricus)

anova(FE_LumbBiomass, ddf = "Kenward-Roger")
sim = simulateResiduals(b.3.1)
plot(sim)

FE_Lumbricus$n <- round(FE_Lumbricus$n, digits = 0)

FE_LumbDens <- glmmTMB(n ~ location*fence + (1|tpgapnm),
               family = poisson("log"),
               FE_Lumbricus)

Anova(FE_LumbDens, type = 3)
sim = simulateResiduals(FE_LumbDens)
plot(sim)

FE_Lumbricus %>%  
  group_by(fence) %>% 
  summarise(mean.count = (mean(n)))

emmeans(FE_LumbDens, specs = pairwise ~ fence|location, p.adjust.method = "bonferroni")

FE_Lumbricus %>%  
  group_by(location, fence) %>% 
  summarise(mean.count = (mean(n)))


###FE Temporal Analysis ----

#Test differences in earthworm density over time within the FE

y0y13_EarthwormDens <- read.csv("RawData/FE_totdense_y0y13.csv")

y0y13_EarthwormDens$count <- round(y0y13_EarthwormDens$count, digits = 0) #Round all count data

cols4 <- c("tpgapnm", "location", "fence", "Year") #Convert vars to factor

y0y13_EarthwormDens[cols4] <- lapply(y0y13_EarthwormDens[cols4], factor)

####Year*Fencing -----

#Test if there are differences between 2006 and 2019 fencing treatments

y0y13_fence <- glmmTMB(count ~ Year*fence + (1|tpgapnm/location), 
                       family = poisson(link = "sqrt"),
                       data = y0y13_EarthwormDens)

Anova(y0y13_fence, type = 3) 
sim = simulateResiduals(y0y13_fence)
plot(sim)

emmeans(y0y13_fence, specs = pairwise ~ Year|fence, p.adjust.method = "bonferroni")

y0y13_EarthwormDens %>%  
  group_by(fence, Year) %>% 
  summarise(mean.afdb = (mean(count)))

####Year&GapLocation -----

#Test if there are differences between 2006 and 2019 gap locations

y0y13_gaplocation <- y0y13_EarthwormDens %>% 
  filter(location != "C" & location != "SB" & location != "SG") #Remove SB and SG plot locations as these were not sampled in 2019. Remove C plot location because this was not sampled in 2006

y0y13_m1 <- glmmTMB(count ~ Year*location + (1|tpgapnm/location), #Location included as random effect to account for repeated measure
                             family = poisson(link = "log"),
                             data = y0y13_gaplocation)

Anova(y0y13_m1, type = 3)
sim = simulateResiduals(y0y13_m1)
plot(sim)

emmeans(y0y13_m1, specs = pairwise ~ Year|location, p.adjust.method = "bonferroni")

y0y13_gaplocation %>%  
  group_by(location, Year) %>% 
  summarise(mean.afdb = (mean(count)))

citation("vegan")
