#!/usr/bin/env python3

import sys, os, math
import numpy as np
import matplotlib.pyplot as plt
import re
plt.rcParams.update({'font.size': 14})

def getmobility(file):
    fname = file#+'/mobility.txt'
    file1 = open(fname, 'r')
    Lines = file1.readlines()
    time = 0
    XX  = []
    fid = 0
    plusend = 0
    minusend = 0
    dt = 0
    for line in Lines:
        if not line :#or line[0] == '%':
            continue
        s = uncode(line).split()
        if len(s) <2:
            pass
        elif line[0] == '%' and s[1] == 'time':
            time =float(s[2])  #2nd column is no of frame
        elif line[0] != '%' :
            dt = float(s[0])
            mt = int(s[1])
            dis_parallel = float(s[2])
            dis_perpend1 = float(s[3])
            dis_perpend2 = float(s[4])
            angle = float(s[5])
            msd = float(s[6])
            XX.append([time,dis_parallel, dis_perpend1, dis_perpend2, angle, msd])
    return np.array(XX),dt

def uncode(arg):
    try:
        return arg.decode('utf-8')
    except:
        return arg

def plot(out,dt):
    time = out[:,0]
    dis_parallel = np.array(out[:,1])/dt
    dis_perpend1 = np.array(out[:,2])/dt
    dis_perpend2 = np.array(out[:,3])/dt
    plt.plot(time,dis_parallel,'o-',color='green',label='parallel to MT')
    plt.plot(time,dis_perpend1,'o-',color='cyan',label='orthogonal')
    plt.plot(time,dis_perpend2,'o-',color='orange',label='vertical')
    plt.legend()
    plt.title('speed of minus end, dt = '+str(dt)+' s \n mean_pa={:.2f},mean_orth={:.2f},mean_vert={:.2f}\n sum1={:.1f},sum2={:.1f},sum3={:.1f}'.format(np.mean(dis_parallel[1::]),np.mean(dis_perpend1[1::]),np.mean(dis_perpend2[1::]),np.sum(dis_parallel[1::]),np.sum(dis_perpend1[1::]),np.sum(dis_perpend2[1::])))
    plt.ylabel('speed, um/s')
    plt.xlabel('time,s')
    plt.grid(alpha=0.5, linestyle='-', linewidth=2)
    plt.tight_layout()
    file = path+'/speed_par_'+str(int(dt))
    np.save(file,np.array([time,dis_parallel]))
    return

def plotangle(out,dt):
    time = out[:,0]
    angle = out[:,4]
    plt.plot(time, angle)
    plt.title('mean rotated angle of MTs, dt = {:.1f}, mean={:.1f}'.format(dt,np.mean(angle[100::])))
    plt.xlabel('time,s')
    plt.ylabel('angle, rad')
    
def plotmsd(dts,msd):
    plt.plot(dts,msd,'o')
    xfit = np.log(dts)
    yfit = np.log(msd)
    z,cov = np.polyfit(xfit, yfit, 1,cov=True)
    m,c = z
    vf = np.exp(c/m)
    plt.loglog(dts,[(t*vf)**m for t in dts],'--',label='exponent={:.2f},v={:.2f}'.format(m,vf))
    plt.legend()
    plt.ylabel('msd, um^2')
    plt.xlabel('time,s') 

def plotspeed(dts,disp_p,speeds,path):
    plt.figure()
    plt.plot(dts,np.array(disp_p),'o-')
    plt.xlabel('dt')
    plt.ylabel('disp_parallel')
    plt.legend()
    plt.grid(alpha=0.5, linestyle='-', linewidth=2)    
    plt.savefig(path+'/disp_par.png')    
    
    plt.figure()
    plt.plot(dts,np.array(speeds),'o-')
    plt.ylim([-0.1,0.1])
    plt.xlabel('dt')
    plt.ylabel('disp_parallel/dt')
    plt.legend()
    plt.title('mean of last two poins={:.3f}'.format(np.mean(speeds[-2::])))
    plt.grid(alpha=0.5, linestyle='-', linewidth=2)
    plt.savefig(path+'/speed_par.png')        

if __name__ == "__main__":
    path = sys.argv[1]
    disp_p = []
    speeds = []
    dts = []
    msd = []
    period = [1,2,5,10,20,40,80,160]
    for i in period:
        inputdata = path+'/mobility'+str(i)+'.txt'
        out,dt = getmobility(inputdata)
        msd.append(np.mean(out[:,5]))
        dts.append(dt)
        disp_p.append(np.mean(out[:,1][1::]))
        speeds.append(np.mean(out[:,1][1::])/dt)
        plt.figure()
        plot(out,dt)
        plt.savefig(path+'/mobility{:.3g}.png'.format(i))
    plt.figure()
    #plotmsd(dts,msd)
    plotspeed(dts,disp_p,speeds,path)

