// Cytosim was created by Francois Nedelec. Copyright 2021 Cambridge University.

#include "assert_macro.h"
#include "point_grid.h"
#include "exceptions.h"
#include "messages.h"
#include "modulo.h"
#include "space.h"
#include "meca.h"

extern Modulo const* modulo;

//------------------------------------------------------------------------------

PointGrid::PointGrid()
{
}


size_t PointGrid::setGrid(Space const* spc, real min_width)
{
    assert_true( min_width > 0 );
    Vector inf, sup;
    spc->boundaries(inf, sup);
    
    size_t cnt[3] = { 1, 1, 1 };
    for ( size_t d = 0; d < DIM; ++d )
    {
        // minimum number of cells in dimension 'd'
        int n = std::floor(( sup[d] - inf[d] ) / min_width);
        
        if ( n < 0 )
            throw InvalidParameter("invalid space:boundaries");
        
#if GRID_HAS_PERIODIC
        assert_true( modulo == spc->getModulo() );
        if ( modulo  &&  modulo->isPeriodic(d) )
        {
            assert_small( modulo->period_[d] - sup[d] + inf[d] );
            // adjust the grid to match the edges
            cnt[d] = std::max(1, n);
            pGrid.setPeriodic(d, true);
        }
        else
#endif
        {
            //extend in any dimension that is not periodic, adjusting the cell size to min_width
            cnt[d] = n + 1;
            real w = cnt[d] * 0.5 * min_width;
            real m = inf[d] + sup[d];
            inf[d] = m - w;
            sup[d] = m + w;
        }
    }
    
    pGrid.setDimensions(inf, sup, cnt);
    return pGrid.nbCells();
}


void PointGrid::createCells()
{
    pGrid.createCells();
    
    //Create side regions suitable for pairwise interactions:
    pGrid.createSideRegions(1);
    
    //report the grid size used
    pGrid.printSummary(Cytosim::log, "   StericGrid");
}


size_t PointGrid::capacity() const
{
    size_t res = 0;
    for ( size_t i = 0; i < pGrid.nbCells(); ++i )
        res += pGrid[i].capacity();
    return res;
}


//------------------------------------------------------------------------------
#pragma mark -


#if ( NUM_STERIC_PANES != 1 )

void PointGrid::add(size_t pan, Mecable const* mec, size_t inx, real rad, real rge) const
{
    if ( pan == 0 || pan > NUM_STERIC_PANES )
        throw InvalidParameter("point:steric is out-of-range");
    
    Vector w = mec->posPoint(inx);
    point_list(w, pan).emplace(Mecapoint(mec, inx), rad, rge, w);
}


void PointGrid::add(size_t pan, Fiber const* fib, size_t inx, real rad, real rge, real sup) const
{
    if ( pan == 0 || pan > NUM_STERIC_PANES )
        throw InvalidParameter("line:steric is out-of-range");
    
    // link in the cell containing the middle of the segment:
    Vector w = fib->posPoint(inx, 0.5);
    locus_list(w, pan).emplace(FiberSegment(fib, inx), rad, rge, sup, w);
}


#endif


//------------------------------------------------------------------------------
#pragma mark - Check two Objects: P = Point; L = Line segment


/**
 This is used to check two spherical objects:
 Solid/Bead/Sphere or the terminal vertex (the tips) of a Fiber
 
 The force is applied if the objects are closer than the
 sum of their radiuses.
 */
void PointGrid::checkPP(Meca& meca, Stiffness const& pam,
                        FatPoint const& aa, FatPoint const& bb)
{
    //std::clog << "   PP- " << bb.pnt_ << " " << aa.pnt_ << '\n';
    Vector vab = bb.cen() - aa.cen();
    const real len = aa.rad_ + bb.rad_;
#if GRID_HAS_PERIODIC
    if ( modulo )
        modulo->fold(vab);
#endif
    real ab2 = vab.normSqr();
    if ( ab2 < len*len )
        meca.addLongLink1(aa.pnt_, bb.pnt_, vab, ab2, len, pam.push);
}


/**
 This is used to check a segment of a fiber against a spherical object:
 Solid/Bead/Sphere or Fiber-tip.
 
 The force is applied if the objects are closer than the sum of their radiuses.
 */
void PointGrid::checkPL(Meca& meca, Stiffness const& pam,
                        FatPoint const& aa, FatLocus const& bb)
{
    //std::clog << "   PL- " << bb.seg_ << " " << aa.pnt_ << '\n';
    const real ran = aa.rad_ + bb.rad_;
    
    // get position of point with respect to segment:
    real ab2 = INFINITY;
    real abs = bb.seg_.projectPoint0(aa.cen(), ab2);
    
    if ( 0 <= abs )
    {
        if ( abs <= bb.len() )
        {
            // the point projects inside the segment
            if ( ab2 < ran*ran )
                meca.addSideSlidingLink(bb.seg_, abs, aa.pnt_, ran, pam.push);
        }
        else
        {
            // the point projects right past the segment, and we check the fiber tip
            if ( bb.isLast() )
            {
                Vector vab = bb.pos2() - aa.cen();
#if GRID_HAS_PERIODIC
                if ( modulo )
                    modulo->fold(vab);
#endif
                ab2 = vab.normSqr();
                if ( ab2 < ran*ran )
                    meca.addLongLink1(aa.pnt_, bb.vertex2(), vab, ab2, ran, pam.push);
            }
        }
    }
    else
    {
        
        Vector vab = bb.pos1() - aa.cen();
#if GRID_HAS_PERIODIC
        if ( modulo )
            modulo->fold(vab);
#endif
        ab2 = vab.normSqr();
        /*
         This code handles the interactions will all the joints of a fiber:
         interact with the node only if this projects on the previous segment
         or if this is the terminal point of a fiber.
         */
        if ( ab2 < ran*ran && ( bb.isFirst() || dot(vab, bb.prevDiff()) <= 0 ))
            meca.addLongLink1(aa.pnt_, bb.vertex1(), vab, ab2, ran, pam.push);
    }
}


/**
 This is used to check a segment of a fiber against another segment of fiber,
 not including the terminal vertex of fibers.
 
 The interaction is applied only if the vertex projects 'inside' the segment.
 */
void PointGrid::checkLL1(Meca& meca, Stiffness const& pam,
                         FatLocus const& aa, FatLocus const& bb)
{
    //std::clog << "   LL1 " << aa.seg_ << " " << bb.vertex1() << '\n';
    const real ran = aa.rge_ + bb.rad_;
    
    // get position of bb.vertex1() with respect to segment 'aa'
    real dis2 = INFINITY;
    real abs = aa.seg_.projectPoint0(bb.pos1(), dis2);
    
    if ((0 <= abs) & (abs <= aa.len()) & (dis2 < ran*ran))
    {
        /*
         bb.vertex1() projects inside segment 'aa'
         */
        const real len = aa.rad_ + bb.rad_;
        real stiff = sign_select(dis2-len*len, pam.push, pam.pull);
        meca.addSideSlidingLink(aa.seg_, abs, bb.vertex1(), len, stiff);
    }
    else if ( abs < 0 )
    {
        if ( aa.isFirst() )
        {
            /*
             Check the projection of aa.vertex1(),
             on the segment represented by 'bb'
             */
            if ( &bb < &aa  &&  bb.isFirst() )
            {
                Vector vab = bb.pos1() - aa.pos1();
#if GRID_HAS_PERIODIC
                if ( modulo )
                    modulo->fold(vab);
#endif
                real ab2 = vab.normSqr();
                real len = aa.rad_ + bb.rad_;
                if ( ab2 < len*len  &&  dot(vab, bb.diff()) >= 0 )
                    meca.addLongLink1(aa.vertex1(), bb.vertex1(), vab, ab2, len, pam.push);
            }
        }
        else
        {
            /*
             Check the projection to the segment located before 'aa',
             and interact if 'bb.vertex1()' falls on the right side of it
             */
            Vector vab = bb.pos1() - aa.pos1();
#if GRID_HAS_PERIODIC
            if ( modulo )
                modulo->fold(vab);
#endif
            if ( dot(vab, aa.prevDiff()) >= 0 )
            {
                real ab2 = vab.normSqr();
                if ( ab2 < ran*ran )
                {
                    real len = aa.rad_ + bb.rad_;
                    real stiff = sign_select(ab2-len*len, pam.push, pam.pull);
                    meca.addLongLink2(aa.vertex1(), bb.vertex1(), vab, ab2, len, stiff);
                }
            }
        }
    }
}


/**
 This is used to check a segment of a fiber against the terminal vertex of a fiber
 
 The interaction is applied only if the vertex projects 'inside' the segment.
 */
void PointGrid::checkLL2(Meca& meca, Stiffness const& pam,
                         FatLocus const& aa, FatLocus const& bb)
{
    //std::clog << "   LL2 " << aa.seg_ << " " << bb.vertex2() << '\n';
    const real ran = aa.rge_ + bb.rad_;
    
    // get position of bb.vertex2() with respect to segment 'aa'
    real dis2 = INFINITY;
    real abs = aa.seg_.projectPoint0(bb.pos2(), dis2);
    
    if ((0 <= abs) & (abs <= aa.len()))
    {
        /*
         bb.vertex2() projects inside segment 'aa'
         */
        if ( dis2 < ran*ran )
        {
            const real len = aa.rad_ + bb.rad_;
            real stiff = sign_select(dis2-len*len, pam.push, pam.pull);
            meca.addSideSlidingLink(aa.seg_, abs, bb.vertex2(), len, stiff);
        }
    }
    else if ( abs < 0 )
    {
        /*
         Check the projection to the segment located before 'aa',
         and interact if 'bb.vertex1()' falls on the right side of it
         */
        Vector vab = bb.pos2() - aa.pos1();
#if GRID_HAS_PERIODIC
        if ( modulo )
            modulo->fold(vab);
#endif
        if ( aa.isFirst() )
        {
            assert_true(bb.isLast());
            real ab2 = vab.normSqr();
            real len = aa.rad_ + bb.rad_;
            if ( ab2 < len*len  && dot(vab, bb.diff()) <= 0 )
                meca.addLongLink1(aa.vertex1(), bb.vertex2(), vab, ab2, len, pam.push);
        }
        else
        {
            if ( dot(vab, aa.prevDiff()) >= 0 )
            {
                real ab2 = vab.normSqr();
                if ( ab2 < ran*ran )
                {
                    real len = aa.rad_ + bb.rad_;
                    real stiff = sign_select(ab2-len*len, pam.push, pam.pull);
                    meca.addLongLink2(aa.vertex1(), bb.vertex2(), vab, ab2, len, stiff);
                }
            }
        }
    }
    else if (( &bb < &aa ) & aa.isLast() )
    {
        /*
         Check the projection of aa.vertex2(),
         on the segment represented by 'bb'
         */
        assert_true(abs > aa.len());
        assert_true(bb.isLast());
        
        Vector vab = bb.pos2() - aa.pos2();
#if GRID_HAS_PERIODIC
        if ( modulo )
            modulo->fold(vab);
#endif
        real ab2 = vab.normSqr();
        real len = aa.rad_ + bb.rad_;
        if ( ab2 < len*len  &&  dot(vab, bb.diff()) <= 0 )
            meca.addLongLink1(aa.vertex2(), bb.vertex2(), vab, ab2, len, pam.push);
    }
}


/**
 This is used to check two FiberSegment, each representing the segment of a Fiber.
 in 3D, the segments are tested for getting within the requested distance.
 in 2D, only the vertices are checked.
 */
void PointGrid::checkLL(Meca& meca, Stiffness const& pam,
                        FatLocus const& aa, FatLocus const& bb)
{
#if ( DIM >= 3 )
    
    const real ran = std::max(bb.rad_+aa.rge_, aa.rad_+bb.rge_);

    /* in 3D, check the shortest distance between two segments, and if close
     enough, use the result to build an interaction */
    real a, b, dis2 = INFINITY;
    if ( ! aa.seg_.belowDistance(bb.seg_, ran, a, b, dis2) )
        return;
    
    if ( aa.seg_.within(a) & bb.seg_.within(b) )
    {
        const real len = aa.rad_ + bb.rad_;
        real stiff = sign_select(dis2-len*len, pam.push, pam.pull);
        meca.addSideSlidingLink(aa.seg_, a, Interpolation(bb.seg_, b), len, stiff);
    }
#endif
    
    //std::clog << "LL " << aa.seg_ << " " << bb.seg_ << '\n';
    checkLL1(meca, pam, aa, bb);
    
    if ( aa.isLast() )
        checkLL2(meca, pam, bb, aa);
    
    checkLL1(meca, pam, bb, aa);
    
    if ( bb.isLast() )
        checkLL2(meca, pam, aa, bb);
}


//------------------------------------------------------------------------------
#pragma mark - Functions to exclude certain pairs from Sterics

/*
 In general, these test will only exclude relatively rare pairs from interacting,
 and thus are less stringent than BigVector::near(): they should be tested after.
 */

/// excluding two spheres when they are from the same Solid
inline static bool not_adjacent(FatPoint const* a, FatPoint const* b)
{
    return a->pnt_.mecable() != b->pnt_.mecable();
}


/// excluding Fiber and Solid from the same Aster
inline static bool not_adjacent(FatPoint const* a, FatLocus const* b)
{
    //a->pnt_.mecable()->Buddy::print(std::clog);
    //b->seg_.fiber()->Buddy::print(std::clog);
    return b->seg_.fiber()->buddy() != a->pnt_.mecable()->buddy();
}


/// excluding segments that are adjacent on the same fiber
inline static bool not_adjacent(FatLocus const* a, FatLocus const* b)
{
    return (( a->seg_.fiber() != b->seg_.fiber() )
            || (( a->seg_.point() > 1 + b->seg_.point() ) | ( b->seg_.point() > 1 + a->seg_.point() )));
    // we cannot use abs() above because `pti_` is unsigned
}

//------------------------------------------------------------------------------
#pragma mark - Check all possible object pairs from two Cells

/**
 This will consider once all pairs of objects from the given lists
 */
void PointGrid::setSterics0(Meca& meca, Stiffness const& stiff,
                            FatPointList & pots, FatLocusList & locs)
{
    for ( FatPoint* ii = pots.begin(); ii < pots.end(); ++ii )
    {
        for ( FatPoint* jj = ii+1; jj < pots.end(); ++jj )
            if ( not_adjacent(ii, jj) )
                checkPP(meca, stiff, *ii, *jj);
        
        for ( FatLocus* kk = locs.begin(); kk < locs.end(); ++kk )
            if ( not_adjacent(ii, kk) )
                checkPL(meca, stiff, *ii, *kk);
    }

    for ( FatLocus* ii = locs.begin(); ii < locs.end(); ++ii )
    {
        for ( FatLocus* jj = ii+1; jj < locs.end(); ++jj )
            if ( not_adjacent(ii, jj) )
                checkLL(meca, stiff, *ii, *jj);
    }
}


/**
 This will consider once all pairs of objects from the given lists,
 assuming that the list are different and no object is repeated
 */
void PointGrid::setSterics0(Meca& meca, Stiffness const& stiff,
                            FatPointList & pots1, FatLocusList & locs1,
                            FatPointList & pots2, FatLocusList & locs2)
{
    assert_true( &pots1 != &pots2 );
    assert_true( &locs1 != &locs2 );
    
    for ( FatPoint* ii = pots1.begin(); ii < pots1.end(); ++ii )
    {
        for ( FatPoint* jj = pots2.begin(); jj < pots2.end(); ++jj )
            if ( not_adjacent(ii, jj) )
                checkPP(meca, stiff, *ii, *jj);
        
        for ( FatLocus* kk = locs2.begin(); kk < locs2.end(); ++kk )
            if ( not_adjacent(ii, kk) )
                checkPL(meca, stiff, *ii, *kk);
    }
    
    for ( FatLocus* ii = locs1.begin(); ii < locs1.end(); ++ii )
    {
        for ( FatPoint* jj = pots2.begin(); jj < pots2.end(); ++jj )
            if ( not_adjacent(jj, ii) )
                checkPL(meca, stiff, *jj, *ii);
        
        for ( FatLocus* kk = locs2.begin(); kk < locs2.end(); ++kk )
        {
            if ( not_adjacent(ii, kk)  )
                checkLL(meca, stiff, *ii, *kk);
        }
    }
}


/**
 This will consider once all pairs of objects from the given lists.
 Compared to `setSterics0()`, this performs additional tests to exclude
 objects that are too far appart to interact, based on BigVector::near()
 */
void PointGrid::setStericsT(Meca& meca, Stiffness const& stiff,
                            FatPointList & pots, FatLocusList & locs)
{
    for ( FatPoint* ii = pots.begin(); ii < pots.end(); ++ii )
    {
        const FatVector pos = ii->pos_;
        
        for ( FatPoint* jj = ii+1; jj < pots.end(); ++jj )
            if ( pos.near(jj->pos_) && not_adjacent(ii, jj) )
                checkPP(meca, stiff, *ii, *jj);
        
        for ( FatLocus* kk = locs.begin(); kk < locs.end(); ++kk )
            if ( pos.near(kk->pos_) && not_adjacent(ii, kk) )
                checkPL(meca, stiff, *ii, *kk);
    }

    for ( FatLocus* ii = locs.begin(); ii < locs.end(); ++ii )
    {
        const FatVector pos = ii->pos_;
        
        for ( FatLocus* jj = ii+1; jj < locs.end(); ++jj )
            if ( pos.near(jj->pos_) && not_adjacent(ii, jj) )
                checkLL(meca, stiff, *ii, *jj);
    }
}


/**
 This will consider once all pairs of objects from the given lists,
 assuming that the list are different and no object is repeated.

 Compared to `setSterics0()`, this performs additional tests to exclude
 objects that are too far appart to interact, based on BigVector::near()
 */
void PointGrid::setStericsT(Meca& meca, Stiffness const& stiff,
                            FatPointList & pots1, FatLocusList & locs1,
                            FatPointList & pots2, FatLocusList & locs2)
{
    assert_true( &pots1 != &pots2 );
    assert_true( &locs1 != &locs2 );

    for ( FatPoint* ii = pots1.begin(); ii < pots1.end(); ++ii )
    {
        const FatVector pos = ii->pos_;

        for ( FatPoint* jj = pots2.begin(); jj < pots2.end(); ++jj )
            if ( pos.near(jj->pos_) && not_adjacent(ii, jj) )
                checkPP(meca, stiff, *ii, *jj);
        
        for ( FatLocus* kk = locs2.begin(); kk < locs2.end(); ++kk )
            if ( pos.near(kk->pos_) && not_adjacent(ii, kk) )
                checkPL(meca, stiff, *ii, *kk);
    }
    
    for ( FatLocus* ii = locs1.begin(); ii < locs1.end(); ++ii )
    {
        const FatVector pos = ii->pos_;

        for ( FatPoint* jj = pots2.begin(); jj < pots2.end(); ++jj )
            if ( pos.near(jj->pos_) && not_adjacent(jj, ii) )
                checkPL(meca, stiff, *jj, *ii);
        
        for ( FatLocus* kk = locs2.begin(); kk < locs2.end(); ++kk )
            if ( pos.near(kk->pos_) && not_adjacent(ii, kk) )
                checkLL(meca, stiff, *ii, *kk);
    }
}

//------------------------------------------------------------------------------
#pragma mark - Check all pairs of Cells


#if ( NUM_STERIC_PANES == 1 )

/**
 Check interactions between objects contained in the grid.
 */
void PointGrid::setSterics0(Meca& meca, Stiffness const& stiff) const
{
    //std::clog << "----" << '\n';
    
    // scan all cells to examine each pair of particles:
    for ( size_t inx = 0; inx < pGrid.nbCells(); ++inx )
    {
        int * region;
        int nr = pGrid.getRegion(region, inx);
        assert_true(region[0] == 0);
        
        FatPointList & baseP = point_list(inx);
        FatLocusList & baseL = locus_list(inx);
        setSterics0(meca, stiff, baseP, baseL);
        
        for ( int reg = 1; reg < nr; ++reg )
        {
            FatPointList & sideP = point_list(inx+region[reg]);
            FatLocusList & sideL = locus_list(inx+region[reg]);
            setSterics0(meca, stiff, baseP, baseL, sideP, sideL);
        }
    }
}


void PointGrid::setStericsT(Meca& meca, Stiffness const& stiff) const
{
    //std::clog << "----" << '\n';
    
    // scan all cells to examine each pair of particles:
    for ( size_t inx = 0; inx < pGrid.nbCells(); ++inx )
    {
        int * region;
        int nr = pGrid.getRegion(region, inx);
        assert_true(region[0] == 0);
        
        FatPointList & baseP = point_list(inx);
        FatLocusList & baseL = locus_list(inx);
        
        setStericsT(meca, stiff, baseP, baseL);
        
        for ( int reg = 1; reg < nr; ++reg )
        {
            FatPointList & sideP = point_list(inx+region[reg]);
            FatLocusList & sideL = locus_list(inx+region[reg]);
            setStericsT(meca, stiff, baseP, baseL, sideP, sideL);
        }
    }
}


void PointGrid::setSterics(Meca& meca, Stiffness const& stiff) const
{
    //std::clog << "----" << '\n';
    if ( pGrid.isPeriodic() )
        setSterics0(meca, stiff);
    else
        setStericsT(meca, stiff);
}

#else

/**
 Check interactions between the FatPoints contained in Pane `pan`.
 */
void PointGrid::setSterics0(Meca& meca, Stiffness const& stiff,
                            const size_t pan) const
{
    // scan all cells to examine each pair of particles:
    for ( size_t inx = 0; inx < pGrid.nbCells(); ++inx )
    {
        int * region;
        int nr = pGrid.getRegion(region, inx);
        assert_true(region[0] == 0);
        
        FatPointList & baseP = point_list(inx, pan);
        FatLocusList & baseL = locus_list(inx, pan);
        
        setSterics0(meca, stiff, baseP, baseL);
        
        for ( int reg = 1; reg < nr; ++reg )
        {
            FatPointList & sideP = point_list(inx+region[reg], pan);
            FatLocusList & sideL = locus_list(inx+region[reg], pan);
            setSterics0(meca, stiff, baseP, baseL, sideP, sideL);
        }
    }
}


void PointGrid::setStericsT(Meca& meca, Stiffness const& stiff,
                            const size_t pan) const
{
    // scan all cells to examine each pair of particles:
    for ( size_t inx = 0; inx < pGrid.nbCells(); ++inx )
    {
        int * region;
        int nr = pGrid.getRegion(region, inx);
        assert_true(region[0] == 0);
        
        FatPointList & baseP = point_list(inx, pan);
        FatLocusList & baseL = locus_list(inx, pan);
        
        setStericsT(meca, stiff, baseP, baseL);
        
        for ( int reg = 1; reg < nr; ++reg )
        {
            FatPointList & sideP = point_list(inx+region[reg], pan);
            FatLocusList & sideL = locus_list(inx+region[reg], pan);
            setStericsT(meca, stiff, baseP, baseL, sideP, sideL);
        }
    }
}


/**
 Check interactions between the FatPoints contained in Panes `pan1` and `pan2`,
 where ( pan1 != pan2 )
 */
void PointGrid::setSterics0(Meca& meca, Stiffness const& stiff,
                            const size_t pan1, const size_t pan2) const
{
    assert_true(pan1 != pan2);
    
    // scan all cells to examine each pair of particles:
    for ( size_t inx = 0; inx < pGrid.nbCells(); ++inx )
    {
        int * region;
        int nr = pGrid.getRegion(region, inx);
        assert_true(region[0] == 0);
        
        FatPointList & baseP = point_list(inx, pan1);
        FatLocusList & baseL = locus_list(inx, pan1);
        
        for ( int reg = 0; reg < nr; ++reg )
        {
            FatPointList & sideP = point_list(inx+region[reg], pan2);
            FatLocusList & sideL = locus_list(inx+region[reg], pan2);
            setSterics0(meca, stiff, baseP, baseL, sideP, sideL);
        }
        
        FatPointList & baseP2 = point_list(inx, pan2);
        FatLocusList & baseL2 = locus_list(inx, pan2);
        
        for ( int reg = 1; reg < nr; ++reg )
        {
            FatPointList & sideP = point_list(inx+region[reg], pan1);
            FatLocusList & sideL = locus_list(inx+region[reg], pan1);
            setSterics0(meca, stiff, baseP2, baseL2, sideP, sideL);
        }
    }
}

void PointGrid::setStericsT(Meca& meca, Stiffness const& stiff,
                            const size_t pan1, const size_t pan2) const
{
    assert_true(pan1 != pan2);
    
    // scan all cells to examine each pair of particles:
    for ( size_t inx = 0; inx < pGrid.nbCells(); ++inx )
    {
        int * region;
        int nr = pGrid.getRegion(region, inx);
        assert_true(region[0] == 0);
        
        FatPointList & baseP = point_list(inx, pan1);
        FatLocusList & baseL = locus_list(inx, pan1);
        
        for ( int reg = 0; reg < nr; ++reg )
        {
            FatPointList & sideP = point_list(inx+region[reg], pan2);
            FatLocusList & sideL = locus_list(inx+region[reg], pan2);
            setStericsT(meca, stiff, baseP, baseL, sideP, sideL);
        }
        
        FatPointList & baseP2 = point_list(inx, pan2);
        FatLocusList & baseL2 = locus_list(inx, pan2);
        
        for ( int reg = 1; reg < nr; ++reg )
        {
            FatPointList & sideP = point_list(inx+region[reg], pan1);
            FatLocusList & sideL = locus_list(inx+region[reg], pan1);
            setStericsT(meca, stiff, baseP2, baseL2, sideP, sideL);
        }
    }
}



void PointGrid::setSterics(Meca& meca, Stiffness const& stiff, size_t pan) const
{
    if ( pGrid.isPeriodic() )
        setSterics0(meca, stiff, pan);
    else
        setStericsT(meca, stiff, pan);
}


void PointGrid::setSterics(Meca& meca, Stiffness const& stiff, size_t pan1, size_t pan2) const
{
    if ( pGrid.isPeriodic() )
        setSterics0(meca, stiff, pan1, pan2);
    else
        setStericsT(meca, stiff, pan1, pan2);
}

#endif


//------------------------------------------------------------------------------
#pragma mark - Display

#ifdef DISPLAY

#include "opengl.h"

void drawBoundaries(Map<DIM> const&);

void PointGrid::drawGrid() const
{
#if ( DIM <= 3 )
    glDisable(GL_LIGHTING);
    glColor3f(0, 1, 0);
    glLineWidth(0.25);
    drawBoundaries(pGrid);
#endif
}
#endif

