// Cytosim was created by Francois Nedelec. Copyright 2007-2017 EMBL.

#include "organizer_set.h"
#include "organizer.h"
#include "mecapoint.h"
#include "glossary.h"
#include "nucleus.h"
#include "bundle.h"
#include "aster.h"
#include "solid.h"
#include "simul.h"


// first Organizer
Organizer * OrganizerSet::first() const
{
    return static_cast<Organizer*>(pool_.front());
}

// find object with given ID
Organizer * OrganizerSet::findID(ObjectID n) const
{
    return static_cast<Organizer*>(inventory_.get(n));
}

//------------------------------------------------------------------------------

void OrganizerSet::step()
{
    Organizer * obj = first(), * nxt;
    while ( obj )
    {
        nxt = obj->next();
        obj->step();
        obj = nxt;
    }
}

//------------------------------------------------------------------------------

Property* OrganizerSet::newProperty(const std::string& cat, const std::string& nom, Glossary&) const
{
    if ( cat == "aster" )   return new AsterProp(nom);
    if ( cat == "bundle" )  return new BundleProp(nom);
    if ( cat == "nucleus" ) return new NucleusProp(nom);
    return nullptr;
}


Object * OrganizerSet::newObject(const ObjectTag tag, PropertyID pid)
{
    if ( tag == Aster::TAG )
    {
        AsterProp * p = simul_.findProperty<AsterProp>("aster", pid);
        return new Aster(p);
    }
    
    if ( tag == Bundle::TAG )
    {
        BundleProp * p = simul_.findProperty<BundleProp>("bundle", pid);
        return new Bundle(p);
    }
    
    if ( tag == Nucleus::TAG )
    {
        NucleusProp * p = simul_.findProperty<NucleusProp>("nucleus", pid);
        return new Nucleus(p);
    }
    
    std::cerr << "Warning: unknown Organizer tag `"+std::string(1,tag)+"' requested\n";
    return nullptr;
}


ObjectList OrganizerSet::newObjects(const std::string& name, Glossary& opt)
{
    Organizer * obj = nullptr;
    Property * p = simul_.properties.find_or_die(name);
    
    if ( p->category() == "aster" )
        obj = new Aster(static_cast<AsterProp*>(p));
    else if ( p->category() == "bundle" )
        obj = new Bundle(static_cast<BundleProp*>(p));
    else if ( p->category() == "nucleus" )
        obj = new Nucleus(static_cast<NucleusProp*>(p));

    ObjectList res;
    if ( obj )
    {
        res = obj->build(opt, simul_);
        res.push_back(obj);
    }
    
    return res;
}


void OrganizerSet::write(Outputter& out) const
{
    if ( size() > 0 )
    {
        out.writeLine("\n#section "+title());
        writeObjects(out, pool_);
    }
}

//------------------------------------------------------------------------------

ObjectID OrganizerSet::findOrganizerID(const Mecable * m) const
{
    ObjectID res = 0;
    for ( Organizer const* o=first(); o; o=o->next() )
        if ( o->check(m) )
            res = std::max(res, o->identity());

    return res;
}


//------------------------------------------------------------------------------

void OrganizerSet::report(std::ostream& os) const
{
    if ( size() > 0 )
    {
        unsigned total = 0;
        os << '\n' << title();
        for ( Property const* i : simul_.properties.find_all("aster", "bundle") )
        {
            size_t cnt = count(match_property, i);
            os << '\n' << std::setw(10) << cnt << " " << i->name();
            ++total;
        }
        for ( Property const* i : simul_.properties.find_all("nucleus", "fake") )
        {
            size_t cnt = count(match_property, i);
            os << '\n' << std::setw(10) << cnt << " " << i->name();
            ++total;
        }
        if ( total > 1 )
            os << '\n' << std::setw(10) << size() << " total";
    }
}


