// Cytosim was created by Francois Nedelec. Copyright 2007-2017 EMBL.

#ifndef INTERFACE_H
#define INTERFACE_H

#include <iostream>
#include "isometry.h"
#include "object.h"

class Glossary;
class Property;
class Simul;

/// Cytosim Application Programming Interface
/*
 A reduced set of commands to control and simulate
 a system of objects within cytosim.
 */
class Interface
{
private:
    
    /// disabled default constructor
    Interface();

    /// Simul member function pointer
    using SimulFuncPtr = void (Simul::*)();
    
    /// perform `cnt` simulation steps also calling Simul::FUNC at each step
    template < SimulFuncPtr FUNC >
    void do_steps(size_t& sss, size_t cnt);

public:
    
    /// associated Simul
    Simul& simul_;
    
    /// construct and associates with given Simul
    Interface(Simul&);
    
    //-------------------------------------------------------------------------------
    
    /// `hold()` is called between commands during the execution process
    /**
     This callback provides an opportunity to stop/examine the simulation at regular
     intervals. It does nothing for `sim` and it displays the system in `play`.
     */
    virtual void hold() {}
    
    //-------------------------------------------------------------------------------
    
    /// create a new Property of category `cat` from values specified in Glossary
    Property*  execute_set(std::string const& cat, std::string const& name, Glossary&);

    /// change values in given Property as specified in Glossary
    void       execute_change(Property*, Glossary&);

    /// change values in Property called `name` as specified in Glossary
    Property*  execute_change(std::string const& name, Glossary&, bool strict);
    
    /// change values of all Property of category `cat`
    void       execute_change_all(std::string const& cat, Glossary&);

    /// read the specification of position and orientation of an object
    Isometry   read_placement(Glossary&);
    
    /// return position and orientation of an object, with verification of 'placement'
    Isometry   find_placement(Glossary&, int placement, size_t num_trials);
    
    /// create 1 object of type `name`, following options in Glossary
    void       execute_new(std::string const& name, ObjectSet*, Glossary&);
    
    /// create `cnt` objects of type `name`, following options in Glossary
    void       execute_new(std::string const& name, Glossary&, size_t cnt);

    /// create `cnt` objects of type `name`, randomly placed in space (no option)
    void       execute_new(std::string const& name, size_t cnt);
    
    /// delete `cnt` objects of type `name`, following options in Glossary
    void       execute_delete(std::string const& name, Glossary&, size_t cnt);
    
    /// mark `cnt`  objects of type `name`, following options in Glossary
    void       execute_mark(std::string const& name, Glossary&, size_t cnt);

    /// cut fibers of type `name`, following different options in Glossary
    void       execute_cut(std::string const& name, Glossary&);
    
    /// cut fibers of type `name`, following different options in Glossary
    void       execute_connect(std::string const& name, Glossary&);

    /// import objects (or `what`) from a file
    void       execute_import(std::string const& filename, std::string const& what, Glossary&);

    /// export objects (or `what`) to a file
    void       execute_export(std::string const& filename, std::string const& what, Glossary&);

    /// write information (specified in `what`) to a file
    void       execute_report(std::string const& filename, std::string const& what, Glossary&);
    
    /// perform `cnt` simulation steps, following options specified in Glossary
    void       execute_run(size_t cnt, Glossary&, bool write_permission);
    
    /// perform `cnt` simulation steps
    void       execute_run(size_t cnt);

    /// execute miscellaneous functions
    void       execute_call(std::string& func, Glossary&);

};

#endif

