// Cytosim was created by Francois Nedelec. Copyright 2020 Cambridge University.
#include "dim.h"
#include "cymdef.h"
#include "simul.h"
#include "messages.h"
#include "exceptions.h"
#include "glossary.h"
#include "simul_prop.h"
#include "wanderer_prop.h"
#include "wanderer.h"


Hand * WandererProp::newHand(HandMonitor* h) const
{
    return new Wanderer(this, h);
}


void WandererProp::clear()
{
    DigitProp::clear();
    diffusion = 0;

}


void WandererProp::read(Glossary& glos)
{
    DigitProp::read(glos);
    glos.set(diffusion, "diffusion");

}


void WandererProp::complete(Simul const& sim)
{
    DigitProp::complete(sim);
    diff_rate    = diffusion / ( step_size * step_size );
    diff_rate_dt = diff_rate * sim.time_step();
    step_size_kT = step_size / sim.prop->kT;

    if ( sim.primed() )
    {
        if ( diff_rate_dt > sim.prop->acceptable_prob )
            Cytosim::warn << name() << ":diffusion is too high ("<<diff_rate_dt<<"): decrease time_step\n";
    }
}


void WandererProp::completeStiffness(Simul const& sim, real stiffness)
{
    U_step_kT_2 = (stiffness*step_size*step_size) / (2*sim.prop->kT);
}


void WandererProp::write_values(std::ostream & os) const
{
    DigitProp::write_values(os);
    write_value(os, "diffusion",  diffusion);
}

