// Cytosim was created by Francois Nedelec. Copyright 2020 Cambridge University.
#include "digit.h"
#include "wanderer.h"
#include "wanderer_prop.h"
#include "glossary.h"
#include "lattice.h"
#include "simul.h"
extern Random RNG;


Wanderer::Wanderer(WandererProp const* p, HandMonitor* h)
: Digit(p,h), nextStepM(0), nextStepP(0), prop(p)
{
    
}


void Wanderer::attach(FiberSite const& site)
{
    Digit::attach(site);
    nextStepM = RNG.exponential();
    nextStepP = RNG.exponential();
}


void Wanderer::stepUnloaded()
{
    assert_true( attached() );
    
    /* Find possible moves from current position, calling outsideMP() first,
     since vacant() should not be called with an invalid position. */
    bool outM = outsideMP(site()-1);
    bool outP = outsideMP(site()+1);
    bool checkM = outM || vacant(site()-1);
    bool checkP = outP || vacant(site()+1);

    // if both positions are occupied, skip
    if ( checkM | checkP )
    {
        nextStepM -= prop->diff_rate_dt * checkM;
        nextStepP -= prop->diff_rate_dt * checkP;

        lati_t step = 0;
        // count number of steps in each direction:
        while ( nextStepM <= 0 )
        {
            --step;
            nextStepM += RNG.exponential();
        }
        while ( nextStepP <= 0 )
        {
            ++step;
            nextStepP += RNG.exponential();
        }

        // perform first step:
        if ( step < 0 )
        {
            if ( outM )
            {
                if ( RNG.test_not(prop->hold_growing_end) )
                    detach();
                return;
            }
            // Here the vacancy of the site has been checked already
            hop(site()-1);
            ++step;
        }
        else if ( step > 0 )
        {
            if ( outP )
            {
                if ( RNG.test_not(prop->hold_growing_end) )
                    detach();
                return;
            }
            // Here the vacancy of the site has been checked already
            hop(site()+1);
            --step;
        }

        //if ( std::abs(step) > 1 )
          //  std::clog << "made multiple wanderer::steps ("<<step<<"): decrease time_step\n";
#if 1
        /* It is not correct to allow multiple steps while other Wandered do not move,
         but in most practical cases this will work fine */
        while ( step )
        {
            lati_t s = std::copysign((lati_t)1, step);
            lati_t t = site() + s;
            step = step - s;
            
            if ( outsideMP(t) )
            {
                if ( RNG.test_not(prop->hold_growing_end) )
                    detach();
                return;
            }
            if ( vacant(t) )
                hop(t);
        }
#endif
    }
}

/**
 f(x) = x / ( exp(x) - 1 ) is a smooth function everywhere:
 f(x) = 1 / ( 1 + x/2 + x^2/6 + ... x^(n-1)/n! ... )
 in particular:
 f(x) = 1 - x/2 for |x| small
 
 argument `cfor` should be the colinear force: dot(force, dirFiber())
 */
void Wanderer::calcPropensities(real const& cfor, real& P, real& M)
{
    constexpr real small = 0.0156250;
    // From Hannabuss et al. 2019:
    // step_size_kT = a/kT
    // prop->U_step_kT_2 = (stiffness*a^2)/(2kT)
    // dG0 = f*a/kT
    
    real dG0 = cfor * prop->step_size_kT;
    
    if ( P > 0 )
    {
        real dG = dG0 - prop->U_step_kT_2;
        // To avoid zero / zero
        if ( abs_real(dG) < small )
            P = 1 - dG * 0.5;
        else
            P = -dG / std::expm1(-dG);
    }
    
    if ( M > 0 )
    {
        real dG = -dG0 - prop->U_step_kT_2;
        // To avoid zero / zero
        if ( abs_real(dG) < small )
            M = 1 - dG * 0.5;
        else
            M = -dG / std::expm1(-dG);
    }
}


void Wanderer::stepLoaded(Vector const& force)
{
    assert_true( attached() );

    // Check that site is outside first, to avoid calling vacant() with an invalid site.
    real propM = outsideMP(site()-1) || vacant(site()-1);
    real propP = outsideMP(site()+1) || vacant(site()+1);
    
    // If both positions are occupied, skip
    if ( propM || propP )
    {
        /* calculate probabilities of displacement,
         dependent on the load along the desired direction of displacement */
        calcPropensities(dot(force, dirFiber()), propP, propM);
        
        nextStepM -= prop->diff_rate_dt * propM;
        nextStepP -= prop->diff_rate_dt * propP;

        // I changed this from while to if because otherwise it would move several times when the force is high,
        // otherwise it should update the force every time?
        // This was leading to a lot of steps per dt I guess.
        
        int step = 0;
        if ( nextStepM <= 0 )
        {
            if ( nextStepP <= 0 )
            {
                // -1 or +1, following the earliest event:
                step = ( nextStepP < nextStepM ) - ( nextStepM < nextStepP );
                nextStepM = RNG.exponential();
                nextStepP = RNG.exponential();
            }
            else
            {
                step = -1;
                nextStepM = RNG.exponential();
            }
        }
        else if ( nextStepP <= 0 )
        {
            step = 1;
            nextStepP = RNG.exponential();
        }
        else
            return;

        lati_t s = site() + step;
        
        if ( outsideMP(s) )
        {
            if ( RNG.test_not(prop->hold_growing_end) )
            {
                detach();
                return;
            }
        }
        else
            hop(s);
    }
}

