// Cytosim was created by Francois Nedelec. Copyright 2021 Cambridge University.

#include "nucleator.h"
#include "nucleator_prop.h"
#include "glossary.h"
#include "exceptions.h"
#include "iowrapper.h"
#include "fiber_prop.h"
#include "fiber_set.h"
#include "simul.h"


//------------------------------------------------------------------------------

Nucleator::Nucleator(NucleatorProp const* p, HandMonitor* h)
: Hand(p,h), prop(p)
{
    nextNuc = RNG.exponential();
}

//------------------------------------------------------------------------------

void Nucleator::makeFiber(Simul& sim, Vector pos, std::string const& fiber_type, Glossary& opt)
{    
    ObjectList objs = sim.fibers.newObjects(fiber_type, opt);
    if ( objs.empty() )
        return;

    Fiber * fib = Fiber::toFiber(objs[0]);
    
    // register the new objects:
    sim.add(objs);
    
    // indicate the origin of nucleation:
    ObjectMark mk = 0;
    if ( opt.set(mk, "mark") )
        Simul::mark(objs, mk);
    else
        Simul::mark(objs, hMonitor->nucleatorID());

    // the Fiber will be oriented depending on specificity:
    Rotation rot;
    
    real ang = 0;
    if ( opt.set(ang, "nucleation_angle") )
    {
        Vector dir = hMonitor->otherDirection(this);
        rot = Rotation::rotationToVector(dir);
#if ( DIM == 2 )
        rot = rot * Rotation::randomRotation(ang);
#elif ( DIM == 3 )
        rot = rot * Rotation::rotationAroundX(RNG.sreal()*M_PI) * Rotation::rotationAroundZ(ang);
#endif
    }
    else switch( prop->specificity )
    {            
        case NucleatorProp::NUCLEATE_PARALLEL:
        {
            Vector dir = hMonitor->otherDirection(this);
            rot = Rotation::randomRotationToVector(dir);
        }
        break;
        
        case NucleatorProp::NUCLEATE_ANTIPARALLEL:
        {
            Vector dir = -hMonitor->otherDirection(this);
            rot = Rotation::randomRotationToVector(dir);
        }
        break;
        
        case NucleatorProp::NUCLEATE_PARALLEL_IF:
        {
            Hand * ha = hMonitor->otherHand(this);
            if ( ha && ha->attached() )
            {
                rot = Rotation::randomRotationToVector(ha->dirFiber());
                // remove key to avoid unused warning:
                opt.clear("direction");
                break;
            }
            fib->mark(0);
        }
        // here is an intentional fallback on the next case:
        
        case NucleatorProp::NUCLEATE_DIRECTED:
        {
            std::string str;
            if ( opt.set(str, "direction") )
            {
                std::istringstream iss(str);
                Vector vec = Movable::readDirection(iss, pos, fib->prop->confine_space_ptr);
                rot = Rotation::randomRotationToVector(vec);
            }
            else {
                rot = Rotation::randomRotation();
            }
        }
        break;

        default:
            throw InvalidParameter("unknown nucleator:specificity");
    }
    
    ObjectSet::rotateObjects(objs, rot);
    
    // shift position by the length of the interaction:
    if ( hMonitor->linkRestingLength() > 0 )
    {
        Vector dir = hMonitor->otherDirection(this);
        pos += dir.randOrthoU(hMonitor->linkRestingLength());
    }

    /*
     We translate Fiber to match the Nucleator's position,
     and if prop->hold_end, the Hand is attached to the new fiber
     */
    if ( prop->hold_end == MINUS_END )
    {
        attachEnd(fib, MINUS_END);
        ObjectSet::translateObjects(objs, pos-fib->posEndM());
    }
    else if ( prop->hold_end == PLUS_END )
    {
        attachEnd(fib, PLUS_END);
        ObjectSet::translateObjects(objs, pos-fib->posEndP());
    }
    else
        ObjectSet::translateObjects(objs, pos-fib->position());

    //std::clog << "nucleated fiber in direction " << fib->dirEndM() << "\n";

    opt.print_warnings(std::cerr, 1, "nucleator:spec\n");
    assert_true(fib->valid());
}


//------------------------------------------------------------------------------
/**
 Does not attach nearby Fiber, but can nucleate
 */
void Nucleator::stepUnattached(Simul& sim, Vector const& pos)
{
    assert_false( attached() );
    
    nextNuc -= prop->rate_dt;
    
    if ( nextNuc < 0 )
    {
        nextNuc = RNG.exponential();
        try {
            Glossary opt(prop->fiber_spec);
            makeFiber(sim, pos, prop->fiber_type, opt);
        }
        catch( Exception & e )
        {
            e << "\nException occurred while executing nucleator:code";
            throw;
        }
    }
}


void Nucleator::stepUnloaded()
{
    assert_true( attached() );
    
    /// OPTION 1: delete entire fiber
    if ( prop->addictive == 2 )
    {
        delete(fiber());
        return;
    }
    
    // may track the end of the Fiber:
    if ( prop->track_end == MINUS_END )
        relocateM();
    else if ( prop->track_end == PLUS_END )
        relocateP();
}


void Nucleator::stepLoaded(Vector const& force)
{
    assert_true( attached() );
    
    // may track the end of the Fiber:
    if ( prop->track_end == MINUS_END )
        relocateM();
    else if ( prop->track_end == PLUS_END )
        relocateP();
}


void Nucleator::detach()
{
    // if `addictive`, give a poisonous goodbye-kiss to the fiber
    if ( prop->addictive )
        fiber()->setEndState(nearestEnd(), STATE_RED);
    
    Hand::detach();
}

