// Cytosim was created by Francois Nedelec. Copyright 2021 Cambridge University

#include "cymdef.h"
#include "meca.h"
#include "modulo.h"
#include "glossary.h"
#include "simul_prop.h"
#include "print_color.h"
#include "display1.h"
#include "display2.h"
#include "display3.h"
#include "saveimage.h"
#include "tictoc.h"
#include <unistd.h>
#include <cstdlib>
#include "filepath.h"
#include "glut.h"

extern void helpKeys(std::ostream&);

extern Modulo const* modulo;

//------------------------------------------------------------------------------
#pragma mark -

void Player::setStyle(const unsigned style)
{
    if ( mDisplay )
    {
        //restore the previous OpenGL state
        glPopAttrib();
        delete(mDisplay);
        mDisplay = nullptr;
    }
    
    //save the current OpenGL state
    glPushAttrib(GL_ALL_ATTRIB_BITS);
    
    switch ( style )
    {
        default:
        case 1: mDisplay = new Display1(&disp);  break;
        case 2: mDisplay = new Display2(&disp);  break;
        case 3: mDisplay = new Display3(&disp);  break;
    }
    disp.style = style;

    //initialize Views associated with opened GLUT windows:
    for ( size_t n = 1; n < glApp::views.size(); ++n )
    {
        View & view = glApp::views[n];
        if ( view.window() > 0 )
        {
            //std::clog << "initializing GLUT window " << n << '\n';
            //glutSetWindow(view.window());
            view.initGL();
            glViewport(0, 0, view.width(), view.height());
        }
    }
}


/**
 Build a message containing the label and the time.
 In live mode, it also adds 'Live' or it indicates the frame index,
 and the force generated by the mouse-controlled Single.
 */
std::string Player::buildLabel() const
{
    std::ostringstream oss;
    oss.precision(3);
    oss << std::setw(8) << std::fixed << simul.time() << "s";
    
    //display the force exerted by the mouse-controled Single:
    Single const* sh = thread.handle();
    if ( sh && sh->attached() )
        oss << "\nHandle: " << sh->force().norm() << "pN";

    if ( thread.alive() && prop.goLive )
    {
        oss << "\nLive";
        //display ratio number-of-time-step / frame
        if ( prop.period > 1 )
            oss << " " << prop.period;
#if ( 0 )
        // display speedup compared to clock time
        static double sec = TicToc::seconds_today();
        static double sim = 0, spd = 0;
        double SEC = TicToc::seconds_today();
        if ( SEC > sec + 1.0 )
        {
            double SIM = simul.time();
            spd = ( SIM - sim ) / ( SEC - sec );
            sim = SIM;
            sec = SEC;
        }
        oss << std::setw(8) << std::fixed << spd << "x ";
#endif
    }
    else if ( thread.currentFrame() > 0 )
    {
        oss << "\nFrame " << thread.currentFrame();
    }

    return oss.str();
}


/**
 This information is displayed in the top corner of the window.
 Calling simul.report() ensures that the message is identical to what
 would be printed by the command 'report'.
 */
std::string Player::buildReport(std::string arg) const
{
    if ( ! arg.empty() )
    {
        Glossary glos;
        std::stringstream ss;
        simul.report(ss, arg, glos);
        std::string res = ss.str();
        // remove extra new-line
        if ( res.size() > 1  &&  res.at(0) == '\n' )
            return res.substr(1);
        return res;
    }
    return "";
}

/**
 This text is normally displayed in the center of the window
 */
std::string Player::buildMemo(int type) const
{
    std::ostringstream oss;
    switch ( type )
    {
        case 0: return "";
        case 1: return "Please, visit www.cytosim.org";
        case 2: helpKeys(oss); return oss.str();
        case 3: glApp::help(oss);  return oss.str();
        case 4: writePlayParameters(oss, true); return oss.str();
        case 5: writeDisplayParameters(oss, true); return oss.str();
    }
    return "";
}

//------------------------------------------------------------------------------
#pragma mark - Display

void Player::autoTrack(FiberSet const& fibers, View& view)
{
    real vec[9] = { 1, 0, 0, 0, 1, 0, 0, 0, 1 };
    
    if ( view.track_fibers & 1 )
    {
        Vector M, G, P;
        FiberSet::infoPosition(fibers.collect(), M, G, P);
        view.move_shift(Vector3(G));
        //std::clog << "auto center: " << G << '\n';
    }
    
    if ( view.track_fibers & 2 )
    {
        // align with mean nematic direction
        real S = FiberSet::infoNematic(fibers.collect(), vec);
        view.align_with(Vector3(vec));
        flashText("Nematic order S = %5.3f", S);
        //view.rotation.setFromMatrix3(vec);
        //view.rotation.conjugate();
        //std::clog << "auto rotate: " << Vector3(vec) << '\n';
    }

    if ( view.track_fibers & 4 )
    {
        real sum = 0;
        real avg[3] = { 0 };
        real mom[9] = { 0 };
        FiberSet::infoComponents(fibers.collect(), sum, avg, mom, vec);
        // get rotation from matrix:
        view.rotation.setFromMatrix3(vec);
        // inverse rotation:
        view.rotation.conjugate();
        //std::clog << "auto quat: " << view.rotation << '\n';
    }
}


/**
 Adjust to see the biggest Space in simul
 */
void Player::autoScale(SpaceSet const& spaces, View& view)
{
    real rad = 0;
    for ( Space const* spc = spaces.first(); spc; spc=spc->next() )
        rad = std::max(rad, spc->max_extension());
    if ( rad > 0 )
    {
        //std::clog << "auto_scale " << rad << '\n';
        view.view_size = GLfloat(2*rad);
        view.zoom_in(0.933033);
        --view.auto_scale;
    }
}


void Player::prepareDisplay(View& view, int mag)
{    
    CHECK_GL_ERROR("before prepareDisplay");
    //----------------- automatic adjustment of viewing area:

    if ( view.auto_scale > 0 )
        autoScale(simul.spaces, view);
    
    //----------------- auto-track:
    
    if ( view.track_fibers )
        autoTrack(simul.fibers, view);
    
    //----------------- texts:
    
    view.setLabel(buildLabel());
    view.setMemo(buildMemo(view.draw_memo));
    view.setMessage(buildReport(prop.report));

    //----------------- set pixel size, unit-size and direction:

    GLfloat pix = view.pixelSize();
    /*
     if `disp.point_value` is set, line width and point size are to be understood
     in 'real' units, and otherwise, they are understood as number of pixels.
     */
    GLfloat val = (disp.point_value > 0 ? disp.point_value/pix : 1 );
    //std::clog << " pixel size = " << pix << '\n';

    mDisplay->setPixelFactors(pix/mag, mag*val);
    mDisplay->setStencil(view.stencil && ( DIM >= 3 ));
    /// adjust reference color used by gle::bright_color
    gle::background_color = view.back_color;
    
    CHECK_GL_ERROR("in prepareDisplay");

    try {
        mDisplay->prepareForDisplay(simul, dispList, view.depthAxis());
        //std::clog << " dispList.size() = " << dispList.size() << '\n';
    }
    catch(Exception & e)
    {
        print_blue(std::cerr, e.brief());
        std::cerr << e.info() << '\n';
    }
}

//------------------------------------------------------------------------------
void Player::drawCytosim()
{
#if 0
    static double sec = TicToc::milliseconds();
    double now = TicToc::milliseconds();
    std::clog << " drawCytosim(" << std::setprecision(3) << simul.time() << "s) " << now-sec << "\n";
    sec = now;
#endif
    try {
        // draw:
        if ( modulo && disp.tile )
            mDisplay->drawTiled(simul, disp.tile);
        else
            mDisplay->drawSimul(simul);

#if DRAW_MECA_LINKS
        if ( disp.draw_links )
        {
            glLineWidth(6);
            glPointSize(8);
            glDisable(GL_LIGHTING);
            glEnable(GL_LINE_STIPPLE);
            glEnableClientState(GL_COLOR_ARRAY);
            simul.drawLinks();
            glDisableClientState(GL_COLOR_ARRAY);
            glDisable(GL_LINE_STIPPLE);
            CHECK_GL_ERROR("Simul::drawLinks()");
        }
#endif
    }
    catch(Exception & e)
    {
        print_blue(std::cerr, e.brief());
        std::cerr << e.info() << " (display)\n";
    }
}


void Player::readDisplayString(View& view, std::string const& str)
{
    //std::clog << "readDisplayString " << str << '\n';
    try
    {
        Glossary glos(str);
        disp.read(glos);
        prop.read(glos);
        const int W = view.width(), H = view.height();
        view.read(glos);
        // window size cannot be changed:
        view.window_size[0] = W;
        view.window_size[1] = H;
    }
    catch( Exception & e )
    {
        print_blue(std::cerr, e.brief());
        std::cerr << e.info() << " (simul:display)\n";
    }
}


/**
 Display the full Scene for export, skipping some features
 */
void Player::drawScene(View& view)
{
    view.openDisplay();
    drawCytosim();
    view.closeDisplay();
    glFinish();
    
    if ( prop.save_images > 0 )
    {
        double t = simul.time();
        if ( prop.saved_image_time != t )
        {
            prop.saved_image_time = t;
            if ( prop.goLive )
                saveView(prop.image_index++, prop.downsample);
            else
                saveView(thread.currentFrame(), prop.downsample);
            // exit if this was the last image requested:
            if ( --prop.save_images == 0 && ( prop.auto_exit & 2 ))
            {
                printf("\n");
                exit(EXIT_SUCCESS);
            }
        }
    }
}


void Player::drawScene(View& view, int mag)
{
    if ( simul.prop->display_fresh )
    {
        readDisplayString(view, simul.prop->display);
        simul.prop->display_fresh = false;
    }
    //thread.debug("drawScene");
    prepareDisplay(view, mag);
    drawScene(view);
}

//------------------------------------------------------------------------------
#pragma mark - Export Image

/**
 Export image from the current OpenGL back buffer,
 in the format specified by 'PlayerProp::image_format',
 in the current working directory
 */
int Player::saveView(const char* filename, const char* format, int downsample, int verbose) const
{
    GLint vp[4];
    glGetIntegerv(GL_VIEWPORT, vp);
    int err = SaveImage::saveImage(filename, format, vp, downsample);
    if ( err == 0 && verbose > 0 )
    {
        printf("\r saved %ix%i snapshot %s    ", vp[2]/downsample, vp[3]/downsample, filename);
        fflush(stdout);
    }
    return err;
}


/**
 Export image from the current OpenGL back buffer,
 in the format specified by 'PlayerProp::image_format',
 in the folder specified in `PlayerProp::image_dir`.
 The name of the file is formed by concatenating 'root' and 'indx'.
 */
int Player::saveView(size_t indx, int downsample, int verbose) const
{
    char str[1024] = { 0 };
    char const* format = prop.image_format.c_str();
    std::string& name = prop.image_name;
    std::string::size_type d = name.find('.');
    if ( d != std::string::npos )
    {
        // if there is a '.', copy name verbatim:
        strncpy(str, name.c_str(), sizeof(str));
        name = name.substr(0, d);
    }
    else
    {
        // with no '.', build name using integer:
        snprintf(str, sizeof(str), "%s%04lu.%s", name.c_str(), indx, format);
    }
    int cwd = FilePath::change_dir(prop.image_dir, true);
    int err = saveView(str, format, downsample, verbose);
    FilePath::change_dir(cwd);
    return err;
}


//------------------------------------------------------------------------------

void displayMagnified(int mag, void * arg)
{
    glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
    static_cast<Player*>(arg)->drawCytosim();
    CHECK_GL_ERROR("in displayMagnified");
}


/**
 save an image where the resolution is magnified by a factor `mag`.
 This requires access to the simulation world.
 */
int Player::saveScene(const int mag, const char* name, const char* format, const int downsample)
{
    View & view = glApp::currentView();
    const int W = view.width(), H = view.height();
    thread.lock();
    
    //std::clog << "saveMagnifiedImage " << W << "x" << H << " mag=" << mag << '\n';
    prepareDisplay(view, mag);
    view.openDisplay();
    int err = SaveImage::saveMagnifiedImage(mag, name, format, W, H, displayMagnified, this, downsample);
    if ( err )
        err = SaveImage::saveCompositeImage(mag, name, format, W, H, view.pixelSize(), displayMagnified, this, downsample);
    if ( !err )
        printf("saved %ix%i snapshot %s\n", mag*W/downsample, mag*H/downsample, name);
    view.closeDisplay();
    
    thread.unlock();
    return err;
}


/**
 save an image where the resolution is magnified by a factor `mag`.
 This requires access to the simulation world.
 */
int Player::saveScene(const int mag, const char* root, unsigned indx, const int downsample)
{
    char str[1024] = { 0 };
    char const* format = prop.image_format.c_str();
    snprintf(str, sizeof(str), "%s%04i.%s", root, indx, format);
    int cwd = FilePath::change_dir(prop.image_dir, true);
    int err = saveScene(mag, str, format, downsample);
    FilePath::change_dir(cwd);
    glApp::postRedisplay();
    return err;
}

