// Cytosim was created by Francois Nedelec.
// Copyright 2019-2020 Sainsbury Laboratory, Cambridge University

#ifndef XPTTRF_H
#define XPTTRF_H


/**
 This is a C-translation of the LAPACK reference implementation of dpttrf()
 to factorize a symmetric definite-positive tridiagonal matrix.
 The method is known as Thomas' algorithm.
*/
void lapack_xpttrf_ori(int size, real* D, real* E, int* INFO)
{
    *INFO = 0;
    for ( int i = 0; i < size-1; ++i )
    {
        if ( D[i] < 0 )
        {
            *INFO = i;
            return;
        }
        real e = E[i];
        E[i] = e / D[i];
        D[i+1] = D[i+1] - e * E[i];
    }
}

/// modified version only loads D[] at one place
void lapack_xpttrf(int size, real* D, real* E, int* INFO)
{
    *INFO = 0;
    real d = D[0];
    for ( int i = 0; i < size-1; ++i )
    {
        if ( D[i] < 0 )
        {
            *INFO = i;
            return;
        }
        real e = E[i] / d;
        d = D[i+1] - e * E[i];
        D[i+1] = d;
        E[i] = e;
    }
}

/**
 This is a C-translation of the LAPACK reference implementation of dpttrf()
 to factorize a symmetric definite-positive tridiagonal matrix.
 The method is known as Thomas' algorithm.
*/
void xpttrf(int size, real* D, real* E)
{
    real d = D[0];
    for ( int i = 0; i < size-1; ++i )
    {
        real e = E[i] / d;
        d = D[i+1] - e * E[i];
        D[i+1] = d;
        E[i] = e;
    }
}

/**
 This is a C-translation of the LAPACK reference implementation of dptts2()
 
 *     Solve A * X = B using the factorization A = L*D*L**T,
 *     overwriting each right hand side vector with its solution.
 
     DO I = 2, N
         B(I) = B(I) - B(I-1) * E(I-1)
     CONTINUE
 
     B(N) = B(N) / D(N)
 
     DO I = N - 1, 1, -1
         B(I) = B(I) / D(I) - B(I+1) * E(I)
     CONTINUE
 */
void lapack_xptts2(int size, const real* D, const real* E, real* B)
{
    assert_true(size > 0);

    for ( int i = 1; i < size; ++i )
        B[i] = B[i] - B[i-1] * E[i-1];
    
    B[size-1] = B[size-1] / D[size-1];
    
    for ( int i = size-2; i >= 0; --i )
        B[i] = B[i] / D[i] - B[i+1] * E[i];
}


inline void lapack_xptts2(int size, int nrhs, real const* D, real const* E, real* B, int LDB)
{
    assert_true( nrhs == 1 ); // in this case, the last argument (LDB) is ignored
    return lapack_xptts2(size, D, E, B);
}

#endif
