#!/usr/bin/env python
#
# A script to submit jobs to the PBS queuing system
#
# F. Nedelec, October 2007 - August 2008 - March 2010

"""
    Submit an array of jobs to the PBS queue, to be started by 'go_sim.py'

Syntax:

    submit.py [repeat] [mem=????] [script.py] file1.cym [file2.cym] [file3.cym] [...]
    
    [repeat] is an integer specifying the number of run for each file, done on the same node

    script.py should be a loadable python module providing a function parse(input, output),
    You may use for example preconfig.py

    The amount of requested memory (default=1GB) can be specified as:
       mem=1GB
       mem=512MB
       ...
       
Examples:

    1. submit.py config.cym
        Submit one job with provided config file

    2. submit.py preconfig.py config.cym.tpl
        Submit a job where the config file will be generated by preconfig.py from config.cym.tpl

    3. submit.py 10 config.cym
        Submit a job to run the same config file 10 times

    4. submit.py config1.cym config2.cym config3.cym ...
        Submit one job for each config file provided

    5. submit.py 10 config1.cym config2.cym config3.cym ...
        Submit a job for each config file provided, with 10 repeats of each


F. Nedelec, March 2010
"""


import sys, os

base  = os.getcwd()
edir  = base+'/errors'
odir  = base+'/output'
sdir  = base+'/todo'
#requested memory:
mem   = '1gb'

#location of qsub:
qsub  = '/opt/pbs/default/bin/qsub'


def makeDirectory(name):
    """create a directory with specified name, if it does not exists"""
    if not os.path.isdir(name):
        try:
            os.mkdir(name)
        except OSError:
            print "directory '%s' could not be created" % name, sys.exit()


def writeJobScript(id, conf, preconf='', cnt=1):
    """create an executable script that will run one PBS job"""
    name = sdir + ( '/R%i' % id )
    file = open(name, 'w')
    file.write("#!/bin/bash\n")
    file.write("#PBS -l select=ncpus=1:mem=%s\n" % mem);
    file.write("cd %s;\n" % base)
    file.write("./go_sim.py run%04i %i %s %s;\n" % (id, cnt, preconf, conf))
    file.write("mv %s done/.;\n" % name)
    file.close()
    os.chmod(name, 0700)
    return name


def submitJob(name):
    """submit the script 'name' to PBS, and delete it"""
    err  = os.path.join(edir, name)
    out  = os.path.join(odir, name)
    cmd  = qsub+" -q clng_new -S /bin/bash -e %s -o %s  %s" % (err, out, name)
    val = os.system(cmd)
    if not val == 0:
        print "qsub failed with value ", val
    os.remove(name)


def writeJobArrayScript(first, last):
    """create an executable file 'cytosim.pbs' that will run a PBS array-job"""
    name = sdir + '/cytosim.pbs'
    file = open(name, 'w')
    file.write("#PBS -l select=mem=%s\n" % mem);
    file.write("#PBS -N cytosim\n")
    file.write("#PBS -M %s@embl.de\n" % os.environ['USER'])
    file.write("#PBS -m abe\n")
    file.write("#PBS -J %i-%i\n" % (first,last))
    file.write("#PBS -q clng_new\n")
    file.write("#PBS -S /bin/bash\n")
    file.write("#PBS -e %s\n" % (edir+'/R^array_index^'))
    file.write("#PBS -o %s\n" % (odir+'/R^array_index^'))
    file.write(sdir+'/R$PBS_ARRAY_INDEX\n')
    file.close()
    os.chmod(name, 0700)
    return name


def submitJobArray(first, last):
    """submit the script written by 'writejobArrayScript' to the PBS queue"""
    name = writeJobArrayScript(first, last)
    val = os.system(qsub+" "+name)
    if not val == 0:
        print "qsub failed with value ", val


def main(args):
    """submit PBS jobs, depending on the arguments provided"""
    
    #check pbs command:
    if not os.path.isfile(qsub):
        sys.stderr.write("Error: qsub '%s' not found\n" % qsub)
        return

    # make directories if necessary
    makeDirectory(edir)
    makeDirectory(odir)
    makeDirectory(sdir)
    makeDirectory(base+'/done')
    makeDirectory(base+'/save')
    
    cnt  = 1
    id   = 0
    name = ''
    preconf = ''
    for arg in args:
        if os.path.isfile(arg) and arg.endswith('.py'):
            preconf = arg
        elif arg.isdigit():
            cnt = int(arg)
        elif os.path.isfile(arg) and arg.endswith('.cym'):
            name = writeJobScript(id, arg, preconf, cnt)
            id += 1
        elif arg.startswith('mem='):
            mem = arg[4:]
        elif arg == "clear":
            print("rm -f %s/R* %s/cytosim.pbs" % (sdir, sdir))
            os.system("rm -f %s/R* %s/cytosim.pbs" % (sdir, sdir))
            sys.exit()
        else:
            print("I do not understand argument [%s] : abort" % arg)
            sys.exit()
    
    if id > 1:
        print("submit.py created %i scripts in %s" % (id, sdir))
        submitJobArray(0, id-1)
    elif not name == '':
        submitJob(name);
    else:
        print("no job submitted")


#------------------------------------------------------------------------

if __name__ == "__main__":
    if len(sys.argv) < 2 or sys.argv[1].endswith("help"):
        print(__doc__)
    else:
        main(sys.argv[1:])

