#!/usr/bin/env python
#
# A script to submit jobs to the LSF queuing system
#
# November 2012

"""
submit_lsf.submit(args)

Action:

    submit an array of jobs to the LSF queue, to be started by 'go_sim.py'

Usage:

    submit_lsf.py [repeat] [mem=????] [script.py] file1.cym [file2.cym] [file3.cym] [...]
    
    [repeat] is an integer specifying the number of run for each file, done on the same node

    script.py should be a loadable python module providing a function parse(input, output),
    You may use for example preconfig.py

    The amount of requested memory (default=1GB) can be specified as:
       mem=1024 (for 1Gb)
       mem=512  (for 512Mb)
       ...
       
Examples:

    1. submit_lsf.py config.cym
        Submit one job with provided config file

    2. submit_lsf.py preconfig.py config.cym.tpl
        Submit a job where the config file will be generated by preconfig.py from config.cym.tpl

    3. submit_lsf.py 10 config.cym
        Submit a job to run the same config file 10 times

    4. submit_lsf.py config1.cym config2.cym config3.cym ...
        Submit one job for each config file provided

    5. submit_lsf.py 10 config1.cym config2.cym config3.cym ...
        Submit a job for each config file provided, with 10 repeats of each


F. Nedelec, March 2010
"""

######
# Nb1: this cript can only launch go_sim.py (where sim is called)
# Nb2: go_sim.py needs a local symbolic link to sim
######


import sys, os

base  = os.getcwd()
edir  = base+'/errors'
odir  = base+'/output'
sdir  = base+'/todo'
arrayname = 'cytosim'
#default requested memory is 1000Mb

#location of bsub:
bsub  = '/usr/share/lsf/7.0/linux2.6-glibc2.3-x86_64/bin/bsub'


def makeDirectory(name):
    """create a directory with specified name, if it does not exists"""
    if not os.path.isdir(name):
        try:
            os.mkdir(name)
        except OSError:
            print "directory '%s' could not be created" % name, sys.exit()


def writeJobScript(id, conf, preconf='', cnt=1):
    """create an executable script that will run one LSF job"""
    name = 'R%i' % id
    fname = os.path.join(sdir,name)
    print 'writeJobScript', fname
    file = open(fname, 'w')
    file.write('#!/bin/bash\n')
    file.write('cd %s;\n' % base)
    # set job ID number, repeats, preconf script, base config
    file.write('%s/go_sim.py run%04i %i %s %s;\n' % (base, id, cnt, preconf, conf))
    print '%s/go_sim.py run%04i %i %s %s;\n' % (base, id, cnt, preconf, conf)
    file.write('mv %s %s/done/.;\n' % (fname, base) )
    file.close()
    os.chmod(fname, 0700)
    return name


def submitJob(name, mem=1024):
    """submit the script 'name' to LSF, and delete it if correctly ran"""
    exe  = os.path.join(sdir, name)
    err  = os.path.join(edir, name)
    out  = os.path.join(odir, name)
    # set memory , queue , CPU number and request logon on host
    cmd  = bsub+' -M %i -R "rusage[mem=%i]" -q medium_priority -n 1 -L /bin/bash' % (mem, mem)
    # set redirection for stderr and sdtout, executable
    cmd += ' -e %s -o %s %s' % (err, out, exe)
    print 'submitJob ', cmd
    val = os.system(cmd)
    if not val == 0:
        print "bsub failed with value ", val
    else:
        os.remove(exe)


def writeJobArrayScript(first, last, mem=1024):
    """create an executable file 'cytosim.pbs' that will run a LSF array-job"""
    name = sdir + '/cytosim.lsf'
    file = open(name, 'w')
    # defines the array name and scope
    cmd  = bsub+' -J "%s[%i-%i]"' % (arrayname, first, last)
    # set memory , queue , CPU number and request logon on host
    cmd += ' -M %i -R "rusage[mem=%i]" -q medium_priority -n 1 -L /bin/bash' % (mem, mem)
    # set input, output and command as Ri with i the job number
    cmd += (' -o %s/R' % odir) + '%I'
    cmd += (' -e %s/R' % edir) + '%I'
    # actual job script
    cmd += ' %s/R\$LSB_JOBINDEX' % sdir
    cmd += ' ;\n'

    print 'JobArrayScript ', cmd
    
    file.write('#!/bin/bash\n')
    file.write(cmd)
    file.close()
    os.chmod(name, 0700)
    return name


def submitJobArray(first, last, mem=1024):
    """submit the script written by 'writejobArrayScript' to the LSF queue"""
    callname = writeJobArrayScript(first, last, mem)
    print 'submit ', callname
    val = os.system(callname)
    if not val == 0:
        print "bsub failed with value ", val


def main(args):
    """submit LSF jobs, depending on the arguments provided"""
    
    #check bsub command:
    if not os.path.isfile(bsub):
        sys.stderr.write("Error: bsub '%s' not found\n" % bsub)
        return

    # make directories if necessary
    makeDirectory(edir)
    makeDirectory(odir)
    makeDirectory(sdir)
    makeDirectory(base+'/done')
    makeDirectory(base+'/save')
    
    cnt  = 1
    id   = 1
    name = ''
    preconf = ''
    for arg in args:
        if os.path.isfile(arg) and arg.endswith('.py'):
            preconf = arg
        elif arg.isdigit():
            cnt = int(arg)
        elif os.path.isfile(arg) and arg.endswith('.cym'):
            name = writeJobScript(id, arg, preconf, cnt)
            id += 1
        elif arg.startswith('mem='):
            mem = int(arg[4:])
            print mem
        elif arg == "clear":
            print("rm -f %s/R* %s/cytosim.lsf" % (sdir, sdir))
            os.system("rm -f %s/R* %s/cytosim.lsf" % (sdir, sdir))
            sys.exit()
        else:
            print("I do not understand argument [%s] : abort" % arg)
            sys.exit()
    
    if id > 1:
        print("submit_lsf.py created %i scripts in %s" % (id, sdir))
        submitJobArray(1, id-1, mem)
    elif not name == '':
        submitJob(name, mem);
    else:
        print("no job submitted")


#------------------------------------------------------------------------

if __name__ == "__main__":
    if len(sys.argv) < 2 or sys.argv[1].endswith("help"):
        print(__doc__)
    else:
        main(sys.argv[1:])

