#!/usr/bin/env python3
#
# plot_cpu.py
#
# plot CPU usage stored in the last line of 'messages.cmo'
#
# Copyright F. Nedelec, 30.11.2018

"""
    Plot the CPU data contained in 'messages.cmo' in different directories

Syntax:

    plot_cpu.py DIRECTORIES

Description:

    This script makes a plot using matplotlib

F. Nedelec, 20 November 2018
"""

import sys, os, subprocess, math
import matplotlib.pyplot as plt
from pyned import uncode

title = os.path.basename(os.getcwd())

#------------------------------------------------------------------------

def get_cpu(filename):
    """ 
        This extracts parameters from the specified file, generated by
        get_platelet.py run???? > scaling.txt
    """
    try:
        last = []
        with open(filename, 'r') as f:
            for line in f:
                prev = last
                last = line
        d = uncode(last).split()
        if len(d) > 1 and d[0] == "end":
            return int(d[1])
        d = uncode(prev).split()
        return int(d[0])
    except:
        sys.stderr.write("  missing file `%s'\n" % filename)
    return []


#------------------------------------------------------------------------

def main(args):
    scale = 1
    paths = []
    for arg in args:
        if os.path.isdir(arg):
            paths.append(arg)
        elif arg == "hour":
            scale = 1.0/3600
        elif arg == "3600":
            scale = 1.0/3600
        else:
            sys.stderr.write("  Error: unexpected argument `%s'\n" % arg)
            sys.exit()
    if not paths:
        paths = ['.']
    # create figure:
    fig, ax = plt.subplots(figsize=(7,5))
    cdir = os.getcwd()
    colors = ['blue', 'black', 'red', 'purple', 'green', 'yellow', 'orange', 'brown', 'gray', 'pink']
    for i, p in enumerate(paths):
        os.chdir(cdir)
        c = get_cpu(p+'/messages.cmo')
        if c:
            col = colors[i%9]
            ax.plot(i, c*scale, 'o', markersize=4, markerfacecolor=col, markeredgecolor=col, markeredgewidth=0, alpha=0.5)
    os.chdir(cdir)
    # complete figure:
    #ax.set_xlim(0, 18)
    ax.set_xlabel('Simulation')
    ax.set_ylim(0, ax.get_ylim()[1])
    if scale == 1:
        ax.set_ylabel('CPU (sec)')
        ax.set_title('CPU (sec)')
    else:
        ax.set_ylabel('CPU (hour)')
        ax.set_title('CPU (hour)')
    fig.savefig('cpu.png', dpi=300)
    plt.close()

#------------------------------------------------------------------------

if __name__ == "__main__":
    if len(sys.argv) < 2 or sys.argv[1].endswith("help"):
        print(__doc__)
    else:
        main(sys.argv[1:])

