#!/usr/bin/env python3

# How I installed matplotlib on 8.12.2020 (Mac osx 10.14):
#
# brew install python3
# pip3 install numpy
# pip3 install nose
# pip3 install matplotlib

"""
    Plot iteration count contained in 'messages.cmo' in different directories

Syntax:

    plot_count.py DIRECTORIES

Description:

    This script makes a plot using matplotlib

F. Nedelec, 8.12.2020
"""

import sys, os, subprocess, math
import matplotlib.pyplot as plt
from pyned import uncode

title = os.path.basename(os.getcwd())

#-------------------------------------------------------------------------------

def get_data(filename):
    """ 
        This extracts parameters from the specified file, generated by
        get_platelet.py run???? > scaling.txt
    """
    F = 0
    G = -1
    res = []
    with open(filename, 'r') as f:
        for line in f:
            str = uncode(line)
            if str[0] == "F":
                F += 1
            else:
                x = str.find(" count ");
                if x > 0:
                    C = int(str[x+8:].split()[0])
                    #sys.stderr.write("count %i\n" % C)
                    if G == F:
                        res.append([F, ( C + oC ) * 0.25])
                    else:
                        G = F
                        oC = C
    return res


#-------------------------------------------------------------------------------

def main(args):
    paths = []
    for arg in args:
        if os.path.isdir(arg):
            paths.append(arg)
        else:
            sys.stderr.write("  Error: unexpected argument `%s'\n" % arg)
            sys.exit()
    if not paths:
        paths = ['.']
    # create figure:
    fig = plt.figure(figsize=(7,5))
    ax = plt.axes()
    colors = ['blue', 'black', 'red', 'purple', 'green', 'yellow', 'orange', 'mustard', 'gray', 'pink']
    for i, p in enumerate(paths):
        data = get_data(p+'/messages.cmo')
        if data:
            x, y = zip(*data)
            print('mean(count) = %f' % (sum(y)/len(y)))
            col = colors[i%10]
            ax.plot(x, y, 'o', markersize=2, markerfacecolor=col, markeredgecolor=col, markeredgewidth=0, alpha=0.25)
    # complete figure:
    ax.set_xlim(0, ax.get_xlim()[1])
    ax.set_ylim(0, ax.get_ylim()[1])
    ax.set_ylim(0, 50)
    ax.set_xlabel('Frame')
    ax.set_ylabel('Count')
    ax.set_title('Iterations')
    fig.savefig(p+'/count.png', dpi=300)
    plt.close()

#-------------------------------------------------------------------------------

if __name__ == "__main__":
    if len(sys.argv) < 2 or sys.argv[1].endswith("help"):
        print(__doc__)
    else:
        main(sys.argv[1:])

