#lang racket

#|
"05_locations.rkt"
|#

(require racket/hash)
(require "../common/common.rkt")
(require "../common/term.rkt")
(require "../common/closure.rkt")
(provide (all-defined-out))

(define gensym-proxy-pool (box (list)))
(define (gensym-proxy x)
  (if (empty? (unbox gensym-proxy-pool))
      (gensym x)
      (let* ([result (car (unbox gensym-proxy-pool))]
             [_      (set-box! gensym-proxy-pool (cdr (unbox gensym-proxy-pool)))])
            result)))

(struct todo (t) #:transparent)
(struct done (v) #:transparent)

(struct Cache (l t) #:transparent)

(struct conf (d t s σ) #:transparent)

(define (load-term t)
  (conf '▿ (*closure t (hasheq)) '() (hasheq)))

(define (unload-value cf)
  (match cf
  [(conf '▵ v '() _) v]
  [_                #f]))

(define (rule-no-and-trans c)
  (match c
  [(conf '▿ (*closure (*app u v) e) s σ)          (cons 1 (conf '▿ (*closure u e) (cons (*app (*var 'hole) (*closure v e)) s) σ))] 
  [(conf '▿ (*closure (*lam x t) e) s σ)          (cons 2 (let ([ll (gensym-proxy 'l)])
                                     (conf '▵ (Cache ll (*closure (*lam x t) e)) s (hash-set σ ll (todo #f)))))]
  [(conf '▿ (*closure (*var x)   e) s σ) (let ([ll (hash-ref e x #f)]) (match (hash-ref σ ll (done (*var x)))
      [(todo (*closure v e))       (cons 3 (conf '▿ (*closure v e) (cons (Cache ll (*var 'hole)) s) σ))]
      [(done t)                    (cons 4 (conf '▵ t s σ))]))]
  [(conf '▵ t (cons (Cache ll (*var 'hole)) s) σ)    (cons 5 (conf '▵ t s (hash-set σ ll (done t))))]
  [(conf '▵ (Cache _ (*closure (*lam x t) e1)) (cons (*app (*var 'hole) ve) s) σ)
                                                       (cons 6 (let ([xx (gensym-proxy x)])
                                                       (conf '▿ (*closure t (hash-set e1 x xx)) s (hash-set σ xx (todo ve)))))]
  [(conf '▵ (Cache ll (*closure (*lam x t) e)) s                               σ) #:when (equal? (hash-ref σ ll) (todo #f))
                                                       (cons 7 (let ([x1 (gensym-proxy x)]
                                                                     [xx (gensym-proxy x)])
                                                       (conf '▿ (*closure t (hash-set e x xx))
                                                           (list* (*lam x1 (*var 'hole)) (Cache ll (*var 'hole)) s)
                                                           (hash-set σ xx (done (*var x1))))))]
  [(conf '▵ (Cache ll (*closure (*lam x t) e)) s                              σ) (cons 8 (conf '▵ (done-v (hash-ref σ ll)) s σ))]
  [(conf '▵ t (cons (*app (*var 'hole) (*closure v e)) s) σ)
                                        (cons 9 (conf '▿ (*closure v e) (cons (*app t (*var 'hole)) s) σ))]
  [(conf '▵ v (cons (*app t  (*var 'hole)) s) σ)  (cons 10 (conf '▵ (*app  t v) s σ))]
  [(conf '▵ v (cons (*lam x1 (*var 'hole)) s) σ)  (cons 11 (conf '▵ (*lam x1 v) s σ))]
  [(conf '▵ v '() σ) (cons #f #f)]))

(define (rule-no c)
  (let* ([original-pool (unbox gensym-proxy-pool)]
         [result        (car (rule-no-and-trans c))]
         [_             (set-box! gensym-proxy-pool original-pool)])
  result))

(define (trans c)
  (cdr (rule-no-and-trans c)))

(define (normalize t)
  (unload-value (for-each-state (load-term t) trans identity)))

; potential function

(define (phi-t t)
  (match t
  [(*app t1 t2)    (+ 3 (phi-t t1) (phi-t t2))]
  [(*lam x t)      (+ 4 (phi-t t))]
  [(*var x)        2]))

(define (phi-v v)
  (match v
  [(Cache l (*closure (*lam x t) e))   1]
  [t                                   0]))

(define (phi-s s)
  (match s
  ['()                                             0]
  [(cons (*app (*var 'hole) (*closure t e)) s)     (+ 2 (phi-s s) (phi-t t))]
  [(cons (*app t (*var 'hole)) s)                  (+ 1 (phi-s s))]
  [(cons (*lam x (*var 'hole)) s)                  (+ 1 (phi-s s))]
  [(cons (Cache l (*var 'hole)) s)                 (+ 1 (phi-s s))]))

(define (locations-v v σ)
  (match v
  [(Cache l (*closure (*lam x t) e))  (hash-set (locations-e (hash->list e) σ) l t)]
  [t                                  (hasheq)]))

(define (locations-l l σ)
  (match (hash-ref σ l)
  [(done v)               (locations-v v σ)]
  [(todo #f)              (hasheq)]
  [(todo (*closure t e))  (locations-e (hash->list e) σ)]))

(define (locations-e e σ)
  (match e
  ['()                    (hasheq)]
  [(cons (cons x l) e)    (hash-matching-union (hash-set (locations-e e σ) l #f) (locations-l l σ))]))

(define (locations-s s σ)
  (match s
  ['()                                             (hasheq)]
  [(cons (*app (*var 'hole) (*closure t e)) s)     (hash-matching-union (locations-s s σ) (locations-e (hash->list e) σ))]
  [(cons (*app t (*var 'hole)) s)                  (locations-s s σ)]
  [(cons (*lam x (*var 'hole)) s)                  (locations-s s σ)]
  [(cons (Cache l (*var 'hole)) s)                 (hash-set (locations-s s σ) l #f)]))

(define (active-locations-s s)
  (foldr (binflip set-add) (seteq) (map Cache-l (filter Cache? s))))

(define (locations-k k)
  (match k
  [(conf '▿ (*closure t e) s σ)  (hash-matching-union (locations-s s σ) (locations-e (hash->list e) σ))]
  [(conf '▵              v s σ)  (hash-matching-union (locations-s s σ) (locations-v v σ))]))
    
(define (inactive-locations-k k)
  (hash-difference-set (locations-k k) (active-locations-s (conf-s k))))

(define (phi-σ k)
  (define (entry-phi kv)
    (match kv
    [(cons a v)
    (match (cons (hash-ref (conf-σ k) a) v)
    [(cons (done              _) _)     0]
    [(cons (todo (*closure t e)) #f)    (phi-t t)]
    [(cons (todo #f) v)                 (+ 2 (phi-t v))])]))
  (apply + (map entry-phi (hash->list (inactive-locations-k k)))))

(define (phi-k k)
  (match k
  [(conf '▿ (*closure t e) s σ) (+ (phi-t t) (phi-s s) (phi-σ k))]
  [(conf '▵                       v s σ) (+ (phi-v v) (phi-s s) (phi-σ k))]))

; decoding

(define (decode-k k)
  (define (plug s t)
    (foldl
      (λ (f t)
        (match f
          [(Cache _ _) t]
          [_         (subst 'hole t (decode-t f))]))
      t s))
  (define (decode-t t)
    (match t
      [(*var x)   (*var x)]
      [(*app l r) (*app (decode-t l) (decode-t r))]
      [(*lam x t) (*lam x (decode-t t))]
      [(*closure t e) (subst-closure decode-l (*closure t (hash-map e list)))]
      [(Cache l (*closure (*lam x t) e)) (subst-closure decode-l (*closure (*lam x t) (hash-map e list)))]))
  (define (decode-l l)
    (match (hash-ref (conf-σ k) l)
      [(todo (*closure t e)) (subst-closure decode-l (*closure t (hash-map e list)))]
      [(done t) (decode-t t)]))
  (plug (conf-s k) (decode-t (conf-t k))))

; printing

(define (raw-plug s t)
  (define (ext-subst x s t)
    (match t
      [(*app l r) (*app (subst x s l) (ext-subst x s r))]
      [(*lam y m) (if (equal? x y) t (*lam y (ext-subst x s m)))]
      [(*var y  ) (if (equal? x y) s t)]
      [(Cache l r) (Cache l (subst x s r))]
      [e  e]))
  (foldl (λ (f t) (ext-subst 'hole t f)) t s))

(struct Cut (t d) #:transparent)

(define (conf-alive-σ k)
  (hash-intersect (conf-σ k) (locations-k k) #:combine (λ (x y) x)))

(define (to-unicode a)
  (define (string-list xs)
    (format "[~a]" (string-join xs  ", ")))
  (match a
  ; constants
  [(*lam 'x (*var 'x)) "I"]
  [(*app (*lam 'x (*app (*var 'x) (*var 'x))) (*lam 'x (*app (*var 'x) (*var 'x)))) "Ω"]
  [e #:when (equal? e (hasheq 'x '𝕩 'y '𝕪 'z '𝕫)) "e_xyz"]

  ; constructors
  [(conf d t s σ)  (format "〈~a, ~a, σ〉~a" (to-unicode t) (to-unicode s) d)]
  [(*app t1 t2)   (format "(~a) ~a" (to-unicode t1) (to-unicode t2))]
  [(*lam x t)     (format "λ~a.~a" x (to-unicode t))]
  [(*var 'hole)   (format "□")]
  [(*var x)       (format "~a" x)]
  [(*closure t e) (format "(~a, ~a)" (to-unicode t) (to-unicode e))]
  [(todo a)       (format "~a✗" (to-unicode a))]
  [(done a)       (format "~a✓" (to-unicode a))]
  [(Cache l t)    (format "~a:=~a" (to-unicode l) (to-unicode t))]
  [(Cut t d)      (format "〈~a〉~a" (to-unicode t) d)]
  [#f             "⊥"]
  [xs #:when (hash? xs)  (string-list (hash-map xs (λ (k v) (format "~a↦~a" k (to-unicode v)))))]
  [xs #:when (list? xs)  (string-list (map to-unicode xs))]
  [x  #:when (symbol? x) (format "~a" x)]
  ))

(define (to-latex a)
  (define (string-list strs)
    (string-append "[" (string-join strs ", ") "]"))
  (match a
  ; constants
  [(*app (*lam 'x (*app (*var 'x) (*var 'x))) (*lam 'x (*app (*var 'x) (*var 'x)))) "\\Omega"]
  [(*lam 'x (*var 'x)) "I"]
  [(*lam 'y (*lam 'z (*app (*lam 'x (*var 'x)) (*var 'z)))) "A"]
  [(*closure (*var 'x) e) #:when (equal? e (hasheq 'x '\{\\mathbbm\{x\}\})) "x^{\\mathbbm{x}}"]
  [e #:when (equal? e (hasheq 'y '\{\\mathbbm\{y\}\} 'z '\{\\mathbbm\{z\}\})) "e_\\mathit{yz}"]
  [e #:when (equal? e (hasheq 'y '\{\\mathbbm\{y\}\} 'z '\{\\mathbbm\{z\}\} 'x '\{\\mathbbm\{w\}\})) (format "e_\\mathit{yz}\\!\\ast\\!~a" (to-latex (hasheq 'x '\{\\mathbbm\{w\}\})))]
  [σ #:when (and (hash? σ) (done? (hash-ref σ '\{\\mathbbm\{x\}\} #f)))
            (format "\\,\\sigma_1 \\!\\ast\\! ~a" (to-latex (hash-difference-set σ (seteq '\{\\mathbbm\{x\}\} '\{\\mathbbm\{y\}\}))))]

  ; constructors
  [(conf '▿ (*closure t e) s σ)(format "\\langle {~a}, {~a}, {~a}\\rangle_\\etrian" (to-latex (*closure t e)) (to-latex s) (to-latex (conf-alive-σ a)))]
  [(conf '▵ v s σ)  (format "\\langle {~a}, {~a}, {~a}\\rangle_\\ctrian" (to-latex v) (to-latex s) (to-latex (conf-alive-σ a)) )]
  [(*app t1 t2)   (format "{\\tapp {(~a)} {~a}}" (to-latex t1) (to-latex t2))]
  [(*lam x t)     (format "{\\tlam {~a} {~a}}" x (to-latex t))]
  [(*var 'hole)   (format "\\hole")]
  [(*var x)       (format "~a" x)]
  [(*closure t e) (format "{\\clos {~a} {\\!~a}}" (to-latex t) (to-latex e))]
  [(box c)        (format "\\{b\\}")]
  [(todo a)       (format "{\\tobedone {~a}}" (to-latex a))]
  [(done a)       (format "{\\done {~a}}" (to-latex a))]
  [(Cache l t)    (format "{{~a}\\!:=\\!{~a}}" (to-latex l) (to-latex t))]
  [(Cut t d)      (format "{\\cut {~a}_~a}" (to-latex t) (if (eq? d '▿) "\\etrian" "\\ctrian"))]
  [#f             "\\bot"]
  [xs #:when (hash? xs)  (string-list (hash-map xs (λ (k v) (format "~a\\!\\mapsto\\!~a" k (to-latex v)))))]
  [(cons x xs)    (format "{\\cons {~a} {~a}}" (to-latex x) (to-latex xs))]
  [xs #:when (list? xs)  (string-list (map to-latex xs))]
  [x  #:when (symbol? x) (format "~a" x)]
  ))

(define (to-refocus k)
  (match k
  [(conf d t s σ) (raw-plug s (Cut t d))]))

(define counter (box 0))
(define (post-incr b)
  (define result (unbox b))
  (set-box! b (+ result 1))
  result)

(define (generate-plot-data)
  (ignore (for-each-state (load-term
    *RKNL-example
    ) trans (λ (k) (printf "~a;~a;~a~n" (post-incr counter) (phi-k k) (phi-σ k))))))

; (generate-plot-data)

(define (generate-example-decodings)
  (set-box! gensym-proxy-pool (list '𝕒 '𝕩 '𝕓 '𝕪 '𝕕 'z₀ '𝕫  '𝕖 '𝕨))
  (ignore (for-each-state (load-term
    *RKNL-example
    ) trans (λ (k) (printf "~a~n" (to-unicode (decode-k k)))))))

(define (generate-example-unicode)
  (set-box! gensym-proxy-pool (list '𝕒 '𝕩 '𝕓 '𝕪 '𝕕 'z₀ '𝕫  '𝕖 '𝕨))
  (ignore (for-each-state (load-term
    *RKNL-example
    ) trans (λ (k) (printf "~a: ~a | ~a →(~a)~n~n" (post-incr counter) (to-unicode (to-refocus k)) (to-unicode (conf-alive-σ k)) (rule-no k))))))
  
(define (generate-example-latex)
  (set-box! gensym-proxy-pool (list '\{\\mathbbm\{a\}\} '\{\\mathbbm\{x\}\} '\{\\mathbbm\{b\}\} '\{\\mathbbm\{y\}\} '\{\\mathbbm\{d\}\} '\{z_0\} '\{\\mathbbm\{z\}\}  '\{\\mathbbm\{e\}\} '\{\\mathbbm\{w\}\}))
  (ignore (for-each-state (load-term
    *RKNL-example
    ) trans (λ (k) (printf "~a:&& ~a &| ~a &&\\stackrel{(\\ref{tr:~a})}{\\to}\\\\[-0.5ex]~n%~n" (post-incr counter) (to-latex (to-refocus k)) (to-latex (conf-alive-σ k)) (rule-no k))))))

; tests

(require rackunit)
(require "../common/term-checks.rkt")

(check-normalizer-of? normalize complete-full-normal-forms)
(check-normalizer-of? normalize open-normal-forms)

(define (steps-for-family family ns)
  (map (λ (n) (count-bounces (load-term (family n)) trans)) ns))

(check-equal? ; c_n c_2 I
  (steps-for-family (λ (n) (*appseq (*Church n) (*Church 2) *id)) (range 1 10))
  (map (λ (n) (+ (* 10 (expt 2 n)) (* 5 n) 5)) (range 1 10)))
(check-equal? ; pred c_n
  (steps-for-family (λ (n) (*app *Church-Cregut-pred (*Church n))) (range 1 10))
  (map (λ (n) (+ (* 30 n) 41)) (range 1 10)))
(check-equal?
  (steps-for-family (λ (n) (*app *Church-pred (*Church n))) (range 2 10))
  (map (λ (n) (+ (* 37 n) 16)) (range 2 10)))
(check-equal? ; λx. c_n ω x
  (steps-for-family *explode1 (range 1 10))
  (map (λ (n) (+ (* 9 n) 15)) (range 1 10)))
(check-equal?
  (steps-for-family *explode3 (range 1 10))
  (map (λ (n) (+ (* 18 n) 14)) (range 1 10)))
(check-equal? ; c_n dub I
  (steps-for-family *explode4 (range 1 10))
  (map (λ (n) (+ (* 18 n) 15)) (range 1 10)))
(check-equal? ; c_n dub (λx. I x)
  (steps-for-family *implode2 (range 1 10))
  (map (λ (n) (+ (* 18 n) 20)) (range 1 10)))

(check-equal? (phi-t *id)    6)
(check-equal? (phi-t *omega) 11)
(check-equal? (phi-t *Omega) 25)
(check-equal? (phi-t (*app (*lam 'y (*lam 'z (*app *id (*var 'z)))) *Omega)) 47)
(check-equal? (phi-t (*lam 'x (*appseq (*var 'c) (*var 'x) (*var 'x)))) 16)
(check-equal? (phi-t *RKNL-example) 66)

