#lang racket

#|
"machine.rkt"
|#

(require "../common/common.rkt")
(require "../common/term.rkt")
(require "../common/closure.rkt")
(provide (all-defined-out))

(struct V (n) #:transparent)
(struct Lam () #:transparent)
(struct Mark (t n) #:transparent)
(struct conf (t s e n) #:transparent)

(define (load-term t)
  (conf t '() '() 0))

(define (unload-value cf)
  (match cf
  [(conf (Mark t n) e '() m)  t]
  [_                         #f]))

(define (trans c)
  (match c
  ; [(conf (*app t1 (*var n)) e s m)                     (conf t1 e (cons (list-ref e n) s) m)]
  [(conf (*app t1 t2) e s m)                           (conf t1 e (cons (*closure t2 e) s) m)] ; 1
  [(conf (*lam '() t) e (cons (*closure t1 e1) s) m)   (conf t (cons (*closure t1 e1) e) s m)] ; 2
  [(conf (*lam '() t) e                        s  m)   (conf t (cons (*closure (V (+ m 1)) '()) e) (cons (Lam) s) (+ m 1))] ; 3
  ; [(conf (*var n) e0 s m)                              (let ([te (list-ref e0 n)]) (conf (*closure-t te) (*closure-e te) s m))]
  [(conf (*var 0) (cons (*closure t e) e1) s m)        (conf t e s m)] ; 4
  [(conf (*var n) (cons             c  e ) s m)        (conf (*var (- n 1)) e s m)] ; 5
  [(conf (V n) e s m)                                  (conf (Mark (*var (- m n)) m) e s m)] ; 6
  [(conf (Mark t n) e '() m)                           #f] ; 7
  [(conf (Mark t n) e (cons (*closure t1 e1) s) m)     (conf t1 e1 (cons (Mark t n) s) m)] ; 8
  [(conf (Mark t n) e (cons            (Lam) s) m)     (conf (Mark (*lam '() t) n) e s m)] ; 9
  [(conf (Mark t n) e (cons     (Mark t1 n1) s) m)     (conf (Mark (*app t1 t) n1) e s m)] ; 10
))

(define (normalize t)
  (unload-value (for-each-state (load-term t) trans identity)))

(require rackunit)
(require "../common/term-checks.rkt")

(check-normalizer-of? normalize
  (named->index-normal-forms complete-full-normal-forms))

(define (is-beta-step c)
  (match c
  [(conf (*lam '() _) _ (cons (*closure t1 e1) _) _) #t]
  [_                                                 #f]))

(define (beta-steps-count t)
  (define (aux acc c)
    (let ([c1 (trans c)])
    (if (eq? c1 #f)
        acc
        (aux (+ acc (if (is-beta-step c) 1 0)) c1))))
  (aux 0 (load-term (named->index t))))

(define (beta-steps-for-family family ns)
  (map (λ (n) (beta-steps-count (family n))) ns))

(check-equal?
  (beta-steps-for-family (λ (n) (*app *Church-pred (*Church n))) (range 1 10))
  (map (λ (n) (+ (* 7 n) 3)) (range 1 10)))

(check-equal? ; c_n c_2 I
  (beta-steps-for-family (λ (n) (*appseq (*Church n) (*Church 2) *id)) (range 0 10))
  (map (λ (n) (- (* 3 (expt 2 n)) 1)) (range 0 10)))
(check-equal? ; pred c_n
  (beta-steps-for-family (λ (n) (*app *Church-Cregut-pred (*Church n))) (range 0 10))
  (map (λ (n) (+ (* 6 n) 8)) (range 0 10)))
(check-equal? ; λx. c_n ω x
  (beta-steps-for-family *explode1 (range 0 10))
  (map (λ (n) (+ (expt 2 n) 1)) (range 0 10)))
(check-equal? ; c_n dub I
  (beta-steps-for-family *explode4 (range 0 10))
  (map (λ (n) (+ (expt 2 n) 1)) (range 0 10)))
(check-equal? ; c_n dub (λx. I x)
  (beta-steps-for-family *implode2 (range 0 10))
  (map (λ (n) (+ (* 2 (expt 2 n)) 1)) (range 0 10)))

(define (steps-for-family family ns)
  (map (λ (n) (count-bounces (load-term (named->index (family n))) trans)) ns))

(check-equal? ; c_n c_2 I
  (steps-for-family (λ (n) (*appseq (*Church n) (*Church 2) *id)) (range 0 10))
  (map (λ (n) (- (* 15 (expt 2 n)) 6)) (range 0 10)))
(check-equal? ; pred c_n
  (steps-for-family (λ (n) (*app *Church-Cregut-pred (*Church n))) (range 1 10))
  (map (λ (n) (+ (* 26 n) 25)) (range 1 10)))
(check-equal? ; λx. c_n ω x
  (steps-for-family *explode1 (range 0 10))
  (map (λ (n) (- (* 12 (expt 2 n)) 3)) (range 0 10)))
(check-equal?
  (steps-for-family *explode3 (range 0 10))
  (map (λ (n) (- (* 21 (expt 2 n)) 12)) (range 0 10)))
(check-equal? ; c_n dub I
  (steps-for-family *explode4 (range 0 10))
  (map (λ (n) (- (* 23 (expt 2 n)) 14)) (range 0 10)))
(check-equal? ; c_n dub (λx. I x)
  (steps-for-family *implode2 (range 0 10))
  (map (λ (n) (- (* 26 (expt 2 n)) 14)) (range 0 10)))

