'''
This file contains the functions that given all projections with 
all possible orderings, return the ordering that would have been
choose by the desired heuristic.
'''


from math import factorial
from heuristics_rules import *
from heuristic_tools import greedy_heuristics, expensive_heuristics, create_pseudorderings

def choose_order_given_projections(projections, heuristic="gmods"):
    '''Returns the order guessed by the heuristic requested'''
    if heuristic in greedy_heuristics or type(heuristic)==int:
        guess = greedy_heuristic_guess(projections, heuristic=heuristic)
        return guess
    elif heuristic in expensive_heuristics:
        return no_greedy_heuristic_guess(projections, heuristic=heuristic)
    else:
        raise Exception('Heuristic not recognised.')


def greedy_heuristic_guess(projections:list, heuristic:str="gmods"):
    '''
    This function is especialized in greedy heuristics.
    One variable is picked at a time, adjusting the ordering accordingly.
    '''
    order = 0 # we start assuming that the best order is the first one
    nvar = len(projections[0]) # the number of variables corresponds with the length of the list describing one of the projections
    
    for i in range(nvar):
        # projections[order] is the projection that if chosen order we assume to be the best. All orders we can still choose from are equal to this one until this point
        try:
            if heuristic != 'greedy_sotd':
                new_var=greedy_choose_variable(projections[order][i], heuristic=heuristic)
            elif i < nvar-1:
                new_var=greedy_choose_variable([projections[ordering][i+1] for ordering in range(factorial(nvar)) if projections[ordering][i]== projections[order][i]], heuristic=heuristic)
            else:
                new_var=0
        except IndexError:
            # The reason of this error is probably that the computation of the projection did not go further, in this case we return the current order
            return order
        
        if type(new_var)==str:
            return order
        order = order + factorial(nvar-i-1) * new_var # the best order is updated with the new information
    return order # the final best order is returned


def no_greedy_heuristic_guess(projections:list, heuristic:str="old_mods"):
    '''
    Looking at the same time at all the projections, 
    the no greedy heuristics make an ordering choice.
    '''
    if heuristic == "sotd":
        sotd_values = [sum([degree for level in projection for polynomial in level for monomial in polynomial for degree in monomial[:-1]]) for projection in projections]
        return min(range(len(sotd_values)), key=sotd_values.__getitem__) # returns the index with the smallest value in the list sotd_values
    elif heuristic in ["old_mods", "logmods", "mods", "acc_logmods"]:
        nvar = len(projections[0])
        pseudorderings = create_pseudorderings(nvar)
        relevant_degrees = [[[max([monomial[var] for monomial in polynomial]) for polynomial in level] for level,var in zip(projection,pseudordering)] for projection, pseudordering in zip(projections, pseudorderings)] # This returns a list of lists, each of those lists correspond to a projection. Those lists contain lists of the degrees of the polynomials in each level wrt the variable that will be projected after.
        heuristic_dict = {'old_mods':old_mods_guess, 'mods':mods_guess,'logmods':logmods_guess, 'acc_logmods':acc_logmods_guess}
        return heuristic_dict[heuristic](relevant_degrees)
    else:
        raise Exception("Heuristic "+heuristic+" not found.")
