#! /usr/bin/env python3

'''
Author: Owen Larson
Description: Python script to create shapefiles from XY text data for:
    Aftershock regions of Aleutian–Alaska megathrust earthquakes, 1938–2021
    Carl Tape, Anthony Lomax
    https://doi.org/10.1029/2022JB024336
'''


import arcpy, os, re, shutil

#Function for inserting Lat and Lon data
def process_txt(txtfile):
    txt = open(txtfile, "r")
    inputTxt = txt.readlines()
    cursor = arcpy.da.InsertCursor(newfc, ["SHAPE@XY"])
    print("Creating point shapefile for:", txtfile)
    for line in inputTxt:
        if line.startswith(">"):
            continue
        xCoord, yCoord = re.findall("[0-9]*\.[0-9]+", line)
        xNum = float(xCoord)
        yNum = float(yCoord)
        newrow = [(xNum, yNum)]
        cursor.insertRow(newrow)
    txt.close()
    del cursor

#Set this to wherever the text files live
os.chdir(r'C:\Users\ojlarson\Desktop\aftershock_regions')
cwd = os.getcwd()
print('Current working directory set to: ' + cwd)

arcpy.env.overwriteOutput = True
arcpy.env.workspace = cwd

#Set this to a NEW, EMPTY directory - this folder and all files within will be deleted
outpath = r"C:\ArcpyOutput"
sr = arcpy.SpatialReference(4326)# GCS_WGS_84 Spatial Reference WKID
print('Output set to ', outpath, '.')

#Check for existence of directory and DELETES ALL FILES WITHIN
if os.path.exists(outpath):
    shutil.rmtree(outpath)
#Create/recreate directory
os.mkdir(outpath)

#Create list of text files
txtList = []
for file in os.listdir(cwd):
    if re.match('tapelomax*', file):
        txtList.append(file)

#Create point feature class, using process_txt() function
for file in txtList:
    newpt = file.rsplit('.',1)[0] + '_point.shp'
    newfc = arcpy.CreateFeatureclass_management(outpath, newpt, "Point", "", "", "", sr)
    process_txt(file)
    print("New point shapefile created: ", newfc)    

#Set arcpy working directory to output location
arcpy.env.workspace = outpath

#Project point shapefiles to NAD 1927, create line and polygon shapefiles for non-composite features
fcList = arcpy.ListFeatureClasses("", "Point")
for fc in fcList:
    if re.match("^tapelomax_\d{4}_point.shp", fc):# and not fc.endswith("NAD.shp"):
        fcNAD = fc[:-4] + "_NAD.shp"
        outSR = arcpy.SpatialReference(102117) #102117 is the wkid of NAD 1927 Alaska Albers (meters)
        arcpy.management.Project(fc, fcNAD, outSR)
        newLine = fc[:-9] + "line.shp"
        newPoly = fc[:-9] + "poly.shp"
        arcpy.management.PointsToLine(fcNAD, newLine, "", "", "CLOSE")
        print("New line shapefile created: ", newLine)
        arcpy.management.FeatureToPolygon(newLine, newPoly)
        print("New polygon shapefile created: ", newPoly)
        path = os.path.join(outpath, fc[:-10])
        os.mkdir(path)

#Merge all polygons with field for year
mergeList = []                            
polyList = arcpy.ListFeatureClasses("", "Polygon")
for fc in polyList:
    if not fc.endswith("Merge.shp"):
        fcMerge = fc[:-4] + "_Merge.shp"
        arcpy.management.CopyFeatures(fc, fcMerge)
        arcpy.AddField_management(fcMerge, "Year", "TEXT", field_length = 50)
        with arcpy.da.UpdateCursor(fcMerge, "Year") as cursor:
            for row in cursor:
                row[0] = re.findall("\d{4}", fc)[0]
                cursor.updateRow(row)
        mergeList.append(fcMerge)
arcpy.management.Merge(mergeList, "tapelomax_eqboundaries.shp")
print("Merged polygon created.")

#Create folders by year, copy shapefiles to respective directories and delete original files
fcList = arcpy.ListFeatureClasses()
mergeFolder = os.path.join(outpath, "Merged Shapefiles")
os.mkdir(mergeFolder)
for fc in fcList:
    if "NAD" in fc:
        arcpy.management.Delete(fc)
        print("Deleted", fc)
    elif "Merge" in fc:
        arcpy.management.Delete(fc)
        print("Deleted", fc)
    elif re.match("^tapelomax_\d{4}_\w*", fc):
        shortName = fc[:14]
        path = os.path.join(outpath, shortName, fc)
        arcpy.management.CopyFeatures(fc, path)
        arcpy.management.Delete(fc)
        print("Copied", fc, "to", path, "and deleted from main dir.")
    else:
        pathMerge = os.path.join(mergeFolder, fc)
        arcpy.management.CopyFeatures(fc, pathMerge)
        arcpy.management.Delete(fc)
        print("Copied", fc, "to", path, "and deleted from main dir.")
        
print("All shapefiles moved to directories by year.")

arcpy.env.overwriteOutput = False
