##PROCEDURE(help) SCPSimple
##HALFLINE This procedure solves the Simple Set Covering Problem (SCP) using a linear binary optimization and Maple\'s "Optimization:-Minimize" function.
##AUTHOR AmirHosein Sadeghimanesh
##DATE May 2022
##CALLINGSEQUENCE
##- SCPSimple( 'U', 'E' )
##PARAMETERS
##- 'U' : a set.
##- 'E' : a set, a list or an array of sets. The elements of 'E' must be subsets of 'U'.
##RETURNS(help)
##- Returns a set of positive integers.
##DESCRIPTION
##- A set covering problem (SCP) is the problem of covering a universal set, __"U"__, using a minimum number of sets in a cover __"E"__. In other words, __"E"__ is a subset of power set of __"U"__ and we want a subset of __"E"__ with the least possible number of sets such that their union is still __"U"__. SCP can be translated to a linear binary optimization problem. This procedure does this translation and then uses Maple\'s "Optimization:-Minimize" function with the option 'assume = binary' to solve this problem. The final output is the set of indices of sets in __"E"__ that are chosen in the solution to SCP. The solution is optimal and not an approximation. It means one can not find any other subset of __"E"__ with smaller cardinal that covers __"U"__.
##- Note that an optimal solution to SCP is not unique, there might be several subsets of __"E"__ covering __"U"__ and all with the same cardinal.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> U := {1, 2, 3, 4, 5}:
##> E := { {1, 2}, {1, 3, 5}, {2, 4}, {1, 4, 5} }:
##> SCPSimple( U, E );
##< [2, 3]
##SEEALSO
##- "SCPPack:-SCPRBasic"
##REFS
##- AmirHosein Sadeghimanesh, Matthew England, \"__"`An SMT solver for non-linear real arithmetic inside Maple/`"__\", 2022.

SCPSimple := proc(
    U :: set,
    E :: {set( set ), list( set ), Array( set )},
    n :: nonnegint := numelems( U ),
    m :: nonnegint := numelems( E )
    )
    :: list( posint ):

    description "This procedure takes two inputs. The first input is the universal set, U, that we want to cover it. The second input is a set of subsets of U, called E, that we want to cover U using a minimum number of the sets in E.":

	local 
		A          :: set,
        i          :: posint,
        j          :: posint,
        M          :: 'Matrix'( { 0, 1 } ),
        f,
        constrs    :: {list, set},
        sols       :: list,
        sols_point :: list( { 0, 1 } ),
        S          :: list( posint ):

	if `union`( seq( A, A in E ) ) <> U then
		error "Whether an element of U is not covered by any set in E or a set in E has an element out of U.":
	end if:

	M := Matrix( m, n ):
	for i from 1 by 1 to m do
		for j from 1 by 1 to n do
			if U[j] in E[i] then
				M[i,j] := 1:
			end if:
		end do:
	end do:

	f := add( cat(x__, i), i = 1..m ):

	constrs := [ seq( add( M[i,j] * cat(x__, i), i = 1..m ) >= 1, j = 1..n ) ]:
	constrs := convert(constrs, set): # to delete the repeated constraints.

	sols := Optimization:-Minimize( f, constrs, assume = binary ):
	sols_point := eval( [ seq( cat( x__, i ), i = 1..m ) ], sols[2] ):
	S :=  select( i -> sols_point[i] = 1, [ seq(j, j = 1..m) ]):

	return( S ):

end proc: