##PROCEDURE(help) SCPRNSAT_ConstrMaker
##HALFLINE This procedure is a sub-procedure of "SCPPack:-SCPRNSAT" and its task is to generate the SAT formulation of the constraints of the nonlinear binary optimization of the SCPR problem. It returns the result in the string form, it can be parsed to a logic formula (with logical operations of Maple\'s "Logic" package).
##AUTHOR AmirHosein Sadeghimanesh
##DATE May 2022
##CALLINGSEQUENCE
##- SCPRNSAT_ConstrMaker( 'M', 'N' )
##PARAMETERS
##- 'M' : a binary matrix.
##- 'N' : a binary matrix. Number of columns of 'N' must be the same as the number of rows of 'M'.
##RETURNS(help)
##- a string that can be parsed to a logical formula with logical operations from Maple\'s "Logic" package.
##DESCRIPTION
##- Consider a set covering problem with reasons, SCPR, with the universal set __"U"__, set of reasons __"R"__ and a labeled cover __"E &subset; `&Pscr;`(U) &times; `&Pscr;`(R)"__ where __"`&Pscr;`"__ stands for the power set. The goal of SCPR problem is to find a subset of __"R"__, such as __"S"__, with the minimum possible number of elements such that the union of sets in __"E"__ which their labels are subset of __"S"__, is equal to __"U"__. This problem can be translated to a binary optimization possibly nonlinear. If every label is a singleton, the optimization problem is linear and using "SCPPack:-SCPRBasic" is recommended. Otherwise "SCPPack:-SCPRNSAT" must be used. The later function translates the nonlinear binary optimization problem to a SAT problem. This procedure is a sub-procedure being called inside "SCPPack:-SCPRNSAT" to translate the constraints of the nonlinear binary optimization problem to a logical formula.
##- "SCPPack:-SCPRNSAT" first creates the binary membership matrices; __"M = [ a_(ij) ]_(m &times; n)"__ and __"N = [ v_(ij) ]_(r &times; m)"__ where __"a_(ij)=1"__ if and only if the __"j"__th element of __"U"__ belongs to the __"i"__th set in __"E"__, and __"v_(ij)=1"__ if and only if the __"i"__th element of __"R"__ belongs to the label of the __"j"__th set of __"E"__.
##- It procedure receives the two binary membership matrices of a SCPR introduced above, then returns a string that can be parsed to a logical formula that is equivalent with the constraints of optimization formulation of the SCPR problem.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> kernelopts( opaquemodules = false ):
##> M := Matrix([ [1, 1, 1, 1, 0, 0, 0, 0], [0, 0, 0, 0, 1, 1, 1, 1], [0, 1, 1, 1, 1, 1, 1, 0], [0, 0, 0, 0, 0, 0, 0, 1] ]):
##> N := Matrix([ [1, 1, 0, 0], [1, 0, 0, 0], [0, 1, 0, 0], [0, 0, 1, 0], [0, 0, 0, 1] ]):
##> SCPPack:-SCPRNSAT_ConstrMaker( M, N );
##< {"Logic:-`&and`( y__1, y__2 )", "Logic:-`&or`( Logic:-`&and`( y__1, y__2 ), y__4 )", "Logic:-`&or`( Logic:-`&and`( y__1, y__3 ), y__4 )", "Logic:-`&or`( Logic:-`&and`( y__1, y__3 ), y__5 )"}
##SEEALSO
##- "SCPPack:-SCPRNSAT"
##- "SCPPack:-ConjunctionMaker"
##- "SCPPack:-Exactly_k_truth_formula"
##REFS
##- AmirHosein Sadeghimanesh, Matthew England, \"__"`An SMT solver for non-linear real arithmetic inside Maple/`"__\", 2022.

SCPRNSAT_ConstrMaker := proc(
    M :: 'Matrix'( { 0, 1 } ),
    N :: 'Matrix'( { 0, 1 } ),
    n :: posint := LinearAlgebra:-ColumnDimension( M ),
    r :: posint := LinearAlgebra:-RowDimension( N ),
    m :: posint := LinearAlgebra:-RowDimension( M )
    )
    :: set( string ):

    description "This procedure recieves two binary matrices as its input. The first matrix is the membership matrix of elements of the universal set in the members of the covering collection. The second matrix is the membership matrix of the reasons in the causing sets of the members of the covering collection. Then it returns a string which can be parsed to a logical formula which is the SAT translation of the constraints in the binary (non)-linear optimization of the SCPR problem.":

	local 
        i          :: posint,
        j          :: posint,
        constr     :: string,
        constrs    :: {Array( string ), set( string )},
        new_value  :: string,
        new_values :: Array( string ):

	constrs := Array([]):

	for j from 1 by 1 to n do

		if add( M[i,j], i = 1..m ) = 1 then

			for i from 1 by 1 to m do
				if M[i,j] = 1 then
					ArrayTools:-Append( constrs, cat( "x__", i ) ):
				end if:
			end do:

		elif add( M[i,j], i = 1..m ) > 1 then

			constr := "Logic:-`&or`( ":

			for i from 1 by 1 to m do
				if M[i,j] = 1 then
					constr := cat( constr, "x__", i, ", " ):
				end if:
			end do:

			constr := cat( constr[1..-3], " )" ):

			ArrayTools:-Append( constrs, constr ):

		end if:

	end do:

	new_values := Array([]):

	for j from 1 by 1 to m do

		if add( N[i,j], i = 1..r ) = 1 then
			for i from 1 by 1 to r do
				if N[i,j] = 1 then
					ArrayTools:-Append( new_values, cat( "y__", i ) ):
				end if:
			end do:

		elif add( N[i,j], i = 1..r ) > 1 then

			new_value := "Logic:-`&and`( ":

			for i from 1 by 1 to r do
				if N[i,j] = 1 then
					new_value := cat( new_value, "y__", i, ", " ):
				end if:
			end do:

			new_value := cat( new_value[1..-3], " )" ):

			ArrayTools:-Append( new_values, new_value ):

		end if:

	end do:

	constrs := SubstituteAllArray( constrs, Array([ seq( cat( "x__", i ), i = 1..m ) ]), new_values ):

	constrs := convert( constrs, set ):

	return( constrs ):

end proc: