##PROCEDURE(help) SCPRNSAT
##HALFLINE This procedure solves a set covering problem with reasons (SCPR) by translating it into a SAT problem.
##AUTHOR AmirHosein Sadeghimanesh
##DATE May 2022
##CALLINGSEQUENCE
##- SCPRNSAT( 'U', 'R', 'E', 'opts' )
##PARAMETERS
##- 'U'         : a set.
##- 'R'         : a set.
##- 'E'         : a list, a set or an array of lists of two sets. The first sets must be subsets of 'U', and the second sets must be subsets of 'R'.
##OPTIONS(collapsed=false)
##- 'SATSolver' : (optional) (kwarg) a string, either '"maplesat"' or '"legacy"'. The default value is '"maplesat"'.
##RETURNS(help)
##- a set, subset of 'R'.
##DESCRIPTION
##- Consider a set covering problem with reasons, SCPR, with the universal set __"U"__, set of reasons __"R"__ and a labeled cover __"E &subset; `&Pscr;`(U) &times; `&Pscr;`(R)"__ where __"`&Pscr;`"__ stands for the power set. The goal of SCPR problem is to find a subset of __"R"__, such as __"S"__, with the minimum possible number of elements such that the union of sets in __"E"__ which their labels are subset of __"S"__, is equal to __"U"__. If the label sets all are singletons, then the problem can be translated to a linear binary optimization problem and "SCPPack:-SCPBasic" can be used. Otherwise the problem will become a nonlinear binary optimization. This problem itself can be translated to a SAT problem as described in the reference below.
## This procedure receives __"U"__, __"R"__ and __"E"__, translate the SCPR problem to a SAT problem and then uses Maple\'s "Logic:-Satisfy" command to solve the SAT problem and finally translate back the answer to a solution to the original SCPR problem.
##- By default it calls "Logic:-Satisfy" with the option 'method = "maplesat"', if the user wants to use '"legacy"' as the SAT-solver instead of '"maplesat"', then the option 'SATSolver = "legacy"' should be given to this procedure which causes calling "Logic:-Satisfy" with the option 'method = "legacy"'.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> U := { seq(i, i = 1 .. 8) }:
##> R := { seq(i, i = 1 .. 5) }:
##>> E := Array([]):
##>> ArrayTools:-Append( E, [ { seq( i, i = 1..4 ) }, {1, 2} ] ):
##>> ArrayTools:-Append( E, [ { seq( i, i = 5..8 ) }, {1, 3} ] ):
##>> ArrayTools:-Append( E, [ { seq( i, i = 2..7 ) }, {4} ] ):
##>> ArrayTools:-Append( E, [ {8}, {5} ] ):
##> SCPRNSAT( U, R, E );
##< [1, 2, 3]
##SEEALSO
##- "SCPPack:-SCPRBasic"
##- "SCPPack:-SCPSimple"
##- "SCPPack:-SCPRNLinearized"
##REFS
##- AmirHosein Sadeghimanesh, Matthew England, \"__"`An SMT solver for non-linear real arithmetic inside Maple/`"__\", 2022.

SCPRNSAT :=proc(
    U :: set,
    R :: set,
    E :: {Array( list( set ) ), list( list( set ) ), set( list( set ) )},
    n :: nonnegint := numelems( U ),
    r :: nonnegint := numelems( R ),
    m :: nonnegint := numelems( E ),
    {
		SATSolver :: { "maplesat", "legacy" } := "maplesat"
    }
    )
    :: list:

    description "This procedure takes three inputs. The first input is the universal set, U, that we want to cover it. The second input is the set of cauese (reasons), R. The third input is a set of subsets of U, each of these sets are caused by a set of reasons from R. The goal is to pick up the smallest subset of R such that the sets caused by the reasons in it, covers U.":

	local
        M                :: 'Matrix'( { 0, 1 } ),
        N                :: 'Matrix'( { 0, 1 } ),
        i                :: posint,
        j                :: posint,
        k                :: posint,
        constrs_set      :: set( string ),
        target_fun       :: string,
        SAT_problem_str  :: string,
        constr           :: string,
        ans              :: set,
        ans_point        :: list( truefalse ),
        reasons          :: list( posint ):

	M := Matrix( m, n ):
	for i from 1 by 1 to m do
		for j from 1 by 1 to n do
			if U[j] in E[i][1] then
				M[i,j] := 1:
			end if:
		end do:
	end do:

	N := Matrix( r, m ):
	for i from 1 by 1 to r do
		for j from 1 by 1 to m do
			if R[i] in E[j][2] then
				N[i,j] := 1:
			end if:
		end do:
	end do:

	constrs_set := SCPRNSAT_ConstrMaker( M, N, n, r, m ):
	
	for k from 1 by 1 to r do

		target_fun := Exactly_k_truth_formula( r, k ):

		SAT_problem_str := cat( "Logic:-`&and`( ", target_fun ):

		for constr in constrs_set do
			SAT_problem_str := cat( SAT_problem_str, ", ", constr ):
		end do:

		SAT_problem_str := cat( SAT_problem_str, " )" ):

		ans := Logic:-Satisfy( parse( SAT_problem_str  ), method = SATSolver ):

		if ans <> NULL then

			ans_point := eval( [ seq( cat(y__, i), i = 1..r ) ], ans ):
			reasons := select( i -> ans_point[i] = true, [ seq(j, j = 1..r) ]):

			return( [ seq( R[i], i in reasons ) ] ):

		end if:

	end do:

	return( [] ):

end proc: