##PROCEDURE(help) SCPRNLinearized_sub
##HALFLINE This is a subprocedure of SCPRNLinearized that generates the necessary constraints on the auxiliary variables to keep them exuivalent of the product of the main variables.
##AUTHOR AmirHosein Sadeghimanesh
##DATE June 2022
##CALLINGSEQUENCE
##- SCPRNSAT( 'L' )
##PARAMETERS
##- 'L'  : an array of sets of positive integers.
##RETURNS(help)
##- an array of linear constraints.
##DESCRIPTION
##- Note that in the linearization approach for solving the SCPR, auxiliary variables are introduced. These auxiliary variables are standing on behalf of product of main variables. For example if there is a term __"y_1 y_2 y_3"__ in a constraint of the original nonlinear binary optimization, then we need two new auxiliary variables __"u_1"__ and __"u_2"__ standing for __"y_1 y_2"__ and __"u_1 y_3"__ respectively. But to force these new binary variables to have the same value as their corresponding products, we need 3 extra constraints for each of them. These three extra constraints for __"u_1"__ are __"u_1 - y_1 <= 0"__, __"u_1 - y_2 <= 0"__ and __"y_1 + y_2 - u_1 <= 1"__. For __"u_2"__ the three constraints are written similarly.
##- The procedure "SCPPack:-SCPRNLinearized" creates an array of sets, each set is the set of indices of main variables in nonlinear terms and also their subsets that are also needed, i.e. if we have the set __"{ 1, 2, 3}"__, then we need one auxiliary variable for this and also one more for the set of indices __"{1, 2}"__. Then it passes this array to this procedure to generate the set of necessary constraints as introduced above. The result will be sent back to "SCPPack:-SCPRNLinearized". 
##- This is a subprocedure of SCPRNLinearized.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> kernelopts( opaquemodules = false ):
##> L := Array([ {1, 2}, {1, 4}, {1, 4, 5}, {1, 4, 5, 8} ]);
##< Vector[row](4, [{1, 2}, {1, 4}, {1, 4, 5}, {1, 4, 5, 8}])
##> SCPPack:-SCPRNLinearized_sub( L );
##< Vector[row](12, [u__1 - y__1 <= 0, u__1 - y__2 <= 0, 1 <= y__1 + y__2 - u__1, u__2 - y__1 <= 0, u__2 - y__4 <= 0, 1 <= y__1 + y__4 - u__2, u__3 - u__2 <= 0, u__3 - y__5 <= 0, 1 <= u__2 + y__5 - u__3, u__4 - u__3 <= 0, u__4 - y__8 <= 0, 1 <= u__3 + y__8 - u__4])
##SEEALSO
##- "SCPPack:-SCPRNLinearized"
##REFS
##- AmirHosein Sadeghimanesh, Matthew England, \"__"`An SMT solver for non-linear real arithmetic inside Maple/`"__\", 2022.

SCPRNLinearized_sub := proc(
	nonlinear_reasons :: 'Array'( set( posint ) )
	)
	:: 'Array':

	description "This procedure generates the constraints controlling the auxiliary variables for the linearization of the nonlinear binary optimization in the command SCPRNLinearized.":

	local
		output     :: 'Array',
		reason     :: set( posint ),
		sub_reason :: set( posint ),
		idx        :: posint,
		x          :: name,
		y          :: name,
		z          :: name:

	output := Array([]):

	for reason in nonlinear_reasons do

		member( reason, nonlinear_reasons, 'idx' ):
		z := cat( u__, idx ):
		sub_reason := reason[1..-2]:
		if numelems(sub_reason) = 1 then
			x := cat( y__, sub_reason[1] ):
		else
			member( sub_reason, nonlinear_reasons, 'idx' ):
			x := cat( u__, idx ):
		end if:
		y := cat( y__, reason[-1] ):

		ArrayTools:-Extend( output, [ z - x <= 0, z - y <= 0, x + y - z >= 1 ] ):

	end do:

	return( output ):

end proc: