##PROCEDURE(help) SCPRNLinearized
##HALFLINE This procedure solves a set covering problem with reasons (SCPR) by linearization of the constraints.
##AUTHOR AmirHosein Sadeghimanesh
##DATE June 2022
##CALLINGSEQUENCE
##- SCPRNSAT( 'U', 'R', 'E' )
##PARAMETERS
##- 'U'         : a set.
##- 'R'         : a set.
##- 'E'         : a list, a set or an array of lists of two sets. The first sets must be subsets of 'U', and the second sets must be subsets of 'R'.
##RETURNS(help)
##- a set, subset of 'R'.
##DESCRIPTION
##- Consider a set covering problem with reasons, SCPR, with the universal set __"U"__, set of reasons __"R"__ and a labeled cover __"E &subset; `&Pscr;`(U) &times; `&Pscr;`(R)"__ where __"`&Pscr;`"__ stands for the power set. The goal of SCPR problem is to find a subset of __"R"__, such as __"S"__, with the minimum possible number of elements such that the union of sets in __"E"__ which their labels are subset of __"S"__, is equal to __"U"__. If the label sets all are singletons, then the problem can be translated to a linear binary optimization problem and "SCPPack:-SCPBasic" must be used. Otherwise the problem will become a nonlinear binary optimization. However, one can linearize the constraints by introducing auxiliary variables and adding extra constraints.
## This procedure receives __"U"__, __"R"__ and __"E"__, linearize the SCPR problem and then uses Maple\'s "Optimization:-Minimize" command with the option 'assume = binary' to solve the linearized version and finally translates back the answer to a solution to the original SCPR problem.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> U := { seq(i, i = 1 .. 8) }:
##> R := { seq(i, i = 1 .. 5) }:
##>> E := Array([]):
##>> ArrayTools:-Append( E, [ { seq( i, i = 1..4 ) }, {1, 2} ] ):
##>> ArrayTools:-Append( E, [ { seq( i, i = 5..8 ) }, {1, 3} ] ):
##>> ArrayTools:-Append( E, [ { seq( i, i = 2..7 ) }, {4} ] ):
##>> ArrayTools:-Append( E, [ {8}, {5} ] ):
##> SCPRNLinearized( U, R, E );
##< [1, 2, 3]
##SEEALSO
##- "SCPPack:-SCPRNSAT"
##- "SCPPack:-SCPRBasic"
##- "SCPPack:-SCPSimple"
##REFS
##- AmirHosein Sadeghimanesh, Matthew England, \"__"`An SMT solver for non-linear real arithmetic inside Maple/`"__\", 2022.

SCPRNLinearized := proc(
	U :: set,
	R :: set,
	E :: {Array( list( set ) ), list( list( set ) ), set( list( set ) )},
	n :: nonnegint := numelems( U ),
	r :: nonnegint := numelems( R ),
	m :: nonnegint := numelems( E )
	)
	:: list:

	description "This procedure takes three inputs. The first input is the universal set, U, that we want to cover it. The second input is the set of cauese (reasons), R. The third input is a set of subsets of U, each of these sets are caused by a set of reasons from R. The goal is to pick up the smallest subset of R such that the sets caused by the reasons in it, covers U.":

	local
		f,
		M                          :: 'Matrix'( { 0, 1 } ),
		i                          :: posint,
		j                          :: posint,
		idx                        :: posint,
		pair                       :: satisfies( j -> j :: list, j -> numelems( j ) = 2 ),
		constrs                    :: 'Array',
		reason                     :: set( posint ),
		reasons                    :: 'Array'( set( posint ) ),
		nonlinear_reasons          :: 'Array'( set( posint ) ),
		nonlinear_reasons_extended :: 'Array'( set( posint ) ),
		substituation_rules        :: 'Array',
		sols                       :: list,
		sols_point                 :: list( { 0, 1 } ),
		S                          :: list( posint ):

	f := add( cat(y__, i), i = 1..r ): # target function of the optimization.

	M := Matrix( m, n ):
	for i from 1 by 1 to m do
		for j from 1 by 1 to n do
			if U[j] in E[i][1] then
				M[i,j] := 1:
			end if:
		end do:
	end do:

	constrs := convert( { seq( add( M[i,j] * cat(x__, i), i = 1..m ) >= 1, j = 1..n ) }, 'Array' ):

	reasons := Array([]):
	ArrayTools:-Extend( reasons, convert( { seq( pair[2], pair in E ) }, list ) ):

	nonlinear_reasons := select[flatten]( reason -> numelems( reason ) >= 2, reasons ):
	nonlinear_reasons_extended := Array([]):
	ArrayTools:-Extend( nonlinear_reasons_extended, convert( { seq( seq( reason[1..i], i = 2..numelems( reason ) ), reason in nonlinear_reasons ) }, list ) ):

	substituation_rules := Array([]):
	for i from 1 by 1 to m do
		if member( E[i][2], nonlinear_reasons_extended, 'idx' ) then
			ArrayTools:-Append( substituation_rules, cat( x__, i ) = cat( u__, idx ) ):
		else
			ArrayTools:-Append( substituation_rules, cat( x__, i ) = cat( y__, E[i][2][1] ) ):
		end if:
	end do:

	constrs := eval( constrs, convert( substituation_rules, list ) ):
	ArrayTools:-Extend( constrs, SCPRNLinearized_sub( nonlinear_reasons_extended ) ):

	userinfo( 5, SCPRNLinearized, 'NoName', printf( "The target function of the optimization is\n    %a\nAnd the linearized constraints are\n    %a\n", f, convert( constrs, list ) ) ):

	sols := Optimization:-Minimize( f, convert( constrs, list ), assume = binary ):
	sols_point := eval( [ seq( cat( y__, i ), i = 1..r ) ], sols[2] ):
	S :=  select( i -> sols_point[i] = 1, [ seq(j, j = 1..r) ]):

	return( [ seq( R[i], i in S ) ] ):

end proc:
