##PROCEDURE(help) SCPRBasic
##HALFLINE This procedure solves a set covering problem (SCP) problem. In this setting every set has a reason and we are aiming to minimize the number of reasons that together cause the universal set being covered. This is a SCPR with the label of every set being a singleton set. Therefore the problem can be translated to a linear binary optimization.
##AUTHOR AmirHosein Sadeghimanesh
##DATE May 2022
##CALLINGSEQUENCE
##- SCPRBasic( 'U', 'R', 'E' )
##PARAMETERS
##- 'U' : a set.
##- 'R' : a set.
##- 'E' : a list, a set or an array of lists of two sets, the first sets must be subsets of 'U' and the second sets must be subsets of 'R' with a single element.
##RETURNS(help)
##- a set (subet of 'R').
##DESCRIPTION
##- Consider a set covering problem with reasons, SCPR, with the universal set __"U"__, set of reasons __"R"__ and a labeled cover __"E &subset; `&Pscr;`(U) &times; `&Pscr;`(R)"__ where __"`&Pscr;`"__ stands for the power set. The goal of SCPR problem is to find a subset of __"R"__, such as __"S"__, with the minimum possible number of elements such that the union of sets in __"E"__ which their labels are subset of __"S"__, is equal to __"U"__. When the label sets are singletons, this problem can be simplified to a simple set covering problem (SCP). Let __"r={|R|}"__. One can define a new set __"E'={ A_1, ..., A_r }"__ where __"A_i"__ is the union of sets in __"E"__ with labels equal to the singleton containing the __"i"__th element of __"R"__. If __"S'"__ is the optimal answer of SCP with __"U"__ and __"E'"__, then the subset of __"R"__ with indices in __"S'"__ is the optimal solution to the original SCPR with __"U"__, __"R"__ and __"E"__.
##- This procedure solves SCPR problem when the reason labels of sets are singletons using the approach introduced above.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> U := { 1, 2, 3, 4, 5, 6, 7 }:
##> R := { a, b, c }:
##> E := { [ { 1, 2, 3, 4, 5, 6 }, { a } ], [ { 1, 2, 3, 4, 5, 6, 7 }, { b } ], [ { 1, 2, 4, 5, 6, 7 }, { c } ] }:
##> SCPRBasic( U, R, E );
##< [b]
##> R := { 1, 2, 3 }:
##> E := { [ { 1, 3, 4, 6 }, { 1 } ], [ { 1, 4, 5, 6, 7 }, { 2 } ], [ { 6, 7 }, { 3 } ], [ { 7 }, { 1 } ], [ { 2, 4, 6 }, { 2 } ] }:
##> SCPRBasic( U, R, E );
##< [1, 2]
##SEEALSO
##- "SCPPack:-SCPSimple"
##REFS
##- AmirHosein Sadeghimanesh, Matthew England, \"__"`An SMT solver for non-linear real arithmetic inside Maple/`"__\", 2022.

SCPRBasic := proc(
    U :: set,
    R :: set,
    E :: { set( list ( set) ), list( list( set ) ), Array( list( set ) ) },
    n :: nonnegint := numelems(U),
    r :: nonnegint := numelems(R)
    )
    :: list:

    description "This procedure takes three inputs. The first input is the universal set, U, that we want to cover it. The second input is the set of cauese (reasons), R. The third input is a set of subsets of U, each of these sets are caused by a reason from R. The goal is to pick up the smallest subset of R such that the sets caused by the reasons in it, covers U.":

	local
        i  :: posint,
        A  :: list,
        EE :: Array( set ),
        SS :: list( posint ):

	EE := Array([]):
	ArrayTools:-Extend( EE, [ seq( `union`( seq( A[1], A in select[flatten]( A -> A[2] = { R[i] }, E ) ) ), i = 1..r ) ] ):

    SS := SCPSimple( U, EE, n, r ):

	return( [ seq( R[i], i in SS ) ] ):

end proc: