##PROCEDURE(help) SCPR
##HALFLINE This procedure solves a set covering problem with reasons (SCPR).
##AUTHOR AmirHosein Sadeghimanesh
##DATE May 2022
##CALLINGSEQUENCE
##- SCPR( 'U', 'R', 'E', 'opts' )
##PARAMETERS
##- 'U'         : a set.
##- 'R'         : a set.
##- 'E'         : a list, a set or an array of lists of two sets. The first sets must be subsets of 'U', and the second sets must be subsets of 'R'.
##OPTIONS(collapsed=false)
##- 'approach'  : (optional) (kwarg) a string, '"SAT"', '"linearization"', '"SCP_or_SAT"' and '"SCP_or_linearization"'. The default value is '"SCP_or_SAT"'. By choosing '"SAT"' or '"Linearization"' the SCPR\'s nonlinear binary optimization gets translated to a SAT problem or gets linearized and becomes a linear binary optimization respectively. Whereas by choosing '"SCP_or_SAT"' or '"SCP_or_linearization"', Maple will fisr check if the SCPR can be written as a simple SCP, if so it solves it as a anormal SCP, otherwise it uses the other approach to handle the nonlinearity. In summary this argument determines which of the three functions "SCPPack:-SCPRBasic", "SCPPack:-SCPRNSAT" or "SCPPack:-SCPRNLinearized" must be called.
##- 'SATSolver' : (optional) (kwarg) a string, either '"maplesat"' or '"legacy"'. The default value is '"maplesat"'. This argument will be used only if the approach chosen to solve the SCPR needs to use a SAT-solver. For example if the chosen approach is __"`linearization/`"__ then this argument is not relevant.
##RETURNS(help)
##- a set, subset of 'R'.
##DESCRIPTION
##- Consider a set covering problem with reasons, SCPR, with the universal set __"U"__, set of reasons __"R"__ and a labeled cover __"E &subset; `&Pscr;`(U) &times; `&Pscr;`(R)"__ where __"`&Pscr;`"__ stands for the power set. The goal of SCPR problem is to find a subset of __"R"__, such as __"S"__, with the minimum possible number of elements such that the union of sets in __"E"__ which their labels are subset of __"S"__, is equal to __"U"__. If the label sets all are singletons, then the problem can be translated to a linear binary optimization problem and "SCPPack:-SCPBasic" must be used. Otherwise the problem will become a nonlinear binary optimization. This problem itself can be translated to a SAT problem as described in the reference below.
## This procedure receives __"U"__, __"R"__ and __"E"__, translate the SCPR problem to a SAT problem and then uses Maple\'s "Logic:-Satisfy" command to solve the SAT problem and finally translate back the answer to a solution to the original SCPR problem.
##- By default it calls "Logic:-Satisfy" with the option 'method = "maplesat"', if the user wants to use '"legacy"' as the SAT-solver instead of '"maplesat"', then the option 'SATSolver = "legacy"' should be given to this procedure which causes calling "Logic:-Satisfy" with the option 'method = "legacy"'.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> U := { seq(i, i = 1 .. 8) }:
##> R := { seq(i, i = 1 .. 5) }:
##>> E := Array([]):
##>> ArrayTools:-Append( E, [ { seq( i, i = 1..4 ) }, {1, 2} ] ):
##>> ArrayTools:-Append( E, [ { seq( i, i = 5..8 ) }, {1, 3} ] ):
##>> ArrayTools:-Append( E, [ { seq( i, i = 2..7 ) }, {4} ] ):
##>> ArrayTools:-Append( E, [ {8}, {5} ] ):
##> SCPR( U, R, E );
##< [1, 2, 3]
##> SCPR( U, R, E, approach = "SAT" );
##< [1, 2, 3]
##> SCPR( U, R, E, approach = "linearization" );
##< [1, 2, 3]
##> SCPR( U, R, E, appraoch = "SCP_or_SAT" );
##< [1, 2, 3]
##> SCPR( U, R, E, appraoch = "SCP_or_linearization" );
##< [1, 2, 3]
##SEEALSO
##- "SCPPack:-SCPRBasic"
##- "SCPPack:-SCPSimple"
##- "SCPPack:-SCPRNSAT"
##- "SCPPack:-SCPRNLinearized"
##REFS
##- AmirHosein Sadeghimanesh, Matthew England, \"__"`An SMT solver for non-linear real arithmetic inside Maple/`"__\", 2022.

SCPR := proc(
	U :: set,
	R :: set,
	E :: {Array( list( set ) ), list( list( set ) ), set( list( set ) )},
	n :: nonnegint := numelems( U ),
	r :: nonnegint := numelems( R ),
	m :: nonnegint := numelems( E ),
	{
		approach  :: { "SAT", "linearization", "SCP_or_SAT", "SCP_or_linearization" } := "SCP_or_SAT",
		SATSolver :: { "maplesat", "legacy" } := "maplesat"
	}
	)
	:: list:

	description "This procedure takes three inputs. The first input is the universal set, U, that we want to cover it. The second input is the set of cauese (reasons), R. The third input is a set of subsets of U, each of these sets are caused by a set of reasons from R. The goal is to pick up the smallest subset of R such that the sets caused by the reasons in it, covers U.":

	local
		pair :: satisfies( j -> j :: list, j -> numelems( j ) = 2 ):

	if approach = "SAT" then
		return( SCPRNSAT( U, R, E, n, r, m, SATSolver ) ):
	elif approach = "linearization" then
		return( SCPRNLinearized( U, R, E, n, r, m ) ):
	else
		if max( seq( numelems( pair[2] ), pair in E ) ) = 1 then
			return( SCPRBasic( U, E, R, n, r ) ):
		elif approach = "SCP_or_SAT" then
			return( SCPRNSAT( U, R, E, n, r, m, SATSolver ) ):
		else
			return( SCPRNLinearized( U, R, E, n, r, m ) ):
		end if:
	end if:

end proc: