##PROCEDURE(help) LegacyFixer
##HALFLINE This procedure fixes the bug when "Logic:-Satisfy" with 'method = legacy' is used. The body of this function is given by \"__"`Stephen Forrest/`"__\" from Maple. We just put it inside a procedure.
##AUTHOR AmirHosein Sadeghimanesh
##DATE May 2022
##CALLINGSEQUENCE
##- LegacyFixer( )
##DESCRIPTION
##- When "Logic:-Satisfy" is called with the optional kwarg 'metod' set to '"legacy"', it may show a wrong answer such as 'false' for all boolean variables (at least in Maple 2022.0). To avoid this issue just run this procedure before using "Logic:-Satisfy" in your Maple session.
##- When loading "SCPPack" module, it automatically checks the first example below and if the output is every variable being 'false', then it execute this procedure. Therefore most likely you do not need to run this command yourself.
##- The code for this procedure is given by \"__"`Stephen Forrest/`"__\".
##EXAMPLES
##- First let us see what this bug really is before loading the "SCPPack". Let __"X"__ and __"Y"__ be two boolean variables. Define __"f = ( X &and; &not; Y ) &or; ( &not; X &and; Y )"__. Obviously this has two solutions; __"(X, Y)=(`true|`, `false|`)"__ and __"(X, Y)=(`false|`, `true|`)"__. Now let us ask "Logic:-Satisfy" command to check it.
##> f := Logic:-`&or`( Logic:-`&and`( X, Logic:-`&not`( Y ) ), Logic:-`&and`( Logic:-`&not`( X ), Y ) );
##< f:=(X Logic:-`&and` (Logic:-`&not` Y)) Logic:-`&or` ((Logic:-`&not` X) Logic:-`&and` Y)
##> Logic:-Export( f, form = boolean );
##< X and not Y or not X and Y
##> Logic:-Satisfy( f );
##< {X = false, Y = true}
##- This is a correct answer. "Logic:-Satisfy" by default uses '"maplesat"'. The user can explicitely ask which SAT-solver to be used by the optional kwarg, 'method'.
##> Logic:-Satisfy( f, method = "maplesat" );
##< {X = false, Y = true}
##- However, when setting '"method"' to '"legacy"' it gives a wrong answer.
##> Logic:-Satisfy( f, method = "legacy" );
##< {X = false, Y = false}
##- Now run by running the fixer, this issue will be solved. As mentioned earlier the user does not need to do it manually and by loaing the package, the fixer will get executed automatically if needed.
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##- Let\'s see if the bug is still here.
##> Logic:-Satisfy( f, method = "legacy" );
##< {X = false, Y = true}
##- If you have doubt that the fixer is not really calling '"legacy"', try the following example where the two SAT-solvers return different solutions.
##> Logic:-Satisfy( (X or Y) and (Z or W), method = "maplesat" );
##< {W = false, X = true, Y = false, Z = true}
##> Logic:-Satisfy( (X or Y) and (Z or W), method = "legacy" );
##< {W = true, X = true, Y = false, Z = false}

LegacyFixer := proc()

    description "Fixing the legacy-logic bug for the current Maple session.":

    kernelopts( opaquemodules = false ):
    unprotect( Logic:-Satisfy ):
    Logic:-Satisfy := subs( `satisfy/sat` = Logic:-SAT:-`satisfy/sat`, eval( Logic:-Satisfy ) ):
    kernelopts( opaquemodules = true ):
    protect( Logic:-Satisfy ):

end proc: