##PROCEDURE(help) Exactly_k_truth_formula
##HALFLINE This procedure receives two natural numbers __"r"__ and __"k"__, where __"k <= r"__, and returns a string that can be parsed to a logic formula equivalent of asking exactly __"k"__ of __"r"__ boolean variables being true.
##AUTHOR AmirHosein Sadeghimanesh
##DATE May 2022
##CALLINGSEQUENCE
##- Exactly_k_truth_formula( 'r', 'k' )
##PARAMETERS
##- 'r' : a positive integer.
##- 'k' : a positive integer not greater than 'r'.
##RETURNS(help)
##- a string.
##DESCRIPTION
##- Let __"y_1, ..., y_r"__ be __"r"__ boolean variables and __"k"__ a natural number __"k <= r"__. We want a logical formula equivalent with exactly __"k"__ of these __"r"__ variables can be true at the same time. This formula is a disjunction of all conjunctions made by "SCPPack:-ConjunctionMaker" for all binary vectors of length __"r"__ with __"k"__ ones.
##- This procedure receives two natural numbers __"r"__ and __"k"__ that __"k <= r"__ and returns a string that can be parsed to a logical formula with logical operations from Maple\'s "Logic" package that is equivalent with exactly __"k"__ of __"r"__ boolean variables can be 'true'.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> kernelopts( opaquemodules = false ):
##> SCPPack:-Exactly_k_truth_formula( 3, 1 );
##< "Logic:-`&or`( Logic:-`&and`( y__1, Logic:-`&not`( y__2 ), Logic:-`&not`( y__3 )), Logic:-`&and`( Logic:-`&not`( y__1 ), y__2, Logic:-`&not`( y__3 )) , Logic:-`&and`( Logic:-`&not`( y__1 ), Logic:-`&not`( y__2 ), y__3) )"
##> f := SCPPack:-Exactly_k_truth_formula( 3, 2 );
##< f := "Logic:-`&or`( Logic:-`&and`( y__1, y__2, Logic:-`&not`( y__3 )), Logic:-`&and`( y__1, Logic:-`&not`( y__2 ), y__3) , Logic:-`&and`( Logic:-`&not`( y__1 ), y__2, y__3) )"
##> Logic:-Satisfy( parse( f ) );
##< {y__1 = false, y__2 = true, y__3 = true}
##SEEALSO
##- "SCPPack:-ConjunctionMaker"
##REFS
##- AmirHosein Sadeghimanesh, Matthew England, \"__"`An SMT solver for non-linear real arithmetic inside Maple/`"__\", 2022.

Exactly_k_truth_formula := proc(
    r :: posint, 
    k :: satisfies( j -> j :: posint, j -> j <= r )
    )
    :: string:

    description "It receives two integers, r and k, then returns a string that can be parsed to a logical formula equivalent with exactly k of r boolean variables being true.":

	local
        formula :: string,
        v       :: {list( {0, 1} ), Array( {0, 1} )},
        i       :: posint:

	formula := "Logic:-`&or`( ":

	v := Array([ seq(1, i = 1..k), seq(0, i = k+1..r) ]):
	
    formula := cat( formula, ConjunctionMaker( v ) ):
	
    for i from 1 by 1 to binomial(r, k)-1 do
		formula := cat( formula, ", " ):
		v := Next_k_choice_r( v ):
		formula := cat( formula, ConjunctionMaker( v ), " " ):
	end do:
	
    formula := cat( formula, ")" ):
	
    return( formula ):

end proc:
