##PROCEDURE(help) Next_k_choice_r
##HALFLINE This procedure recieves a binary vector of length r with k ones and returns the next binary vector of the same property when these vectors are ordered lexicographically with the most left side position being greater. It moves in the descending direction.
##AUTHOR AmirHosein Sadeghimanesh
##DATE May 2022
##CALLINGSEQUENCE
##- Next_k_choice_r( 'vec' )
##PARAMETERS
##- 'vec' : a binary list or array.
##RETURNS(help)
##- a binary array of the same length and the same number of ones as the input\'s.
##DESCRIPTION
##- Consider the set of binary vectors of length __"r"__ with __"k"__ ones. For two such vectors __"u=(u_1,...,u_r)"__ and __"v=(v_1,...,v_r)"__ that are not equal, we let __"u<v"__ if and only if for the first index __"i"__ that __"u_i<>v_i"__ we have __"u_i<v_i"__.
##- This procedure receives a binary vector and then returns the next binary vector of the same length and the same number of ones, in the above mentioned order in the descending direction. If there is no such next vector, i.e. the input vector, __"v=(v_1,...,v_r)"__ satisfies __"v_i=&lbrace;(0; i=1,...,r-k)//(1; i=r-k+1,...,r)"__, then it raises an error message stating there is no next vector.
##- Any vector of length __"r"__ with __"k"__ ones can be seen as selection of __"k"__ objects from __"r"__ ordered objects in this way that the __"i"__th object is selected if and only if the __"i"__th entry of this vector is __"1"__.
##EXAMPLES
##> libname ,= "C:\\Home\\DEWCADCoventry GitHub\\Packages\\SCPPack":
##> with( SCPPack ):
##> kernelopts( opaquemodules = false ):
##> SCPPack:-Next_k_choice_r( [1, 0] );
##< Vector[row](2, [0, 1])
##> M := Matrix( binomial( 5, 3 ), 5 ):
##>> v := Array([ 1, 1, 1, 0, 0 ]):
##>> M[ 1, .. ] := v:
##>> for i from 2 by 1 to binomial(5, 3) do
##>>  v := SCPPack:-Next_k_choice_r( v ):
##>>  M[ i, .. ] := v:
##>> end do:
##>> M;
##< Matrix(10, 5, [[1, 1, 1, 0, 0], [1, 1, 0, 1, 0], [1, 1, 0, 0, 1], [1, 0, 1, 1, 0], [1, 0, 1, 0, 1], [1, 0, 0, 1, 1], [0, 1, 1, 1, 0], [0, 1, 1, 0, 1], [0, 1, 0, 1, 1], [0, 0, 1, 1, 1]])
##SEEALSO
##- "binomial"

Next_k_choice_r := proc(
    vec :: {list( {0, 1} ), Array( {0, 1} )},
    k   :: posint := add(j, j in vec),
    r   :: posint := numelems(vec)
    )
    :: Array( {0, 1} ):

    description "This procedure recieves a binary vector representing a choice of k elements from an ordered set of r elements. Then returns another vector with a similar property. The new vector is the vector after the input vector if we order these vectors lexicographically and in a descending direction.":

	local
        i               :: posint,
        idx_1           :: nonnegint,
        idx_2           :: posint,
        last_ones_count :: posint,
        new_vec         :: Array( {0, 1} ):

	new_vec := convert( vec, Array ):

	idx_1 := 0:

	for idx_2 from r by -1 to 1 do

		if vec[ idx_2 ] <> 0 then
			idx_1 := idx_2:
			break:
		end if:

	end do:

	if idx_1 = 0 then
		error "The input vector does not have any 1 in it.":
	end if:

	if idx_1 <> r then
		new_vec[ idx_1 ] := 0:
		new_vec[ idx_1 + 1 ] := 1:
		return new_vec:
	end if:
	# so idx_1 = r

	for idx_1 from r-1 by -1 to 1 do

		if vec[ idx_1 ] = 0 then

			last_ones_count := r - idx_1: # number of 1's in last sequal 1's

			for idx_2 from idx_1 - 1 by -1 to 1 do

				if vec[ idx_2 ] <> 0 then

					new_vec( idx_2 ) := 0:

					for i from 1 by 1 to last_ones_count + 1 do
						new_vec[ idx_2 + i ] := 1:
					end do:

					for i from max( idx_2 + last_ones_count + 2, r - last_ones_count + 1 ) by 1 to r do
						new_vec[ i ] := 0:
					end do:

					return new_vec:

				end if:

			end do:
			# so vec[ idx_2 ] did not hit any 1

			error "There is no next k-choice_r vector.":

		end if:

	end do:
	# the vector is all 1.

	error "There is no next k-choice-r vector.":

end proc:
		
					
		
	