#####
#
# "EU climate change mitigation targets compromise forest ecosystem services and biodiversity"
# Supporting figures for supplementary info
# 2022-03-15
#
#####


# FES effects
# - Effect of top-down on FESB by country
# - Effects on FESB indicators by country 

# Represented countries:
# - Finland
# - Germany
# - Norway
# - Sweden



### load libraries
library(dplyr)
library(ggplot2)
library(tidyr)
library(ggpubr)





########################################################################################
# Effects on Ecosystem services level
########################################################################################

# -----------
# normalized FES indicators - difference between top-down and bottom up
# -----------

df.es <- df.es_effect_all %>% 
  # Group the effects by ES category and calculate the mean of the differences
  group_by(country, policy, scenario, globiom,es) %>% 
  summarise(mean_diff = mean(diff)) %>% 
  mutate(es = factor(es, levels = c("Wood","Bioenergy","Nonwood","Game","Biodiversity","Water","Climate","Recreation", "Resilience"))) %>% 
  ungroup()


# -----------
# Plot the heat map
# -----------

# explore value range for plotting
maxValue <- max(df.es$mean_diff)
minValue <- min(df.es$mean_diff)
range <- round((if(-minValue > maxValue){-minValue} else {maxValue})+ 0.04, 1 ) #get next higher decimal (0.04)
range

# plot Function for the difference among top-down and bottom-up
plot.tableFkt <- function(data, COUNTRY){
  
  if (COUNTRY == "Finland"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Germany"){
    TITLE <- "Germany (Bavaria)"
  } else if (COUNTRY == "Norway"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Sweden"){
    TITLE <- COUNTRY
  }
  
  
  # adapt policy scenario name for plotting
  data <- data %>% 
    mutate(policy_plot = ifelse(policy %in% "BAU", "NFS",
                                ifelse(policy %in% "Biodiv", "BDS", "BES")))
  # order factor level
  data$policy_plot <- factor(data$policy_plot, levels = c("NFS", "BDS", "BES"))
  
  plot <- data %>% 
    
    # country to plot
    filter(country %in% COUNTRY) %>% 
    
    ggplot(aes(x = globiom, y = es)) +
    geom_tile(aes(fill = mean_diff)) +
    scale_x_discrete(expand = c(0,0))+
    scale_y_discrete(expand = c(0,0), limits = rev(levels(data$es))) +
    #scale_fill_gradient("Impact", low = "lightblue", high = "blue") +
    scale_fill_gradientn("Impact (loss & gain)\n",
                         #colours=c("red","white", "blue"),
                         colours = c("#E46726", "grey99",  "#6D9EC1"),
                         na.value = "grey98",
                         limits = c(-range,range), # adjust limits to set breaks
                         breaks = c(-range,-range/2, 0, range/2,range ),
                         oob=squish
    ) +
    
    
    theme_minimal() +
    theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust=1),
          axis.title.x = element_blank(),
          axis.title.y = element_blank(),
          legend.position="bottom",
          legend.key.width=unit(1.5,"cm"),
          plot.title = element_text(hjust = 0.5)) +
    
    #ylab("Ecosystem service") +
    #labs(title = "Impact on level of forest ecosystem services") +
    facet_grid(. ~ policy_plot)
  plot
  
  return(plot)
  
}


# call the above plot function
plot.fin_es_effect <- plot.tableFkt(df.es, "Finland")
plot.nor_es_effect <- plot.tableFkt(df.es, "Norway")
plot.swe_es_effect <- plot.tableFkt(df.es, "Sweden")
plot.ger_es_effect <- plot.tableFkt(df.es, "Germany")







########################################################################################
# Effects on indicator level
########################################################################################


# ----------
# Adjust names of indicators for plot
# ----------

# ----------
# Finland
# ----------

# 1 N_where_D_gt_40         
# 2 prc_V_deciduous         
# 3 V_deadwood              
# 4 Biomass                 
# 5 CARBON_STORAGE          
# 6 CAPERCAILLIE            
# 7 HAZEL_GROUSE            
# 8 HSI_MOOSE               
# 9 MUSHROOMS               
# 10 BILBERRY                
# 11 COWBERRY                
# 12 Recreation              
# 13 Scenic                  
# 14 Harvested_V             
# 15 i_Vm3                   
# 16 Ratio_CCF_forests       
# 17 Ratio_SA_forests        
# 18 Ratio_Broadleave_forests
# 19 Ratio_CCF_onPeat

df.fin <- df.es_effect_all %>% filter(country %in% "Finland") %>% mutate(ind_plot = as.character(NA) )

df.fin[df.fin$indicator %in% "N_where_D_gt_40",]$ind_plot <- "LargeTrees_BC"
df.fin[df.fin$indicator %in% "prc_V_deciduous",]$ind_plot <- "DeciduousTrees_BC"
df.fin[df.fin$indicator %in% "V_deadwood",]$ind_plot <- "DeadwoodVolume_BC"
df.fin[df.fin$indicator %in% "Biomass",]$ind_plot <- "HarvestedResidues_BE"
df.fin[df.fin$indicator %in% "CARBON_STORAGE",]$ind_plot <- "CarbonStorage_CR"
df.fin[df.fin$indicator %in% "CAPERCAILLIE",]$ind_plot <- "HsiCapercaillie_GA"
df.fin[df.fin$indicator %in% "HAZEL_GROUSE",]$ind_plot <- "HsiHazelGrouse_GA"
df.fin[df.fin$indicator %in% "HSI_MOOSE",]$ind_plot <- "HsiMoose_GA"
df.fin[df.fin$indicator %in% "ALL_MARKETED_MUSHROOMS",]$ind_plot <- "Mushrooms_NW" 
df.fin[df.fin$indicator %in% "BILBERRY",]$ind_plot <- "Bilberry_NW"
df.fin[df.fin$indicator %in% "COWBERRY",]$ind_plot <- "Cowberry_NW"
df.fin[df.fin$indicator %in% "Recreation",]$ind_plot <- "RecreationIndex_REC"
df.fin[df.fin$indicator %in% "Scenic",]$ind_plot <- "ScenicIndex_REC"
df.fin[df.fin$indicator %in% "Harvested_V",]$ind_plot <- "HarvestedRoundwood_WP"
df.fin[df.fin$indicator %in% "i_Vm3",]$ind_plot <- "Increment_WP"
df.fin[df.fin$indicator %in% "Ratio_CCF_forests",]$ind_plot <- "ConservationRegimes_BC"
df.fin[df.fin$indicator %in% "Ratio_SA_forests",]$ind_plot <- "SetAside_BC"
df.fin[df.fin$indicator %in% "Ratio_Broadleave_forests",]$ind_plot <- "AdaptionToCC_RES"
df.fin[df.fin$indicator %in% "Ratio_CCF_onPeat",]$ind_plot <- "CCFonPeatland_WA"

df.fin$ind_plot <- factor(df.fin$ind_plot, levels = rev(c("Increment_WP", "HarvestedRoundwood_WP", "HarvestedResidues_BE",
                                                          "Bilberry_NW","Cowberry_NW", "Mushrooms_NW",
                                                          "HsiMoose_GA", "HsiCapercaillie_GA", "HsiHazelGrouse_GA",
                                                          "SetAside_BC", "ConservationRegimes_BC", "DeadwoodVolume_BC", "DeciduousTrees_BC", "LargeTrees_BC",
                                                          "CCFonPeatland_WA",
                                                          "CarbonStorage_CR",
                                                          "RecreationIndex_REC", "ScenicIndex_REC",
                                                          "AdaptionToCC_RES")) )



# ----------
# Germany
# ----------

# 1 MeanBiodiv           
# 2 Ratio_SA_forests     
# 3 Shannon              
# 4 spi                  
# 5 energy_prods         
# 6 living_carbon        
# 7 Recreation_Aesthetics
# 8 pnv_dist             
# 9 Storms_Risk          
# 10 crown_cover          
# 11 TotStdVolume         
# 12 HarvestedVolume      
# 13 Increment_Year       
# 14 TotVolPulpWood       
# 15 TotVolSawLogs    

df.ger <- df.es_effect_all %>% filter(country %in% "Germany") %>% mutate(ind_plot = as.character(NA) )

df.ger[df.ger$indicator %in% "MeanBiodiv",]$ind_plot <- "BiodiversityInd_BC"
df.ger[df.ger$indicator %in% "Ratio_SA_forests",]$ind_plot <- "SetAside_BC"
df.ger[df.ger$indicator %in% "Shannon",]$ind_plot <- "ShannonIndex_BC"
df.ger[df.ger$indicator %in% "spi",]$ind_plot <- "SpeciesProfileIndex_BC"
df.ger[df.ger$indicator %in% "energy_prods",]$ind_plot <- "EnergyProducts_BE"
df.ger[df.ger$indicator %in% "living_carbon",]$ind_plot <- "LivingCarbon_CR"
df.ger[df.ger$indicator %in% "Recreation_Aesthetics",]$ind_plot <- "RecreationAesthetics_REC"
df.ger[df.ger$indicator %in% "pnv_dist",]$ind_plot <- "PotNaturalVegetation_RES"
df.ger[df.ger$indicator %in% "Storms_Risk",]$ind_plot <- "Storm&BeetleRisk_RES"
df.ger[df.ger$indicator %in% "crown_cover",]$ind_plot <- "CrownCoverage_WA"
df.ger[df.ger$indicator %in% "TotStdVolume",]$ind_plot <- "StandingVolume_WA"
df.ger[df.ger$indicator %in% "HarvestedVolume",]$ind_plot <- "HarvestedVolume_WP"
df.ger[df.ger$indicator %in% "Increment_Year",]$ind_plot <- "Increment_WP"
df.ger[df.ger$indicator %in% "TotVolPulpWood",]$ind_plot <- "HarvestedPulpwood_WP"
df.ger[df.ger$indicator %in% "TotVolSawLogs",]$ind_plot <- "HarvestedSawlogs_WP"

df.ger$ind_plot <- factor(df.ger$ind_plot, levels = rev(c("Increment_WP", "HarvestedVolume_WP", "HarvestedSawlogs_WP", "HarvestedPulpwood_WP", 
                                                          "EnergyProducts_BE",
                                                          "BiodiversityInd_BC", "ShannonIndex_BC", "SpeciesProfileIndex_BC", "SetAside_BC",
                                                          "CrownCoverage_WA", "StandingVolume_WA",
                                                          "LivingCarbon_CR",
                                                          "RecreationAesthetics_REC",
                                                          "Storm&BeetleRisk_RES",
                                                          "PotNaturalVegetation_RES")))


# ----------
# Norway
# ----------

# 1 bilberry_ha              
# 2 dead_vol_m3_ha_year      
# 3 MIS_ha                   
# 4 GROT_t_ha_year           
# 5 CO2_forest_t_ha_year     
# 6 harv_cityplots_m3_ha_year
# 7 Shan_ha_year             
# 8 harv_versk_m3_ha_year    
# 9 harv_net_nok_ha_year     
# 10 harv_vol_m3_ha_year  

df.nor <- df.es_effect_all %>% filter(country %in% "Norway") %>% mutate(ind_plot = as.character(NA) )

df.nor[df.nor$indicator %in% "bilberry_ha",]$ind_plot <- "BilberryCover_BC"
df.nor[df.nor$indicator %in% "dead_vol_m3_ha_year",]$ind_plot <- "DeadwoodVolume_BC"
df.nor[df.nor$indicator %in% "MIS_ha",]$ind_plot <- "MisArea_BC"
df.nor[df.nor$indicator %in% "GROT_t_ha_year",]$ind_plot <- "HarvestedResidues_BE"
df.nor[df.nor$indicator %in% "CO2_forest_t_ha_year",]$ind_plot <- "CO2inForest_CR"
df.nor[df.nor$indicator %in% "harv_cityplots_m3_ha_year",]$ind_plot <- "HarvestInCityForest_REC"
df.nor[df.nor$indicator %in% "Shan_ha_year",]$ind_plot <- "ShannonIndex_REC"
df.nor[df.nor$indicator %in% "harv_versk_m3_ha_year",]$ind_plot <- "HarvestInProtectedAreas_WA"
df.nor[df.nor$indicator %in% "harv_net_nok_ha_year",]$ind_plot <- "HarvestNetValue_WP"
df.nor[df.nor$indicator %in% "harv_vol_m3_ha_year",]$ind_plot <- "HarvestedVolume_WP"

df.nor$ind_plot <- factor(df.nor$ind_plot, levels = rev(c("HarvestNetValue_WP", "HarvestedVolume_WP", 
                                                          "HarvestedResidues_BE",
                                                          "MisArea_BC", "DeadwoodVolume_BC", "BilberryCover_BC",
                                                          "HarvestInProtectedAreas_WA", 
                                                          "CO2inForest_CR",
                                                          "HarvestInCityForest_REC", "ShannonIndex_REC")) )

# ----------
# Sweden
# ----------

# 1 DeadWoodVolume         
# 2 old_deciduous          
# 3 set_aside              
# 4 SimulatedResidue       
# 5 TotalCarbon            
# 6 RecreationIndex        
# 7 VolumeDecidous         
# 8 CCF                    
# 9 AnnualIncrementNetTotal
# 10 NPV                    
# 11 yearly_harvest  

df.swe <- df.es_effect_all %>% filter(country %in% "Sweden") %>% mutate(ind_plot = as.character(NA) )

df.swe[df.swe$indicator %in% "DeadWoodVolume",]$ind_plot <- "DeadwoodVolume_BC"
df.swe[df.swe$indicator %in% "old_deciduous",]$ind_plot <- "OldDeciduousForest_BC"
df.swe[df.swe$indicator %in% "set_aside",]$ind_plot <- "SetAside_BC"
df.swe[df.swe$indicator %in% "SimulatedResidue",]$ind_plot <- "HarvestedResidues_BE"
df.swe[df.swe$indicator %in% "TotalCarbon",]$ind_plot <- "CarbonStorage_CR"
df.swe[df.swe$indicator %in% "RecreationIndex",]$ind_plot <- "RecreationIndex_REC"
df.swe[df.swe$indicator %in% "VolumeDecidous",]$ind_plot <- "DeciduousVolume_RES"
df.swe[df.swe$indicator %in% "CCF",]$ind_plot <- "ShareOfCCF_WA"
df.swe[df.swe$indicator %in% "AnnualIncrementNetTotal",]$ind_plot <- "Increment_WP"
df.swe[df.swe$indicator %in% "NPV",]$ind_plot <- "NetPresentValue_WP"
df.swe[df.swe$indicator %in% "yearly_harvest",]$ind_plot <- "HarvestedVolume_WP"

df.swe$ind_plot <- factor(df.swe$ind_plot, levels = rev(c("NetPresentValue_WP",  "Increment_WP", "HarvestedVolume_WP",
                                                          "HarvestedResidues_BE", 
                                                          "SetAside_BC", "DeadwoodVolume_BC", "OldDeciduousForest_BC", 
                                                          "CarbonStorage_CR",
                                                          "RecreationIndex_REC",
                                                          "ShareOfCCF_WA",
                                                          "DeciduousVolume_RES")) )




# -----------
# Illustrate the individual values for indicators - SUPPLEMENTARY material
# -----------

plotFctIndicator <- function(data, COUNTRY){
  
  if (COUNTRY == "Finland"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Germany"){
    TITLE <- "Germany (Bavaria)"
  } else if (COUNTRY == "Norway"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Sweden"){
    TITLE <- COUNTRY
  }
  
  plot <- data %>% 
    
    mutate(policy_plot = ifelse(policy %in% "BAU", "NFS",
                                ifelse(policy %in% "Biodiv", "BDS", "BES")) ) %>% 
    mutate(policy_plot = factor(policy_plot, levels = c("NFS", "BDS", "BES"))) %>% 
    
    # country to plot
    #filter(country %in% COUNTRY) %>% 
    
    ggplot(aes(diff, ind_plot, fill = scenario)) +
    geom_col(position="dodge") +
    theme_minimal() +
    theme(legend.position = "bottom",
          axis.title.y = element_blank(),
          plot.title = element_text(hjust = 0.5)
          )+
    facet_grid(. ~ policy_plot) +
    scale_fill_brewer("Mitigation target", palette="Set2", label = c("1p5 (RCP1.9)", "NDC (RCP4.5)")) +
    xlab("Impact (loss & gain)") #+
    #ylab("Indicator") #+
    #labs(title = "Impact on level of indicators") 
  plot
  
  return(plot)
  
}


plot.indFin <- plotFctIndicator(df.fin, "Finland")
plot.indGer <- plotFctIndicator(df.ger, "Germany")
plot.indNor <- plotFctIndicator(df.nor, "Norway")
plot.indSwe <- plotFctIndicator(df.swe, "Sweden")



# -----------
# Annotate Figure
# -----------

## FIN
plot.indFin_a <- annotate_figure(plot.indFin, top = text_grob("Impact on level of indicators",  size = 14), fig.lab = "b)", fig.lab.face = "bold") # face = "bold"
plot.fin_es_effect_a <- annotate_figure(plot.fin_es_effect, top = text_grob("Impact on forest ecosystem services and biodiversity",  size = 14), fig.lab = "a)", fig.lab.face = "bold") # face = "bold"

plot.comb_fin <- ggpubr::ggarrange( plot.fin_es_effect_a, plot.indFin_a, ncol = 1, nrow= 2)
ggsave(plot = plot.comb_fin, paste0(outp_path,"SI_ind_fes_fin.jpg"), width=6, height=9)


## GER
plot.indGer_a <- annotate_figure(plot.indGer, top = text_grob("Impact on level of indicators",  size = 14), fig.lab = "b)", fig.lab.face = "bold") # face = "bold"
plot.ger_es_effect_a <- annotate_figure(plot.ger_es_effect, top = text_grob("Impact on forest ecosystem services and biodiversity",  size = 14), fig.lab = "a)", fig.lab.face = "bold") # face = "bold"

plot.comb_ger <- ggpubr::ggarrange( plot.ger_es_effect_a, plot.indGer_a, ncol = 1, nrow= 2)
ggsave(plot = plot.comb_ger, paste0(outp_path,"SI_ind_fes_ger.jpg"), width=6, height=9)


## NOR
plot.indNor_a <- annotate_figure(plot.indNor, top = text_grob("Impact on level of indicators",  size = 14), fig.lab = "b)", fig.lab.face = "bold") # face = "bold"
plot.nor_es_effect_a <- annotate_figure(plot.nor_es_effect, top = text_grob("Impact on forest ecosystem services and biodiversity",  size = 14), fig.lab = "a)", fig.lab.face = "bold") # face = "bold"

plot.comb_nor <- ggpubr::ggarrange(plot.nor_es_effect_a, plot.indNor_a, ncol = 1, nrow= 2)
ggsave(plot = plot.comb_nor, paste0(outp_path,"SI_ind_fes_nor.jpg"), width=6, height=9)


## SWE
plot.indSwe_a <- annotate_figure(plot.indSwe, top = text_grob("Impact on level of indicators",  size = 14), fig.lab = "b)", fig.lab.face = "bold") # face = "bold"
plot.swe_es_effect_a <- annotate_figure(plot.swe_es_effect, top = text_grob("Impact on forest ecosystem services and biodiversity",  size = 14), fig.lab = "a)", fig.lab.face = "bold") # face = "bold"

plot.comb_swe <- ggpubr::ggarrange(plot.swe_es_effect_a, plot.indSwe_a, ncol = 1, nrow= 2)
ggsave(plot = plot.comb_swe, paste0(outp_path,"SI_ind_fes_swe.jpg"), width=6, height=9)


rm(list = ls()[grep("plot", ls())])






