#####
#
# "Climate change mitigation targets compromise forest ecosystem services and biodiversity"
# Plot GLOBIOM demands provided by IIASA
# 2022-03-15
# 
#####


library(dplyr)
library(ggplot2)
library(tidyr)
library(ggpubr)
library(reshape2)
library(gridExtra)
library(ggsci)



### If new results should be saved
save_outp <- TRUE



# -----------
# Finland
# -----------

fin_1p5 <- read.csv(paste0(path, "globiom/FINglobiom1p5.csv"), header = TRUE, stringsAsFactors = TRUE, sep= ";") %>% 
  mutate(globiom = "1p5")
fin_ndc <- read.csv(paste0(path, "globiom/FINglobiomNDC.csv"), header = TRUE, stringsAsFactors = TRUE, sep= ";") %>% 
  mutate(globiom = "NDC")

fin_globiom <- rbind(fin_1p5, fin_ndc) %>% mutate(country = "Finland") %>% 
  mutate_at(c("Sawlog_uB", "Pulp_uB", "Fuel_oB", "Residues"), ~ .* 1000000 ) 
rm(fin_1p5, fin_ndc)


# -----------
# Germany
# -----------

ger_1p5 <- read.csv(paste0(path, "globiom/GERGlobiom1p5.csv"), header = TRUE, stringsAsFactors = TRUE, sep= ",") %>% 
  select(-X) %>% 
  mutate(globiom = "1p5")
ger_ndc <- read.csv(paste0(path, "globiom/GERGlobiomNDC.csv"), header = TRUE, stringsAsFactors = TRUE, sep= ",") %>% 
  select(-X) %>% 
  mutate(globiom = "NDC")

ger_globiom <- rbind(ger_1p5, ger_ndc) %>% mutate(country = "Germany")
rm(ger_1p5, ger_ndc)


# -----------
# Norway
# -----------

nor_1p5 <- read.csv(paste0(path, "globiom/NORglobiom1p5.csv"), header = TRUE, stringsAsFactors = TRUE, sep= ";") %>% 
  mutate(globiom = "1p5")
nor_ndc <- read.csv(paste0(path, "globiom/NORglobiomNDC.csv"), header = TRUE, stringsAsFactors = TRUE, sep= ";") %>% 
  mutate(globiom = "NDC")

nor_globiom <- rbind(nor_1p5, nor_ndc) %>% mutate(country = "Norway") %>% 
  mutate_at(c("Sawlog_uB", "Pulp_uB", "Fuel_oB", "Residues"), ~ .* 1000000 ) 
rm(nor_1p5, nor_ndc)


# -----------
# Sweden
# -----------

swe_1p5 <- read.csv(paste0(path, "globiom/SWEglobiom1p5.csv"), header = TRUE, stringsAsFactors = TRUE, sep= ";") %>% 
  mutate(globiom = "1p5")
swe_ndc <- read.csv(paste0(path, "globiom/SWEglobiomNDC.csv"), header = TRUE, stringsAsFactors = TRUE, sep= ";") %>% 
  mutate(globiom = "NDC")

swe_globiom <- rbind(swe_1p5, swe_ndc) %>% mutate(country = "Sweden") %>% 
  mutate_at(c("Sawlog_uB", "Pulp_uB", "Fuel_oB", "Residues"), ~ .* 1000000 ) 
rm(swe_1p5, swe_ndc)




# -----------
# combine Countries
# -----------

globiom <- rbind(fin_globiom, ger_globiom, nor_globiom, swe_globiom) %>% 
  mutate(total = Sawlog_uB + Pulp_uB + Fuel_oB/1.136  + Residues) %>% 
  filter(year >= 2020)

rm(fin_globiom, ger_globiom, nor_globiom, swe_globiom) 


# plot total demands

# Specify id.vars: the variables to keep but not split apart on
globiom_long <- melt(globiom, id.vars=c("year", "globiom", "country")) %>% 
  mutate(country = ifelse(country %in% "Germany", "Bavaria", country))

globiom_long$variable <- factor(globiom_long$variable, c("Residues", "Fuel_oB", "Pulp_uB", "Sawlog_uB", "total")) 
  


plot.demand_stack <- globiom_long %>% 
  mutate(country = factor(country, c("Finland", "Sweden", "Norway","Bavaria"))) %>% 
  filter(variable != "total") %>% 
  mutate(value = value / 1000000) %>% 
  ggplot(aes(x = year, y = value, fill = variable)) +
  geom_area(position = 'stack') +
  # theme_light() +
  # theme(axis.text.x = element_text(angle=90, hjust=1)) +
  
  theme_bw()+
  theme(axis.text.x = element_text(angle = 90),
        legend.position="bottom",
        axis.title.x=element_blank(),
        legend.title=element_blank(),
        #panel.grid.major = element_blank()
        panel.grid.minor = element_blank()
  ) +
  facet_grid(globiom ~ country) +
  scale_x_continuous(breaks = seq(2010, 2100, by = 10)) +
  guides(fill = guide_legend(title = "Assortment"))+
  scale_fill_brewer(palette="Set2", labels = c("Residues", "Fuelwood", "Pulpwood", "Sawlogs")) +
  ylab(expression(Harvest~demand~"["~Mm^{3}~year^{-1}~"]"))
plot.demand_stack  

if(save_outp == TRUE) {ggsave(plot = plot.demand_stack, paste0(outp_path,"globiom_demand_stack.tiff"), width=7, height=4.5)}
if(save_outp == TRUE) {ggsave(plot = plot.demand_stack, paste0(outp_path,"globiom_demand_stack.pdf"), width=7, height=4.5)}
rm(plot.demand_stack)



plot.demand_total <- globiom_long %>% 
  mutate(country = ifelse(country %in% "Germany", "Bavaria", country)) %>% 
  mutate(country = factor(country, c("Finland",  "Sweden", "Norway","Bavaria"))) %>%
  filter(variable %in% "total") %>% 
  filter(globiom != "LTS") %>% 
  mutate(value = value / 1000000) %>% 
  ggplot(aes(year, value, color = country, lty = globiom)) +
  geom_line() +
  theme_minimal() +
  theme(axis.text.x = element_text(angle=45, hjust=1)
        ,panel.grid.minor = element_blank()
        ,panel.grid.major = element_blank()
        ,panel.border = element_rect(fill=NA, colour = "grey")
        ,axis.ticks = element_line(colour = "grey")
        #,legend.position = "bottom"
  ) +
  scale_linetype_manual(name = "Mitigation target", values=c("solid", "dashed"), labels = c("1p5 (RCP1.9)", "NDC (RCP4.5)")) +
  #scale_color_brewer(palette = "Set2") +
  scale_color_jco(name = "Study region") +
  scale_x_continuous(breaks = seq(2010, 2100, by = 10)) +
  expand_limits(y = 0)+
  ylab(expression(Mm^{3}~year^{-1})) +
  labs(subtitle="a) Total harvest demand")
plot.demand_total 


# get %-values 
yr2020 <- globiom_long %>% 
  filter(variable %in% "total") %>% 
  filter(year %in% c(2020)) %>% 
  rename(value2020 = value)

globiom_long %>% 
  filter(variable %in% "total") %>% 
  filter(year %in% c(2100)) %>% 
  rename(value2100 = value) %>% 
  left_join(yr2020, by = c("globiom", "country", "variable")) %>% 
  mutate(increase = round((value2100 - value2020) / value2020 * 100) )
rm(yr2020)



# plot demands per hectare

# Productive forest in GLOBIOM
fin_tot_area <- 22.218
ger_tot_area <- 2.676
nor_tot_area <- 12.112
swe_tot_area <- 28.073

# Strictly protected forest in GLOBIOM
fin_protect_area <- 1.577478
ger_protect_area <- 0.036
nor_protect_area <- 3.99696
swe_protect_area <- 2.442351


# calculate averages per hectare related to the managed forests
globiom_avg_demand <- globiom_long  %>% 
  filter(variable == "total") %>% 
  mutate(tot_area = ifelse(country %in% "Finland", fin_tot_area,
                           ifelse(country %in% "Bavaria", ger_tot_area, 
                                  ifelse(country %in% "Sweden", swe_tot_area,
                                         nor_tot_area))),
         protect_area = ifelse(country %in% "Finland", fin_protect_area,
                               ifelse(country %in% "Bavaria", ger_protect_area, 
                                      ifelse(country %in% "Sweden", swe_protect_area,
                                             nor_protect_area)))) %>% 
  mutate(demand_area = (tot_area - protect_area)* 1000000,
         # only for managed area
         demand_ha = value / demand_area) %>% 
  group_by(country, globiom) %>% 
  summarise(avg_demand_ha = mean(demand_ha),
            min_demand_ha = min(demand_ha),
            max_demand_ha = max(demand_ha)) %>% 
  ungroup()


# plot
plot.demand_avg <- globiom_avg_demand %>% 
  mutate(country = ifelse(country == "Germany", "Bavaria", country)) %>% 
  mutate(country = factor(country, c("Finland",  "Sweden", "Norway", "Bavaria"))) %>% 
  ggplot(aes(globiom, avg_demand_ha, fill = globiom)) + 
  
  theme_light() +
  theme(axis.text.x = element_text(angle=45, hjust=1),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        legend.position = "none",
        strip.text = element_text(color = "black")) +
  #geom_bar(stat="identity",  fill="gray40") +
  geom_bar(stat="identity")+
  
  scale_fill_brewer(palette="Dark2") +
  geom_errorbar(aes(ymin=min_demand_ha, ymax= max_demand_ha), width=.2,
                position=position_dodge(.9)) +
  geom_text(aes(label=round(avg_demand_ha,digits = 1)), y= 0.8, color="white", size=3.5)+
  #scale_y_continuous(breaks =seq(0, 16, by = 2)) +
  
  #geom_text(aes(label=round(avg_demand_ha,digits = 1)), vjust = 1.5, color="white", size=3)+
  expand_limits(y = 15)+
  
  facet_grid(. ~ country) + 
  ylab(expression(m^{3}~ha^{-1}~year^{-1})) +
  xlab("Mitigation target") +
  labs(subtitle = "b) Average harvest demand per hectare")
plot.demand_avg


plot.combined_demand <- ggpubr::ggarrange(plot.demand_total, plot.demand_avg,ncol = 2, nrow= 1)

if(save_outp == TRUE) {ggsave(plot = plot.combined_demand, paste0(outp_path,"globiom_demand1.tiff"), width=9, height=3.25)}
if(save_outp == TRUE) {ggsave(plot = plot.combined_demand, paste0(outp_path,"globiom_demand1.pdf"), width=9, height=3.25)}


rm(plot.combined_demand, plot.demand_avg, plot.demand_total)
rm(globiom_long, globiom_avg_demand)












