#####
#
# "Climate change mitigation targets compromise forest ecosystem services and biodiversity"
# Main Analysis/graphics for joint publication
# 2022-03-15
#
#####


# Includes (so far):
#
# Effects on Ecosystem services and biodiversity 
# - Based on indicator results
# 
# Bottom-up analysis
# - Assortment structure of harvests
# - Regime structure of harvests
#
# Top-down analysis
# - Assortment structure of harvests
# - regime structure of harvests


# Represented countries:
# - Finland
# - Germany
# - Norway: preliminary, includes still the harvesting peak
# - Sweden



### load libraries
library(dplyr)
library(ggplot2)
library(tidyr)
library(reshape2)
library(ggpubr)


### If new results should be saved
save_outp <- TRUE




########################################################################################
# Effects on Ecosystem services and biodiversity - based on indicator results
########################################################################################


# -----------
# normalized FESB indicators - difference between top-down and bottom up
# -----------

df.es <- df.es_effect_all %>% 
  # Group the effects by ES category and calculate the mean of the ES (equal weights)
  group_by(country, policy, scenario, es) %>% 
  summarise(mean_diff = mean(diff) ) %>% 
  # keep the common ES among countries
  filter(!es %in% c("Nonwood", "Game", "Resilience")) %>% 
  mutate(es = factor(es, levels = c("Wood","Bioenergy","Biodiversity","Water","Climate","Recreation"))) %>% 
  ungroup()


  
# -----------
# Plot the heat map
# -----------

# plot Function for the difference
library("scales")


# explore value range for plotting
maxValue <- max(df.es$mean_diff)
minValue <- min(df.es$mean_diff)
range <- round((if(-minValue > maxValue){-minValue} else {maxValue})+ 0.04, 1 ) #get next higher decimal (0.04)
range



plot.tableFkt <- function(data, COUNTRY){
  
  # # test the function
  # COUNTRY = "Sweden"
  # data = df.es

  if (COUNTRY == "Finland"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Germany"){
    TITLE <- "Germany (Bavaria)"
  } else if (COUNTRY == "Norway"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Sweden"){
    TITLE <- COUNTRY
  }
  
  
  # adapt policy scenario name for plotting
  data <- data %>% 
    mutate(policy_plot = ifelse(policy %in% "BAU", "NFS",
                                ifelse(policy %in% "Biodiv", "BDS", "BES"))) %>% 
    mutate(scenario_plot = ifelse(scenario %in% "noCC", "1p5", "NDC"))
  # order factor level
  data$policy_plot <- factor(data$policy_plot, levels = c("NFS", "BDS", "BES"))
  
  plot <- data %>% 
    
    # country to plot
    filter(country %in% COUNTRY) %>% 
    
    ggplot(aes(x = scenario_plot, y = es)) +
    geom_tile(aes(fill = mean_diff)) +
    #geom_text(aes(label = mean_diff), color = "gray30", size = 2.5) + #, fontface = "bold"
    scale_x_discrete(expand = c(0,0))+
    scale_y_discrete(expand = c(0,0), limits = rev(levels(data$es))) +
    scale_fill_gradientn("Impact (loss & gain)\n",
                         #colours=c("red","white", "blue"),
                         colours = c("#E46726", "grey99",  "#6D9EC1"),
                         na.value = "grey98",
                         
                         limits = c(-range,range), # adjust limits to set breaks
                         breaks = c(-range,-range/2, 0, range/2,range ),
                         oob=squish
                         ) +
   
    
    theme_minimal() +
    theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust=1),
          axis.title.x = element_blank(),
          axis.title.y = element_blank(),
          legend.position="bottom",
          legend.key.width=unit(2,"cm")) +
    
    #ylab("Ecosystem service") +
    labs(subtitle = TITLE) +
    facet_grid(. ~ policy_plot)
  plot
  
  return(plot)
  
}


# call the above plot function
plot.fin_es_effect <- plot.tableFkt(df.es, "Finland")
plot.nor_es_effect <- plot.tableFkt(df.es, "Norway")
plot.swe_es_effect <- plot.tableFkt(df.es, "Sweden")
plot.ger_es_effect <- plot.tableFkt(df.es, "Germany")

plot.combined <- ggpubr::ggarrange(plot.fin_es_effect, plot.nor_es_effect, 
                                   plot.swe_es_effect, plot.ger_es_effect,
                                    common.legend = TRUE,legend = "bottom",
                                    ncol = 2, nrow= 2)

plot.combined2 <- annotate_figure(plot.combined, top = text_grob("Impact on forest ecosystem services and biodiversity", face = "bold", size = 12))

if(save_outp == TRUE) {ggsave(plot = plot.combined2, paste0(outp_path,"cross-scale_es_effect_new2.tiff"), width=8, height=6)}
if(save_outp == TRUE) {ggsave(plot = plot.combined2, paste0(outp_path,"cross-scale_es_effect_new2.pdf"), width=8, height=6)}

rm(plot.fin_es_effect, plot.ger_es_effect, plot.nor_es_effect, plot.swe_es_effect)
rm(plot.combined, plot.combined2)
rm(df.es)





########################################################################################
# BOTTOM-UP - Assortment and regime structure under policy scenarios
########################################################################################


# -----------
# Assortment structure
# -----------

# -----------
# plot Function assortment shares - Differences
# -----------

# take: gl_national_sort_mean, harvest_national_sort_mean_gv1
df.sort_diff_gv1 <- harvest_sort_mean_gv1 %>% 
  # merge the mean national assortment demands (without columns policy 3)
  left_join(gl_national_sort_mean[,-4], by = c("country", "scenario", "globiom","sort")) %>% 
  # difference between simulated harvest and demand
  mutate(diff = mean.x - mean.y) %>% 
  mutate(sort = factor(sort, levels = c("Residues","PulpFuel", "Sawlog" ))) 
  

plot.sortDiffFct <- function(data, COUNTRY){
  
  if (COUNTRY == "Finland"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Germany"){
    TITLE <- "Germany (Bavaria)"
  } else if (COUNTRY == "Norway"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Sweden"){
    TITLE <- COUNTRY
  }
  
  # get min & max value to expand the y-axis
  total <- data %>% filter(diff < 0 ) %>%  group_by(country, policy, scenario, globiom) %>% summarise(value = sum(diff))
  minValue = min(total$value)
  total <- data %>% filter(diff > 0 ) %>%  group_by(country, policy, scenario) %>% summarise(value = sum(diff))
  maxValue = max(total$value)
  
  # adapt policy scenario name for plotting
  data <- data %>% 
    mutate(policy_plot = ifelse(policy %in% "GLOBIOM", "GLOBIOM",
                                  ifelse(policy %in% "BAU", "NFS",
                                         ifelse(policy %in% "Biodiv", "BDS", "BES"))))
    data$policy_plot <- factor(data$policy_plot, levels = c("GLOBIOM","NFS", "BDS", "BES"))
  
  plot <- data %>% 
    
    filter(country %in% COUNTRY) %>% 
    
    ggplot(aes(fill=sort, y=diff, x=policy_plot)) + 
    geom_bar(position="stack", stat="identity") +
    geom_hline(yintercept=0, size = 0.5)+
    
    #theme_light() +
    theme_minimal()+
    theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust=1),
          axis.title.x = element_blank(),
          legend.title=element_blank(),
          panel.grid.major = element_blank(),
          panel.grid.minor = element_blank()#,
          # panel.border = element_rect(colour = "darkgrey", fill=NA)
          ) +
    #facet_grid(. ~ scenario) +
    facet_grid(.~globiom) +
    scale_fill_brewer(palette="Set2", label = c("Residues", "Pulp- & Fuelwood", "Sawlogs")) +
    #scale_fill_brewer(palette="Pastel1") +
    
    {if(COUNTRY %in% c("Germany","Norway"))ylim(minValue *0.7, maxValue *0.7)} +
    {if(COUNTRY %in% c("Finland","Sweden"))ylim(minValue, maxValue)}+
    
    ylab(expression(Mm^{3}~year^{-1})) +
    labs(subtitle = TITLE)
  plot
  
  return(plot)
  
}

plot.fin_sort_diff_gv1 <- plot.sortDiffFct(df.sort_diff_gv1, "Finland")
plot.ger_sort_diff_gv1 <- plot.sortDiffFct(df.sort_diff_gv1, "Germany")
plot.nor_sort_diff_gv1 <- plot.sortDiffFct(df.sort_diff_gv1, "Norway")
plot.swe_sort_diff_gv1 <- plot.sortDiffFct(df.sort_diff_gv1, "Sweden")

plot.combined_sort_diff_gv1 <- ggpubr::ggarrange(plot.fin_sort_diff_gv1, plot.swe_sort_diff_gv1,  plot.nor_sort_diff_gv1, plot.ger_sort_diff_gv1,
                                                 common.legend = TRUE,legend = "bottom",
                                                 ncol = 4, nrow= 1)

rm(plot.fin_sort_diff_gv1, plot.ger_sort_diff_gv1, plot.nor_sort_diff_gv1, plot.swe_sort_diff_gv1)
rm(df.sort_diff_gv1 )  




# -----------
# Management structure
# -----------



# plot function for the regime shares

plot.regimeFkt <- function(gl_data, data, COUNTRY){
  
  if (COUNTRY == "Finland"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Germany"){
    TITLE <- "Germany (Bavaria)"
  } else if (COUNTRY == "Norway"){
    TITLE <- COUNTRY
  } else if (COUNTRY == "Sweden"){
    TITLE <- COUNTRY
  }
  
  # adapt policy scenario name for plotting
  data <- data %>% 
    mutate(policy_plot = ifelse(policy %in% "GLOBIOM", "GLOBIOM",
                                  ifelse(policy %in% "BAU", "NFS",
                                         ifelse(policy %in% "Biodiv", "BDS", "BES"))))
    data$policy_plot <- factor(data$policy_plot, levels = c("GLOBIOM","NFS", "BDS", "BES"))
  
  # get max value to expand the y-axis
  total <- data %>% group_by(policy, scenario, globiom ,country) %>% summarise(value = sum(mean))
  maxValue = max(total$value)
  
  # maxValue = round(max(gl_data$mean), digits = 0)
  # maxValue = round_any(max(gl_data$mean), 10, f = ceiling) 
  
  gl_data <- gl_data %>% filter(country %in% COUNTRY)
  
  plot <- data %>% 
    
    filter(country %in% COUNTRY) %>% 
    
    ggplot(aes(fill=regime_6class, y=mean, x=policy_plot)) + 
    geom_bar(position="stack", stat="identity") +
    
    geom_hline(data = gl_data, aes(yintercept = mean), linetype="dashed", 
               color = "red", size=0.5) +
    
    #theme_article() +
    theme_minimal()+
    theme(axis.text.x = element_text(angle = 45, vjust = 1, hjust=1),
          axis.title.x = element_blank(),
          legend.title=element_blank(),
          panel.grid = element_blank()) +
    facet_grid(. ~ globiom) +
    scale_fill_viridis_d(labels = c("Intensify RF",
                                    "Rotation forestry (RF)",
                                    "Extensify RF",
                                    "Adaption to CC",
                                    "Continuous cover" )) +
    
   {if(COUNTRY %in% c("Finland","Sweden"))scale_y_continuous(limits = c(0, 120), breaks = seq(0, 120, by = 20))} +
   {if(COUNTRY %in% c("Germany","Norway"))expand_limits(y = maxValue / 3)} +
   
    ylab(expression(Mm^{3}~year^{-1})) +
    labs(subtitle = TITLE)
  plot
  
  return(plot)
  
}

plot.finland_regime_gv1 <- plot.regimeFkt(gl_national_mean, harvest_regime_mean_gv1, "Finland")
plot.germany_regime_gv1 <- plot.regimeFkt(gl_national_mean, harvest_regime_mean_gv1, "Germany")
plot.norway_regime_gv1 <- plot.regimeFkt(gl_national_mean, harvest_regime_mean_gv1, "Norway")
plot.sweden_regime_gv1 <- plot.regimeFkt(gl_national_mean, harvest_regime_mean_gv1, "Sweden")

plot.combined_regime_gv1 <- ggpubr::ggarrange(plot.finland_regime_gv1, plot.sweden_regime_gv1, plot.norway_regime_gv1, plot.germany_regime_gv1,
                                   common.legend = TRUE,legend = "bottom",
                                   ncol = 4, nrow= 1)

rm(plot.finland_regime_gv1, plot.germany_regime_gv1, plot.norway_regime_gv1, plot.sweden_regime_gv1)


a <- annotate_figure(plot.combined_sort_diff_gv1, fig.lab = "a)", fig.lab.face = "bold")
b <- annotate_figure(plot.combined_regime_gv1, fig.lab = "b)", fig.lab.face = "bold")

ab <- ggpubr::ggarrange(a, b, ncol = 1, nrow=2)
ab <- annotate_figure(ab, top = text_grob("Bottom-up: optimize for national policy demands", face = "bold", size = 14))

if(save_outp ==TRUE) {ggsave(plot = ab, paste0(outp_path,"cross-scale_bottom-up2.jpg"), width=10, height=7)}
if(save_outp ==TRUE) {ggsave(plot = ab, paste0(outp_path,"cross-scale_bottom-up2.pdf"), width=10, height=7)}

rm(plot.combined_regime_gv1, plot.combined_sort_diff_gv1, a, b, ab)







########################################################################################
# TOP-DOWN - Assortment and regime structure under policy scenarios
########################################################################################

# -----------
# Assortment structure
# -----------

# -----------
# call plot function for assortments - DIFFERENCES
# -----------

# take: gl_national_sort_mean, harvest_national_sort_mean_gv1
df.sort_diff_gv2 <- harvest_sort_mean_gv2 %>% 
  # merge the mean national assortment demands (without columns policy 3)
  left_join(gl_national_sort_mean[,-4], by = c("country", "scenario", "globiom","sort")) %>% 
  # difference between simulated harvest and demand
  mutate(diff = mean.x - mean.y) %>% 
  mutate(sort = factor(sort, levels = c("Residues","PulpFuel", "Sawlog" ))) 


plot.fin_sort_diff_gv2 <- plot.sortDiffFct(df.sort_diff_gv2, "Finland")
plot.ger_sort_diff_gv2 <- plot.sortDiffFct(df.sort_diff_gv2, "Germany")
plot.nor_sort_diff_gv2 <- plot.sortDiffFct(df.sort_diff_gv2, "Norway")
plot.swe_sort_diff_gv2 <- plot.sortDiffFct(df.sort_diff_gv2, "Sweden")

plot.combined_sort_diff_gv2 <- ggpubr::ggarrange(plot.fin_sort_diff_gv2, plot.swe_sort_diff_gv2, plot.nor_sort_diff_gv2, plot.ger_sort_diff_gv2,
                                                 common.legend = TRUE,legend = "bottom",
                                                 ncol = 4, nrow= 1)

rm(plot.fin_sort_diff_gv2, plot.ger_sort_diff_gv2, plot.nor_sort_diff_gv2, plot.swe_sort_diff_gv2)
rm(df.sort_diff_gv2)




# -----------
# Management structure
# -----------

# -----------
# call the plot function for regime
# -----------

plot.fin_regime_gv2 <- plot.regimeFkt(gl_national_mean, harvest_regime_mean_gv2, "Finland")
plot.ger_regime_gv2 <- plot.regimeFkt(gl_national_mean, harvest_regime_mean_gv2, "Germany")
plot.nor_regime_gv2 <- plot.regimeFkt(gl_national_mean, harvest_regime_mean_gv2, "Norway")
plot.swe_regime_gv2 <- plot.regimeFkt(gl_national_mean, harvest_regime_mean_gv2, "Sweden")

plot.combined_regime_gv2 <- ggpubr::ggarrange(plot.fin_regime_gv2, plot.swe_regime_gv2, plot.nor_regime_gv2, plot.ger_regime_gv2,
                                   common.legend = TRUE,legend = "bottom",
                                   ncol = 4, nrow= 1)

a <- annotate_figure(plot.combined_sort_diff_gv2, fig.lab = "a)", fig.lab.face = "bold")
b <- annotate_figure(plot.combined_regime_gv2, fig.lab = "b)", fig.lab.face = "bold")

ab <- ggpubr::ggarrange(a, b, ncol = 1, nrow=2)
ab <- annotate_figure(ab, top = text_grob("Top-Down: prioritizing EU mitigation targets above national policy demands", face = "bold", size = 14))

if(save_outp ==TRUE) {ggsave(plot = ab, paste0(outp_path,"cross-scale_topdown2.jpg"), width=10, height=7)}
if(save_outp ==TRUE) {ggsave(plot = ab, paste0(outp_path,"cross-scale_topdown2.pdf"), width=10, height=7)}


rm(plot.fin_regime_gv2, plot.ger_regime_gv2, plot.nor_regime_gv2, plot.swe_regime_gv2)
rm(plot.combined_regime_gv2, plot.combined_sort_diff_gv2, a, b, ab)









