#####
#
# "EU climate change mitigation targets compromise forest ecosystem services and biodiversity"
# Read the aggregated data of all study regions under the top-down and bottom-up approach
# 2022-03-16
#
#####


### load libraries
library(dplyr)
library(tidyr)


### Set the working path to this R-project
path <- paste0(getwd(),"/")

outp_path <- paste0(path, "outp/")


# ----------
# read the combined results of all study regions
# ----------

# -----------
# effect on forest ecosystem services
# -----------
df.es_all.t <- read.csv(paste0(path,"input/df.es_all.csv" ), sep = ";", header = TRUE) 

# Normalize function
normalizeFct <- function(x) {(x - min(x)) / (max(x) - min(x))}

# Indicators related to management regime share
regime_indicator <- c("Ratio_CCF_forests", "Ratio_SA_forests", "Ratio_Broadleave_forests", "Ratio_CCF_onPeat", "set_aside", "CCF" )

df.es_all <- df.es_all.t %>% 
  # normalize each indicator by the min and max; over all policy & climate scenarios, years, and bottom-up & top-down (v1, v2)
  # except the ones addressing a regime share
  group_by(country, indicator) %>%
  mutate(norm = ifelse(!indicator %in% regime_indicator, normalizeFct(areaWeightedAverage), areaWeightedAverage)) %>% 
  ungroup()

# split into bottom-up (v1) and top-down (v2)
df.es_gv1_all <- df.es_all %>%  filter(OptApproach %in% "v1") 
df.es_gv2_all <- df.es_all %>%  filter(OptApproach %in% "v2") 

# calculate difference between normalized values
df.es_effect_all <- df.es_gv1_all %>% left_join(df.es_gv2_all, by = c("country","policy","scenario","globiom","year","indicator", "es") ) %>%
  mutate(t_diff = norm.y - norm.x) %>%
  # calculate the mean over time
  group_by(country,policy,scenario,globiom,es,indicator) %>%
  summarise(diff = mean(t_diff),
            min_diff = min(t_diff),
            max_diff = max(t_diff),
            sd_diff = sd(t_diff)) %>% 
  ungroup()


# -----------
# interpolated GLOBIOM demands targeted in the optimization
# -----------
df.gl_national_all <- read.csv(paste0(path,"input/df.gl_national_all.csv" ), sep = ";", header = TRUE) 

# Mean over years by assortment
gl_national_sort_mean <- df.gl_national_all %>% 
  select(-tot_demand) %>% 
  # same name for assortments for merge the data later
  rename(Sawlog = GSawlog_uB,
         PulpFuel = GPulpFuel_uB,
         Residues = GResidues) %>% 
  gather("sort","demand", 2:4) %>% 
  # mean by assortment over time
  group_by(country, scenario, globiom, sort ) %>% 
  summarise(mean = mean(demand)) %>%  # divide by million
  mutate(policy = "GLOBIOM") %>% 
  ungroup() %>% 
  # order columns
  select(c("country","scenario","globiom","policy","sort","mean"))

# National mean
gl_national_mean <- df.gl_national_all %>% 
  group_by(country, scenario, globiom ) %>% 
  summarise(mean = mean(tot_demand)) 


# -----------
# optimized regime shares
# -----------
regime_share_all <- read.csv(paste0(path,"input/regime_share_all.csv" ), sep = ";", header = TRUE)


# -----------
# Optimized harvests under bottom-up approach (v1)
# -----------
opt_harvests_gv1 <- read.csv(paste0(path,"input/opt_harvests_gv1.csv" ), sep = ";", header = TRUE)

# By assortments
# total yearly values
harvest_sort_timely_gv1 <- opt_harvests_gv1 %>% 
  # remove column for the total simulated harvest
  select(-c("tot_simulated")) %>% 
  group_by(country,policy,scenario,globiom,year) %>%  #, regime_6class
  summarise(Sawlog = sum(Sawlog),
            PulpFuel = sum(PulpFuel),
            Residues = sum(Residues)) %>% 
  gather("sort", "harvest", 6:8) %>% 
  ungroup()

# mean by assortment over time 
harvest_sort_mean_gv1  <- harvest_sort_timely_gv1 %>% group_by(country,scenario, globiom ,policy, sort) %>% 
  summarise(mean =  mean(harvest))%>%  
  ungroup()

# By management class
harvest_regime_mean_gv1 <- opt_harvests_gv1 %>% 
  # calculate means values over years 
  group_by(country,policy,scenario,globiom,regime_6class) %>%
  summarise(mean = mean(tot_simulated)) %>%  
  # filter out set aside
  filter(regime_6class != "SA") %>% 
  mutate(regime_6class = factor(regime_6class, levels = c("I-BAU", "BAU", "E-BAU", "ACC","CCF",  "SA" ))) %>% 
  ungroup()



# -----------
# Optimized harvests under top-down approach (v2)
# -----------
opt_harvests_gv2 <- read.csv(paste0(path,"input/opt_harvests_gv2.csv" ), sep = ";", header = TRUE)

# By assortments
# total yearly values
harvest_sort_timely_gv2 <- opt_harvests_gv2 %>% 
  # remove column for the total simulated harvest
  select(-c("tot_simulated")) %>% 
  group_by(country,policy,scenario,globiom,year) %>%  #, regime_6class
  summarise(Sawlog = sum(Sawlog),
            PulpFuel = sum(PulpFuel),
            Residues = sum(Residues)) %>% 
  gather("sort", "harvest", 6:8) %>% 
  ungroup()

# mean by assortment over time 
harvest_sort_mean_gv2  <- harvest_sort_timely_gv2 %>% group_by(country,scenario, globiom ,policy, sort) %>% 
  summarise(mean =  mean(harvest))%>%  
  ungroup()

# By management class
harvest_regime_mean_gv2 <- opt_harvests_gv2 %>% 
  # calculate mean values over years 
  group_by(country,policy,scenario,globiom,regime_6class) %>%
  summarise(mean = mean(tot_simulated)) %>%  
  # filter out set aside
  filter(regime_6class != "SA") %>% 
  mutate(regime_6class = factor(regime_6class, levels = c("I-BAU", "BAU", "E-BAU", "ACC","CCF",  "SA" ))) %>% 
  ungroup()


# -----------
# Effects of optimization scenarios on main forest statistics
# -----------

df.forestStat <- read.csv(paste0(path,"input/df.forestStat_all.csv" ), sep = ";", header = TRUE)

