(*| .. coq:: none |*)
Set Warnings "-notation-overridden, -ambiguous-paths".

(*|
====================
QRC1 in Coq: summary
====================

This project is a formalization of the QRC1 logic defined in [1] and refined in
[2]. This file lists the main definitions and results already formalized,
linking them with the names and numbering of [2], with a side note on some
results from [1].

[1] A. de Almeida Borges and J. J. Joosten (2020).
Quantified Reflection Calculus with one modality.
Advances in Modal Logic 13: 13-32.
`arXiv:2003.13651 [math.LO] <https://arxiv.org/pdf/2003.13651.pdf>`_

[2] A. de Almeida Borges and J. J. Joosten (2022).
An escape from Vardanyan’s Theorem.
Journal of Symbolic Logic.
`arXiv:2102.13091 [math.LO] <https://arxiv.org/pdf/2102.13091.pdf>`_

See the banner at the top for instructions on how to navigate this file.
|*)

From mathcomp Require Import all_ssreflect finmap.
Unset Printing Implicit Defensive.

(*|
Language
========
|*)

From QRC1 Require Import Language.

(*| Variables are natural numbers. |*)
Print VarName.

(*| A signature declares finitely many constants and predicate symbols. |*)
Print signature.

(*| A term is either a variable or a constant. |*)
Print term.

(*|
A formula is either T, a predicate symbol applied to a tuple of terms, a
conjunction of two other formulas, a diamond, or a universal quantifier.
|*)
Print formula.

(*|
Formulas live within the qsp_scope (for quantified and strictly positive) and we
provide the following notations.
|*)
Open Scope qsp_scope.
Locate "A /\ B".
Locate "<> A".

(*|
The free variables and constants of a formula are defined as usual, as is the
definition of closed formula.
|*)
Print fv.
Print constants.
Print closed.

(*|
The substitution of a term for a term is defined without regard for possible
captures of the new term by existing quantifications.
|*)
Print sub. Locate "A `[ t1 <- t2 ]".
(*|
We then define the notion of a term being free for a variable or more generally
a term in a formula, which protects against this case and is assumed as needed.
|*)
Print freefor.
Print freefort.

(*|
Finally we define three notions of complexity for qsp formulas (Definition 3.3).
|*)
Print modaldepth. (* maximum number of nested diamonds *)
Print quantifierdepth. (* maximum number of nested quantifiers *)
Print depth. (* maximum number of nested symbols *)

(*|
QRC1
====
|*)

From QRC1 Require Import QRC1.

(*| The definition of QRC1 proof (Definition 3.1). |*)
Print QRC1Proof.
Locate "|- A ~> B".

(*| Lemma 3.2. |*)
Check AllC.
Check All_sub.
Check Diam_All.
Check alphaconversion.
Check TermIr.
Check Const_AllIr.

(*| Lemma 3.4. |*)
Check QRC1Proof_modaldepth.
Check Diam_irreflexive.

(*|
Kripke Semantics
================
|*)

From QRC1 Require Import KripkeSemantics.
Open Scope fset.

(*| We only implement finite frames and models. |*)

(*|
Not-necessarily adequate frames and models (Definition 4.1).

We stipulate that the worlds of a frame are a finite set of some choiceType
named WType, and that the domain elements are a finite set of some choiceType
named MType. This restricts the implemented definition of frame to finite
frames, which is weaker than the definition presented in [1], but sufficient for
the completeness proof.

We use the notions of rawFrame and rawModel to refer to possibly non adequate
frames and models.

Unlike in [1], we define eta functions for every possible pair of worlds, which
avoids unnecessary dependent types at the level of rawFrames.
|*)
Print rawFrame.
Print rawModel.

(*|
We can now define adequate frames and models (Definition 4.2).

Note that since we require eta functions for every pair of worlds, we require an
additional property in order for a frame to be adequate, namely that all eta
functions between a world and itself be the identity.
|*)
Print frame.
  Print adequateF.
    Check transitiveFP.
    Check transetaFP.
    Check idetaFP.
Print model.
  Print adequateM.
    Check concordantMP.

(*|
An assignment is a function from variables to domain elements, parametrized on a
world.
|*)
Print assignment.

(*| Assignments have a canonical extension to terms. |*)
Print assignment_of_term.
Locate "g +".

(*|
A w-assignment can be lifted to a u-assignment by composing with eta u.

Again, we do not require that w R u as in [1], avoiding unwieldy dependent types
at this stage. In practice, we will only rely on this notation when w R u does
hold (see the definition of sat below).
|*)
Locate "g ` u".

(*|
Two assignments are X-alternative (or just x-alternative when X = {x}) when they
agree on every variable except possibly those in X.
|*)
Print Xaltern.
Print xaltern.

(*| We now define the notion of satisfaction (Definition 4.3). |*)
Print sat.

(*|
Kripke soundness
----------------
|*)

(*| The soundness theorem (Theorem 4.4). |*)
Check soundness.
Print Assumptions soundness.

(*|
The soundness proof is reasonably straightforward and as such is left out of
[2]. However, it requires about 750 lines of extra definitions, lemmas, and
proofs. We summarize those in what follows, following the soundness proof
detailed in Section 4 of [1] `arXiv:2003.13651 [math.LO]
<https://arxiv.org/pdf/2003.13651.pdf>`_.
|*)

(*| Remark 4.5 |*)
Check sat_Xalternfv.

(*| Lemma 4.6 |*)
Check substitution_formula.


(*| Remark 4.7 |*)
Set Printing Implicit Defensive.
Check sat_noconstants.
Unset Printing Implicit Defensive.

(*|
We now work towards frames restricted to some world r and all its successors. We
follow a slightly different strategy than the one presented in [1], and as such
the intermediate lemmas don't exactly match. The idea is to replace the
interpretation of some constant c with some r-domain element d at the world r
and by eta u d at every successor of r. Since this breaks the concordance of the
model, we first drop every other world, obtaining a model that satisfies exactly
the same formulas at r. Here the main difference is that we do not immediately
drop the extra worlds; we keep them around and change the adequateness condition
instead, so that it checks only r and its successors. We then proceed with this
not entirely adequate model and only after the replacement is done do we drop
the extra worlds and show that the resulting model is fully adequate. This
strategy proved much easier to implement than dropping the extra worlds first.
|*)

(*| The reflexive closure of R. |*)
Print Rs.

(*|
A different notion of adequacy, that only checks a root world and its
successors.
|*)
Print adequateFr.
  Check transitiveFrP.
  Check transetaFrP.
  Check idetaFrP.
Print adequateMr.
  Check concordantMrP.

(*|
An adequate frame is also adequate in the adequateFr sense, and the same for
models.
|*)
Check frame_adequateFr.
Check model_adequateMr.

(*|
Given some interpretation I, a new interpretation that behaves as I except it
interprets a given constant c as a given r-domain element d instead. This new
interpretation will not lead to an adequate model unless r is the root (the
concordance condition may fail).
|*)
Print replace_I.
Print replace_rawModel.

Set Printing Implicit Defensive.

(*|
We use the following notation to refer to raw models where the interpretation of
c has been replaced by d on world r and its successors.
|*)
Locate "M `[ r , c <- d ]".

(*|
A model replacing some constant interpretation with an element of the domain of
r is adequate if ignoring every world that is not either r or its successor.
|*)
Check replace_adequateMr.

(*|
Changing the interpretation of a constant that does not appear in A is
irrelevant.
|*)
Check sat_replace_noconstants.

(*|
Checking whether A is satisfied at world u is independent of which root is
picked, as long as u is either the root or a successor of the root.
|*)
Check sat_replace_root.
(*|
In particular, we can pick the world where we want to check A as the root.
|*)
Check sat_replace_root_self.

(*|
A is satisfied at w in a model M iff A[x <- Const c] is satisfied at w in M with
the interpretation of c replaced by the interpretation of x This is not quite
Lemma 4.12 yet, because the new model is not adequate.
|*)
Check sat_replace.

(*| We now start working toward actually drop the useless worlds. |*)

(*| A restricted set of worlds. |*)
Print restrict_world.

(*| We can now define restricted raw frames and raw models (Definition 4.8). |*)
Print restrict_rawFrame.
  Print restrict_R.
  Print restrict_domain.
  Print restrict_eta.
Print restrict_rawModel.
  Print restrict_I.
  Print restrict_J.

(*|
Restricted raw frames and models are adequate in the normal sense (Remark 4.9).
|*)
Check restrict_frame_adequate.
Check restrict_model_adequate.

(*| And thus we can define restricted frames and models. |*)
Print restrict_frame.
Print restrict_model.

(*| Remark 4.10 |*)
Check sat_restrict.

(*|
Restricting M`[r, c <- d] to r and its successors gives us an adequate model.
|*)
Check restrict_replace_adequate.

(*| Definition 4.11 |*)
Print restrict_replace.

(*|
We can finally prove the key lemma for the soundness of ConstE (Lemma 4.12).
|*)
Check sat_restrict_replace.

Unset Printing Implicit Defensive.

(*|
Kripke completeness
-------------------
|*)

From QRC1 Require Import Closure Pairs.

(*| A pair is just a pair of finite sets of formulas. |*)
Print pair.
  Print formulas.

(*| The notion of Phi-MCW is defined here as wfpair. |*)
Print wfpair.
  Print closedfs.
  Print consistent.
  Check maximalP.
  Check witnessedP.

(*|
The closed closure of a formula is defined by induction on the depth of the
formula (Definition 5.1). The closure of a set of formulas is the union of the
closures of the individual formulas.
|*)
Check closure_eq.
Print closurefs.

(*|
The number of different constants in a formula and the maximum of this value for
a set of formulas (Definition 5.2).
|*)
Print constantcount.
Print constantcountfs.

(*| Remark 5.3 |*)
Check constantcount_sub.
Check constantcount_closure.
Check constantcount_closurefs.

(*|
From now on we can no longer continue without classical logic. Thus we depend on
the following axiom.
|*)
Check Classical_Prop.classic.

(*| The Lindenbaum Lemma (Lemma 5.4) |*)
Check lindenbaum.
Print Assumptions lindenbaum.

(*|
We define RR as the target relation between worlds of the canonical model
(Definition 5.5).
|*)
Check RRP.

(*|
RR is transitive and irreflexive (when restricted to consistent pairs) (Lemma
5.6).
|*)
Check RR_trans.
Check RR_irr.

(*| The pair existence lemma (Lemma 5.7) |*)
Check pair_existence.
Print Assumptions pair_existence.


(*|
The formalization ends here for now. We aim to complete Section 5 in the future.
|*)
